#include "file.h"

#include "crypto/hash.h"
#include "log/global.h"
#include "string/coder.h"

#include <util/stream/file.h>
#include <util/system/file.h>
#include <util/system/filemap.h>
#include <util/system/fs.h>
#include <util/system/fstat.h>

#include <sys/stat.h>

namespace NPassport::NUtils {
    static TFile OpenRdOnlyFile(const TString& path) {
        return TFile(path, OpenExisting | RdOnly | Seq);
    }

    static TFile OpenWrOnlyFile(const TString& path) {
        return TFile(path, CreateAlways | WrOnly | Seq);
    }

    TFileLoader::TFileLoader(const TString& path, TFuncForFileBody func, TDuration period)
        : Path_(path)
        , FuncForFileBody_(func)
    {
        Init(period);
    }

    TFileLoader::TFileLoader(const TString& path, TFuncForFileName func, TDuration period)
        : Path_(path)
        , FuncForFileName_(func)
    {
        Init(period);
    }

    void TFileLoader::Init(TDuration period) {
        const TFile file = OpenRdOnlyFile(Path_);
        const time_t mtime = TFileStat(file).MTime;

        RunImpl(file, mtime);

        if (period) {
            Refresher_ = std::make_unique<NUtils::TRegularTask>(
                [this]() { this->Run(); },
                period);
        }
    }

    void TFileLoader::Run() {
        try {
            const TFile file = OpenRdOnlyFile(Path_);
            const time_t mtime = TFileStat(file).MTime;

            if (mtime != LastModified_) {
                RunImpl(file, mtime);
            }
        } catch (const std::exception& e) {
            TLog::Warning("FileLoader: File data update failed. Path: %s. %s", Path_.c_str(), e.what());
        } catch (...) {
            TLog::Error("FileLoader: File data update failed: %s. Unexpected exception", Path_.c_str());
        }
    }

    void TFileLoader::RunImpl(const TFile& file, const time_t mtime) {
        if (FuncForFileBody_) {
            RunForFileBody(file, mtime);
        } else {
            RunForFileName(mtime);
        }
    }

    void TFileLoader::RunForFileBody(const TFile& file, const time_t mtime) {
        TFileMap m(file);
        if (m.Length() == 0) {
            throw yexception() << "FileLoader: file is empty: " << Path_;
        }

        TFileMap::TMapResult r = m.Map(0, m.Length());
        const TStringBuf body((char*)r.MappedData(), r.MappedSize());

        FuncForFileBody_(body, mtime);
        LastModified_ = mtime;

        TLog::Info() << "FileLoader: File data updated successfully: " << Path_
                     << ". md5(hex)=" << Bin2hex(TCrypto::Md5(body))
                     << ". Last modification time: " << LastModified_;
    }

    void TFileLoader::RunForFileName(const time_t mtime) {
        FuncForFileName_(Path_);
        LastModified_ = mtime;

        TLog::Info() << "FileLoader: File data updated successfully: " << Path_
                     << ". Last modification time: " << LastModified_;
    }

    TString ReadFile(const TString& path, time_t* mtime) {
        const TFile file = OpenRdOnlyFile(path);
        if (mtime) {
            *mtime = TFileStat(file).MTime;
        }

        TFileMap m(file);
        if (m.Length() == 0) {
            throw yexception() << "FileLoader: file is empty: " << path;
        }

        TFileMap::TMapResult r = m.Map(0, m.Length());
        return TString((char*)r.MappedData(), r.MappedSize());
    }

    TString ReadFile(const TString& path, const TString& defaultValue, time_t* mtime) {
        try {
            return ReadFile(path, mtime);
        } catch (const yexception&) {
            return defaultValue;
        }
    }

    void WriteFileViaTmp(const TString& path, TStringBuf body) {
        size_t pos = path.rfind('/');
        if (pos == TString::npos) {
            throw yexception() << "File path is malformed: " << path;
        }

        if (!NFs::MakeDirectoryRecursive(path.substr(0, pos))) {
            throw yexception() << "Disk cache does not exist: " << path;
        }

        const TString tmpPath = path + ".tmp";
        TFileOutput(OpenWrOnlyFile(tmpPath)).Write(body);

        if (!NFs::Rename(tmpPath, path)) {
            throw yexception() << "Failed to move disk cache from '" << tmpPath << "' to '" << path << "'. "
                               << LastSystemErrorText();
        }
    }
}
