#pragma once

#include <util/generic/strbuf.h>
#include <util/generic/string.h>
#include <util/stream/output.h>

namespace NPassport::NUtils {
    // Ip address, supports both ipv4 && ipv6
    class TIpAddr {
    public:
        TIpAddr() = default;
        TIpAddr(const TString& strAddr);
        constexpr TIpAddr(ui8 a, ui8 b, ui8 c, ui8 d) noexcept;
        constexpr TIpAddr(ui16 a, ui16 b, ui16 c, ui16 d, ui16 e, ui16 f, ui16 g, ui16 h) noexcept;

        static TString Normalize(const TString& strIp);

        // parse string representation of address
        // return true on success
        bool Parse(const TString& strAddr);
        bool Parse(const TString& strAddr, size_t b, size_t e) {
            return Parse(strAddr.substr(b, e));
        }

        // parse numeric representation of address
        bool ParseBase64(const TStringBuf strAddr);

        // returns true if it is ip v4 address, false otherwise
        bool IsIpv4() const;
        bool IsIpv6() const;

        ui32 ProjectId() const;

        bool operator==(const TIpAddr& addr) const {
            return (Low_ == addr.Low_) && (High_ == addr.High_);
        }

        bool operator!=(const TIpAddr& addr) const {
            return (Low_ != addr.Low_) || (High_ != addr.High_);
        }

        bool operator<(const TIpAddr& addr) const {
            return (High_ < addr.High_) ||
                   (High_ == addr.High_ && Low_ < addr.Low_);
        }

        bool operator<=(const TIpAddr& addr) const {
            return (High_ < addr.High_) ||
                   (High_ == addr.High_ && Low_ <= addr.Low_);
        }

        bool operator>(const TIpAddr& addr) const {
            return (High_ > addr.High_) ||
                   (High_ == addr.High_ && Low_ > addr.Low_);
        }

        bool operator>=(const TIpAddr& addr) const {
            return (High_ > addr.High_) ||
                   (High_ == addr.High_ && Low_ >= addr.Low_);
        }

        // convert address to string form
        operator TString() const {
            return ToString();
        }
        TString ToString() const;
        TString ToBase64String() const;
        TString ToPackedStringV6() const;
        TString ToPackedStringShortest() const;
        TString ToStringHalfV6() const;

        bool IsLoopback() const;  // 127.0.0.1, ::1, ...
        bool IsPrivate() const;   // 192.168.0.1, 10.0.0.1, ...
        bool IsMulticast() const; // 224.0.0.255, 234.0.0.1, ...

        // apply bit mask to a given address and return range bounds
        TIpAddr GetRangeStart(int mask_width) const;
        TIpAddr GetRangeEnd(int mask_width) const;

        // return next and previous addresses
        TIpAddr Next() const;
        TIpAddr Prev() const;

        // arithmetic helpers
        TIpAddr& operator+=(const TIpAddr& other);
        TIpAddr& operator+=(ui64 n);
        TIpAddr& operator-=(const TIpAddr& other);
        TIpAddr& operator-=(ui64 n);
        TIpAddr& operator<<=(unsigned n);
        TIpAddr& operator>>=(unsigned n);

    private:
        // address in host byte order
        ui64 Low_ = 0;
        ui64 High_ = 0;
        friend struct std::hash<TIpAddr>;
    };

    inline IOutputStream& operator<<(IOutputStream& out, const TIpAddr& addr) {
        out << TString(addr);
        return out;
    }
}

namespace std {
    template <>
    struct hash<NPassport::NUtils::TIpAddr> {
        std::size_t operator()(const NPassport::NUtils::TIpAddr& ip) const {
            return hash<ui64>()(ip.Low_) + hash<ui64>()(ip.High_);
        }
    };
}
