#include "file_logger.h"

#include "registry.h"

#include <passport/infra/libs/cpp/utils/thread_local_id.h>

#include <util/stream/output.h>
#include <util/string/cast.h>
#include <util/system/getpid.h>
#include <util/system/thread.h>

#include <fcntl.h>
#include <sys/stat.h>
#include <sys/uio.h>

namespace NPassport::NUtils {
    static const size_t BUF_SIZE = 512;
    static const size_t RESERVE_COUNT = 10240;
    static const mode_t OPEN_MODE = S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH;

    TFileLogger::TFileLogger(const TString& filename, const TString& level, bool printLevel, const TString& timeFormat, const int linesPerShot)
        : Filename_(filename)
        , TimeFormat_(timeFormat)
        , UseDefaultFormat_(timeFormat == "_DEFAULT_")
        , UseDefaultWithPidFormat_(timeFormat == "_DEFAULT_WITH_PID_")
        , Pid_(IntToString<10>(GetPID()))
        , PrintLevel_(printLevel)
        , NeedReopen_(false)
        , Stopping_(false)
        , Level_(StringToLevel(level))
    {
        Queue_.reserve(RESERVE_COUNT);

        TString::size_type pos = 0;
        while (true) {
            pos = Filename_.find('/', pos + 1);
            if (TString::npos == pos) {
                break;
            }
            TString name = Filename_.substr(0, pos);
            int res = mkdir(name.c_str(), OPEN_MODE | S_IXUSR | S_IXGRP | S_IXOTH);
            if (-1 == res && EEXIST != errno) {
                Cerr << "failed to create dir: " << name << ". Errno: " << errno << Endl;
            }
        }

        Y_ENSURE(OpenFile(), "Failed to open file: " << Filename_);
        WritingThread_ = std::thread([this, linesPerShot]() {
            TThread::SetCurrentThreadName("logger");
            this->WritingThread(linesPerShot);
        });

        TLogRegistry::GetInstance().Add(this);
    }

    TFileLogger::~TFileLogger() {
        TLogRegistry::GetInstance().Remove(this);

        Stopping_.store(true);
        QueueCondition_.notify_one();
        WritingThread_.join();

        if (Fd_ != -1) {
            close(Fd_);
        }
    }

    void TFileLogger::Debug(const char* format, ...) const {
        va_list args;
        va_start(args, format);
        Log(ELevel::DEBUG, format, args);
        va_end(args);
    }

    void TFileLogger::Info(const char* format, ...) const {
        va_list args;
        va_start(args, format);
        Log(ELevel::INFO, format, args);
        va_end(args);
    }

    void TFileLogger::Warning(const char* format, ...) const {
        va_list args;
        va_start(args, format);
        Log(ELevel::WARNING, format, args);
        va_end(args);
    }

    void TFileLogger::Error(const char* format, ...) const {
        va_list args;
        va_start(args, format);
        Log(ELevel::ERROR, format, args);
        va_end(args);
    }

    TFileLogger::TLoggingStream TFileLogger::Debug(size_t reserve) {
        return TLoggingStream(*this, ELevel::DEBUG, reserve);
    }

    TFileLogger::TLoggingStream TFileLogger::Info(size_t reserve) {
        return TLoggingStream(*this, ELevel::INFO, reserve);
    }

    TFileLogger::TLoggingStream TFileLogger::Warning(size_t reserve) {
        return TLoggingStream(*this, ELevel::WARNING, reserve);
    }

    TFileLogger::TLoggingStream TFileLogger::Error(size_t reserve) {
        return TLoggingStream(*this, ELevel::ERROR, reserve);
    }

    TFileLogger::ELevel TFileLogger::GetLevel() const {
        return Level_;
    }

    void TFileLogger::Log(const ELevel level, const char* format, va_list args) const {
        if (level < Level_) {
            return;
        }

        char fmt[BUF_SIZE];
        PrepareFormat(fmt, sizeof(fmt), level, format);

        va_list tmpargs;
        va_copy(tmpargs, args);
        size_t size = vsnprintf(nullptr, 0, fmt, tmpargs);
        va_end(tmpargs);

        if (size > 0) {
            TString data(size, 0);
            vsprintf((char*)data.data(), fmt, args);

            AddToQueue(std::move(data));
        }
    }

    void TFileLogger::Log(const ELevel level, TString&& str) const {
        if (level < Level_) {
            return;
        }

        Log(std::move(str));
    }

    void TFileLogger::Log(TString&& str) const {
        if (!str.EndsWith('\n')) {
            str.push_back('\n');
        }

        AddToQueue(std::move(str));
    }

    void TFileLogger::BuildPrefix(const ELevel level, IOutputStream& out) const {
        char fmt[128];
        size_t size = PrepareFormat(fmt, sizeof(fmt), level, "");
        out << TStringBuf(fmt, size - 1);
    }

    void TFileLogger::Rotate() {
        NeedReopen_ = true;
    }

    bool TFileLogger::OpenFile() {
        if (Fd_ != -1) {
            close(Fd_);
        }
        Fd_ = open(Filename_.c_str(), O_WRONLY | O_CREAT | O_APPEND, OPEN_MODE);
        if (Fd_ == -1) {
            Cerr << "File logger cannot open file for writing: " << Filename_ << Endl;
        }
        return Fd_ != -1;
    }

    size_t TFileLogger::PrepareFormat(char* buf, size_t size, const ELevel level, const char* format) const {
        --size; // for terminating 0x00
        size_t total = 0;

        auto moveBufPtr = [&buf, &size, &total](size_t len) {
            total += len;
            buf += len;
            size -= len;
        };

        if (UseDefaultFormat_ || UseDefaultWithPidFormat_) {
            struct timeval tv {};
            gettimeofday(&tv, nullptr);
            struct tm tm {};
            localtime_r(&tv.tv_sec, &tm);

            moveBufPtr(strftime(buf, size, "%Y-%m-%dT%T.", &tm));
            moveBufPtr(std::sprintf(buf, "%03lu ", ui64(tv.tv_usec / 1000)));
            moveBufPtr(std::sprintf(buf, "%s ", UseDefaultFormat_ ? GetThreadLocalRequestId().c_str() : Pid_.c_str()));
        } else if (!TimeFormat_.empty()) {
            struct tm tm {};
            time_t t = time(nullptr);
            localtime_r(&t, &tm);

            moveBufPtr(strftime(buf, size, TimeFormat_.c_str(), &tm));
            *buf = ' ';
            moveBufPtr(1);
        }

        if (PrintLevel_) {
            moveBufPtr(snprintf(buf, size, "%s: %s\n", LevelToString(level), format));
        } else {
            moveBufPtr(snprintf(buf, size, "%s\n", format));
        }

        return total;
    }

    void TFileLogger::AddToQueue(TString&& str) const {
        std::unique_lock lock(QueueMutex_);
        Queue_.push_back(std::move(str));

        // Trying to decrease csw counter
        // It is not necessary to notify for every row - huge logs cause too many system calls
        // Timers are comfort for human eyes and cheap for system
        auto now = TInstant::Now();
        if (now - LastNotify_ > TDuration::MilliSeconds(500)) {
            QueueCondition_.notify_one(); // system call
            LastNotify_ = now;
        }
    }

    void TFileLogger::WritingThread(const size_t linesPerShot) {
        std::vector<iovec> iovector;
        iovector.resize(linesPerShot);

        std::vector<TString> queueCopy;
        queueCopy.reserve(RESERVE_COUNT);
        while (true) {
            queueCopy.clear();

            {
                std::unique_lock lock(QueueMutex_);
                if (Queue_.empty()) {
                    // timed_wait - protection against loss notify()
                    QueueCondition_.wait_for(lock, std::chrono::milliseconds(500));
                }
                std::swap(queueCopy, Queue_);
            }

            if (queueCopy.empty() && Stopping_.load(std::memory_order_relaxed)) {
                return;
            }

            if (NeedReopen_.load(std::memory_order_relaxed)) {
                NeedReopen_ = false;
                if (OpenFile() && ELevel::INFO >= Level_) {
                    char fmt[64];
                    size_t size = PrepareFormat(fmt, sizeof(fmt), ELevel::INFO, "");
                    TString msg = TStringBuilder() << TStringBuf(fmt, size - 1) << "File reopened" << Endl;
                    queueCopy.insert(queueCopy.cbegin(), std::move(msg));
                }
            }

            if (Fd_ == -1) {
                for (const TString& s : queueCopy) {
                    Cerr << s;
                }
                continue;
            }

            auto curStr = queueCopy.begin();
            while (curStr != queueCopy.end()) {
                ssize_t toWrite = 0;
                size_t endPos = 0;

                for (; endPos < iovector.size() && curStr != queueCopy.end(); ++endPos, ++curStr) {
                    iovector[endPos].iov_base = const_cast<char*>(curStr->data());
                    iovector[endPos].iov_len = curStr->size();
                    toWrite += curStr->size();
                }

                size_t beginPos = 0;
                ssize_t writen = 0;
                while (writen < toWrite) {
                    ssize_t res = ::writev(Fd_, iovector.data() + beginPos, endPos - beginPos);
                    if (res < 0) {
                        Cerr << "Failed to write to log " << Filename_
                             << " : " << strerror(errno) << Endl;
                    } else {
                        writen += res;
                        if (writen >= toWrite) {
                            break;
                        }

                        while (static_cast<size_t>(res) >= iovector[beginPos].iov_len) {
                            res -= iovector[beginPos].iov_len;
                            ++beginPos;
                        }

                        if (res) {
                            iovector[beginPos].iov_len -= res;
                            iovector[beginPos].iov_base = static_cast<char*>(iovector[beginPos].iov_base) + res;
                        }
                    }
                }
            }
        }
    }

    TFileLogger::ELevel TFileLogger::StringToLevel(const TString& level) {
        if (level == "INFO") {
            return ELevel::INFO;
        }
        if (level == "DEBUG") {
            return ELevel::DEBUG;
        }
        if (level == "WARNING") {
            return ELevel::WARNING;
        }
        if (level == "ERROR") {
            return ELevel::ERROR;
        }
        ythrow yexception() << "bad log string to level cast";
    }

    const char* TFileLogger::LevelToString(const TFileLogger::ELevel level) {
        switch (level) {
            case ELevel::INFO:
                return "INFO";
            case ELevel::DEBUG:
                return "DEBUG";
            case ELevel::WARNING:
                return "WARNING";
            case ELevel::ERROR:
                return "ERROR";
        }
    }

    TFileLogger::TLoggingStream::TLoggingStream(TFileLogger& parent, TFileLogger::ELevel lvl, size_t reserve)
        : Parent_(parent)
        , Lvl_(lvl)
    {
        Str.Reserve(reserve);

        Parent_.BuildPrefix(Lvl_, Str);
    }

    TFileLogger::TLoggingStream::~TLoggingStream() {
        try {
            if (!Str.empty()) {
                Parent_.Log(Lvl_, std::move(Str.Str()));
            }
        } catch (...) {
        }
    }
}
