#pragma once

#include "logger.h"

#include <util/datetime/base.h>
#include <util/generic/fwd.h>
#include <util/generic/string.h>
#include <util/string/builder.h>

#include <condition_variable>
#include <mutex>
#include <thread>
#include <vector>

namespace NPassport {
    class TLoggingStream;
}

namespace NPassport::NUtils {
    class TFileLogger: public ILogger {
    public:
        class TLoggingStream;

    public:
        TFileLogger(const TString& filename, const TString& level, bool printLevel, const TString& timeFormat = TString(), const int linesPerShot = 1024);

        ~TFileLogger() override;

        void Y_PRINTF_FORMAT(2, 3) Debug(const char* format, ...) const;
        void Y_PRINTF_FORMAT(2, 3) Info(const char* format, ...) const;
        void Y_PRINTF_FORMAT(2, 3) Warning(const char* format, ...) const;
        void Y_PRINTF_FORMAT(2, 3) Error(const char* format, ...) const;

        TLoggingStream Debug(size_t reserve = 256);
        TLoggingStream Info(size_t reserve = 256);
        TLoggingStream Warning(size_t reserve = 256);
        TLoggingStream Error(size_t reserve = 256);

        ELevel GetLevel() const;

        void Log(const ELevel level, const char* format, va_list args) const override;
        void Log(const ELevel level, TString&& str) const override;
        void Log(TString&& str) const override;

        void BuildPrefix(const ELevel level, IOutputStream& out) const override;

        void Rotate();

        static const char* LevelToString(const ELevel level);

    private:
        bool OpenFile();
        size_t PrepareFormat(char* buf, size_t size, const ELevel level, const char* format) const;

        void AddToQueue(TString&& str) const;

        void WritingThread(const size_t linesPerShot);

        static ELevel StringToLevel(const TString& level);

    private:
        // File name
        const TString Filename_;

        // Time format specification
        const TString TimeFormat_;
        const bool UseDefaultFormat_;
        const bool UseDefaultWithPidFormat_;
        const TString Pid_;

        const bool PrintLevel_;

        // File descriptor
        int Fd_ = -1;
        std::atomic_bool NeedReopen_;

        // Writing queue.
        // All writes happens in separate thread. All someInternal methods just
        // push string into queue and signal conditional variable.

        // Logger is stopping.
        std::atomic<bool> Stopping_;

        // Writing queue.
        mutable std::vector<TString> Queue_;

        // Condition and mutex for signalling.
        mutable std::condition_variable QueueCondition_;
        mutable std::mutex QueueMutex_;

        mutable TInstant LastNotify_;

        // Writing thread.
        std::thread WritingThread_;

        const ELevel Level_;

        friend class NPassport::TLoggingStream;
        friend class TFileLogger::TLoggingStream;
    };

    class TFileLogger::TLoggingStream: TMoveOnly {
    public:
        TLoggingStream(NUtils::TFileLogger& parent, ILogger::ELevel lvl, size_t reserve);
        TLoggingStream(TLoggingStream&&) = default;
        ~TLoggingStream();

        TStringStream Str;

    private:
        NUtils::TFileLogger& Parent_;
        ILogger::ELevel Lvl_;
    };

    template <class T>
    static inline TFileLogger::TLoggingStream&& operator<<(TFileLogger::TLoggingStream&& builder, const T& t) {
        builder.Str << t;
        return std::move(builder);
    }
}
