#include "global.h"

#include "file_logger.h"
#include "std_logger.h"

#include <util/generic/yexception.h>
#include <util/stream/format.h>
#include <util/stream/printf.h>

#include <atomic>

namespace NPassport {
    static NUtils::ILogger::ELevel CastLvl(TLoggingStream::ELevel lvl) {
        switch (lvl) {
            case TLoggingStream::ELevel::Debug:
                return NUtils::ILogger::ELevel::DEBUG;
            case TLoggingStream::ELevel::Info:
                return NUtils::ILogger::ELevel::INFO;
            case TLoggingStream::ELevel::Warning:
                return NUtils::ILogger::ELevel::WARNING;
            case TLoggingStream::ELevel::Error:
                return NUtils::ILogger::ELevel::ERROR;
        }
    }

    TLoggingStream::TLoggingStream(ELevel lvl, size_t reserve, const NUtils::ILogger& parent)
        : Parent_(parent)
        , Lvl_(lvl)
    {
        Str.Reserve(reserve);

        Parent_.BuildPrefix(CastLvl(Lvl_), Str);
    }

    TLoggingStream::~TLoggingStream() {
        try {
            if (!Str.empty()) {
                Parent_.Log(CastLvl(Lvl_), std::move(Str.Str()));
            }
        } catch (...) {
        }
    }

    /*
     * Log is thread-safe.
     * All logging messages sent:
     *  - after Log::init()
     *  - before Log::reset()
     * All logging threads stopped before Log::reset().
     */

    // NOLINTNEXTLINE(cppcoreguidelines-avoid-non-const-global-variables)
    TLog TLog::instance_;

    void TLog::Init(std::shared_ptr<NUtils::ILogger> l) {
        instance_.Impl_ = std::move(l);
    }

    void TLog::Reset() {
        instance_.DefautInit();
    }

    static const TString GOT_SIGHUP = "DEBUG: Got SIGHUP\n";
    static const TString GOT_SIGINT = "DEBUG: Got SIGINT\n";
    static const TString GOT_SIGTERM = "DEBUG: Got SIGTERM\n";
    static const TString GOT_SIGUSR1 = "DEBUG: Got SIGUSR1\n";

    void TLog::LogGotSignal(TLog::ESignal sig) {
        const TString* s;
        switch (sig) {
            case ESignal::Sighup:
                s = &GOT_SIGHUP;
                break;
            case ESignal::Sigint:
                s = &GOT_SIGINT;
                break;
            case ESignal::Sigterm:
                s = &GOT_SIGTERM;
                break;
            case ESignal::Sigusr1:
                s = &GOT_SIGUSR1;
                break;
        }

        ++instance_.Counters_.Debug;
        instance_.Impl_->Log(TString(*s));
    }

    void TLog::Debug(const char* format, ...) {
        va_list args;
        va_start(args, format);
        ++instance_.Counters_.Debug;
        instance_.Impl_->Log(NUtils::ILogger::ELevel::DEBUG, format, args);
        va_end(args);
    }

    void TLog::Info(const char* format, ...) {
        va_list args;
        va_start(args, format);
        ++instance_.Counters_.Info;
        instance_.Impl_->Log(NUtils::ILogger::ELevel::INFO, format, args);
        va_end(args);
    }

    void TLog::Warning(const char* format, ...) {
        va_list args;
        va_start(args, format);
        ++instance_.Counters_.Warning;
        instance_.Impl_->Log(NUtils::ILogger::ELevel::WARNING, format, args);
        va_end(args);
    }

    void TLog::Error(const char* format, ...) {
        va_list args;
        va_start(args, format);
        ++instance_.Counters_.Error;
        instance_.Impl_->Log(NUtils::ILogger::ELevel::ERROR, format, args);
        va_end(args);
    }

    TLog::TLog() {
        DefautInit();
    }

    void TLog::DefautInit() {
        Impl_ = std::make_unique<NUtils::TStdLogger>();
    }

    TLoggingStream TLog::CreateStream(TLoggingStream::ELevel level, size_t reserve) {
        return TLoggingStream(level, reserve, *TLog::instance_.Impl_);
    }

    /*
     * There could be detached threads - the best solution is shared_ptr
     *
     * One more feature: should be printing to stderr in ylast
     */

    TDbPoolLog::TDbPoolLog()
        : Impl_(std::make_unique<NUtils::TStdLogger>(
              NUtils::TStdLoggerSettings{
                  .Group = NUtils::TStdLoggerSettings::EGroup::DbPool,
              }))
    {
    }

    TDbPoolLog::TDbPoolLog(const TString& filename)
        : Impl_(std::make_shared<NUtils::TFileLogger>(filename, "DEBUG", false, "_DEFAULT_"))
    {
    }

    void TDbPoolLog::Debug(const char* format, ...) const {
        va_list args;
        va_start(args, format);
        Impl_->Log(NUtils::ILogger::ELevel::DEBUG, format, args);
        va_end(args);
    }

    void TDbPoolLog::Info(const char* format, ...) const {
        va_list args;
        va_start(args, format);
        Impl_->Log(NUtils::ILogger::ELevel::INFO, format, args);
        va_end(args);
    }

    void TDbPoolLog::Warning(const char* format, ...) const {
        va_list args;
        va_start(args, format);
        Impl_->Log(NUtils::ILogger::ELevel::WARNING, format, args);
        va_end(args);
    }

    void TDbPoolLog::Error(const char* format, ...) const {
        va_list args;
        va_start(args, format);
        Impl_->Log(NUtils::ILogger::ELevel::ERROR, format, args);
        va_end(args);
    }

    TLoggingStream TDbPoolLog::CreateStream(TLoggingStream::ELevel level, size_t reserve) const {
        return TLoggingStream(level, reserve, *Impl_);
    }
}
