#pragma once

#include <passport/infra/libs/cpp/utils/atomic.h>

#include <util/generic/noncopyable.h>
#include <util/generic/string.h>
#include <util/string/builder.h>
#include <util/system/compiler.h>
#include <util/system/types.h>

#include <memory>
#include <mutex>

namespace NPassport::NUtils {
    class ILogger;
}

namespace NPassport {
    class TLoggingStream: TMoveOnly {
    public:
        enum class ELevel {
            Debug,
            Info,
            Warning,
            Error,
        };

        TLoggingStream(ELevel lvl, size_t reserve, const NUtils::ILogger& parent);
        TLoggingStream(TLoggingStream&&) = default;
        ~TLoggingStream();

        TStringStream Str;

    private:
        const NUtils::ILogger& Parent_;
        ELevel Lvl_;
    };

    template <class T>
    static inline TLoggingStream& operator<<(TLoggingStream& builder, const T& t) {
        builder.Str << t;
        return builder;
    }

    template <class T>
    static inline TLoggingStream&& operator<<(TLoggingStream&& builder, const T& t) {
        builder.Str << t;
        return std::move(builder);
    }

    class TLog: TNonCopyable {
    public:
        static void Init(std::shared_ptr<NUtils::ILogger> l);
        static void Reset();

        static void Y_PRINTF_FORMAT(1, 2) Debug(const char* format, ...);
        static void Y_PRINTF_FORMAT(1, 2) Info(const char* format, ...);
        static void Y_PRINTF_FORMAT(1, 2) Warning(const char* format, ...);
        static void Y_PRINTF_FORMAT(1, 2) Error(const char* format, ...);

        static TLoggingStream Debug(size_t reserve = 256) {
            ++instance_.Counters_.Debug;
            return CreateStream(TLoggingStream::ELevel::Debug, reserve);
        }

        static TLoggingStream Info(size_t reserve = 256) {
            ++instance_.Counters_.Info;
            return CreateStream(TLoggingStream::ELevel::Info, reserve);
        }

        static TLoggingStream Warning(size_t reserve = 256) {
            ++instance_.Counters_.Warning;
            return CreateStream(TLoggingStream::ELevel::Warning, reserve);
        }

        static TLoggingStream Error(size_t reserve = 256) {
            ++instance_.Counters_.Error;
            return CreateStream(TLoggingStream::ELevel::Error, reserve);
        }

        enum class ESignal {
            Sighup,
            Sigint,
            Sigterm,
            Sigusr1,
        };

        static void LogGotSignal(ESignal sig);

        struct TCounters {
            NUtils::TAtomicNum<> Debug;
            NUtils::TAtomicNum<> Info;
            NUtils::TAtomicNum<> Warning;
            NUtils::TAtomicNum<> Error;
        };

        static const TCounters& GetCounters() {
            return instance_.Counters_;
        }

    private:
        TLog();

        void DefautInit();
        static TLoggingStream CreateStream(TLoggingStream::ELevel level, size_t reserve);

        TCounters Counters_;
        std::shared_ptr<NUtils::ILogger> Impl_;

        // NOLINTNEXTLINE(cppcoreguidelines-avoid-non-const-global-variables)
        static TLog instance_;
        friend class TLoggingStream;
    };

    class TDbPoolLog {
    public:
        TDbPoolLog();
        TDbPoolLog(const TString& filename);

        TLoggingStream Debug(size_t reserve = 256) const {
            return CreateStream(TLoggingStream::ELevel::Debug, reserve);
        }

        TLoggingStream Info(size_t reserve = 256) const {
            return CreateStream(TLoggingStream::ELevel::Info, reserve);
        }

        TLoggingStream Warning(size_t reserve = 256) const {
            return CreateStream(TLoggingStream::ELevel::Warning, reserve);
        }

        TLoggingStream Error(size_t reserve = 256) const {
            return CreateStream(TLoggingStream::ELevel::Error, reserve);
        }

        void Y_PRINTF_FORMAT(2, 3) Debug(const char* format, ...) const;
        void Y_PRINTF_FORMAT(2, 3) Info(const char* format, ...) const;
        void Y_PRINTF_FORMAT(2, 3) Warning(const char* format, ...) const;
        void Y_PRINTF_FORMAT(2, 3) Error(const char* format, ...) const;

    private:
        TLoggingStream CreateStream(TLoggingStream::ELevel level, size_t reserve) const;

    private:
        std::shared_ptr<NUtils::ILogger> Impl_;
    };
}
