#include "std_logger.h"

#include <util/stream/format.h>
#include <util/stream/printf.h>

namespace NPassport::NUtils {
    TStdLogger::TStdLogger(const TStdLoggerSettings& settings)
        : Settings_(settings)
    {
    }

    void TStdLogger::Log(const ELevel level, const char* format, va_list args) const {
        TStringStream res;
        BuildPrefix(level, res);

        Printf(res, format, args);

        SynLog(std::move(res));
    }

    void TStdLogger::Log(const ELevel level, TString&& str) const {
        TStringStream res;
        BuildPrefix(level, res);

        res << str;

        SynLog(std::move(res));
    }

    void TStdLogger::Log(TString&& str) const {
        Log(ELevel::DEBUG, std::move(str));
    }

    void TStdLogger::BuildPrefix(const ELevel level, IOutputStream& out) const {
        const TInstant now = TInstant::Now();
        out << now.FormatLocalTime("%Y-%m-%d %T") << "." << LeftPad(now.MilliSecondsOfSecond(), 3, '0');

        out << " " << Settings_.Group << " " << level << ": ";
    }

    void TStdLogger::SynLog(TStringStream&& stream) const {
        stream << Endl;

        IOutputStream* out = nullptr;
        switch (Settings_.Destination) {
            case TStdLoggerSettings::EDestination::Stderr:
                out = &Cerr;
                break;
            case TStdLoggerSettings::EDestination::Stdout:
                out = &Cout;
                break;
        }

        // It is thread safe operation: we send the whole string to stderr.
        // If we'll write to stderr with parts like `...<< foo << bar <<...`,
        //   then lines will be mixed.

        (*out) << stream.Str();
    }
}
