#pragma once

#include <util/datetime/base.h>
#include <util/generic/noncopyable.h>
#include <util/generic/string.h>
#include <util/system/event.h>

#include <atomic>
#include <functional>
#include <thread>

namespace NPassport::NUtils {
    class TRegularTask {
    public:
        using TFunc = std::function<void()>;

        TRegularTask(TFunc f,
                     TDuration period,
                     const TString& name = TString());
        virtual ~TRegularTask();

        void SetPeriod(TDuration p) {
            Period_ = p;
        }

        void RunNow();

    protected:
        virtual TDuration GetTimeToSleep() const;

        std::atomic<TDuration> Period_;
        TAutoEvent Event_;

    private:
        void RefreshingThread(TString name);

        const TFunc F_;
        std::thread RefreshingThread_;
        std::atomic_bool Stopping_;
    };

    /*
     * Decorator allowes:
     * 1. to be sure that thread will not access already destroyed with dtor members
     * 2. simplify ctor of T: no `TDuration period` in ctor params
     */
    template <class T>
    class TRegularTaskDecorator: public T, TNonCopyable {
    public:
        struct TDecoratorArgs {
            TDuration Period;
            TString Name;
        };

        template <typename... VarArgs>
        TRegularTaskDecorator(const TDecoratorArgs& decoratorArgs,
                              VarArgs&&... varArgs)
            : T(std::forward<VarArgs>(varArgs)...)
            , Task_([this]() { T::Run(); }, decoratorArgs.Period, decoratorArgs.Name)
        {
        }

        template <typename... VarArgs>
        static std::shared_ptr<TRegularTaskDecorator<T>> CreateShared(
            const TDecoratorArgs& decoratorArgs,
            VarArgs&&... varArgs) {
            return std::make_shared<TRegularTaskDecorator<T>>(decoratorArgs, std::forward<VarArgs>(varArgs)...);
        }

        template <typename... VarArgs>
        static std::unique_ptr<TRegularTaskDecorator<T>> CreateUnique(
            const TDecoratorArgs& decoratorArgs,
            VarArgs&&... varArgs) {
            return std::make_unique<TRegularTaskDecorator<T>>(decoratorArgs, std::forward<VarArgs>(varArgs)...);
        }

        ~TRegularTaskDecorator() override = default;

        void RunRegularTaskNow() {
            Task_.RunNow();
        }

    private:
        TRegularTask Task_;
    };
}
