#pragma once

#include <util/generic/strbuf.h>
#include <util/generic/string.h>
#include <util/generic/yexception.h>
#include <util/stream/format.h>

namespace NPassport::NUtils {
    // Base64
    TString Bin2base64url(const TStringBuf buf, bool pad = false);
    TString Base64url2bin(const TStringBuf buf);

    bool IsBase64url(const TStringBuf str);

    TString BinToBase64(const TStringBuf buf, bool pad = true);
    TString Base64ToBin(const TStringBuf buf);

    bool IsBase64(const TStringBuf str);

    // URL
    TString Urlencode(const TStringBuf in);
    TString Urldecode(const TStringBuf in);

    // Hex
    TString Bin2hex(const TStringBuf buf);
    TString Hex2bin(const TStringBuf buf);

    // Base32-like, non-standard alphabet!
    TString BinToBase32(const TStringBuf buf);
    TString Base32ToBin(const TStringBuf buf);
    unsigned char BinToBase32(ui8 val);
    ui8 Base32ToBin(unsigned char val);

    // put integer value to/from byte buffer
    // Little Endian
    template <typename T>
    void ToBytesLsb(T value, TString& buf, size_t pos = 0) {
        static_assert(std::is_integral_v<T>, "Only integer types are supported");
        Y_ENSURE(pos + sizeof(T) <= buf.size(), "Buffer position out of range");
        for (size_t i = 0; i < sizeof(T); ++i) {
            buf[pos + i] = (ui8)((value >> (8 * i)) & 0xff);
        }
    }

    // Little Endian
    template <typename T>
    T FromBytesLsb(TStringBuf buf, size_t pos = 0) {
        static_assert(std::is_integral_v<T>, "Only integer types are supported");
        Y_ENSURE(pos + sizeof(T) <= buf.size(), "Buffer position out of range");
        T result = 0;
        for (size_t i = 0; i < sizeof(T); ++i) {
            ui8 byte = buf[pos + i];
            result |= ((T)byte) << (8 * i);
        }
        return result;
    }

    // Big Endian
    template <typename T>
    void ToBytesMsb(T value, TString& buf, size_t pos = 0) {
        static_assert(std::is_integral_v<T>, "Only integer types are supported");
        Y_ENSURE(pos + sizeof(T) <= buf.size(), "Buffer position out of range");
        for (size_t i = 0; i < sizeof(T); ++i) {
            buf[pos + i] = (ui8)((value >> (8 * (sizeof(T) - 1 - i))) & 0xff);
        }
    }

    // Big Endian
    template <typename T>
    T FromBytesMsb(TStringBuf buf, size_t pos = 0) {
        static_assert(std::is_integral_v<T>, "Only integer types are supported");
        Y_ENSURE(pos + sizeof(T) <= buf.size(), "Buffer position out of range");
        T result = 0;
        for (size_t i = 0; i < sizeof(T); ++i) {
            ui8 byte = buf[pos + i];
            result |= ((T)byte) << (8 * (sizeof(T) - 1 - i));
        }
        return result;
    }
}
