#include "format.h"

#include "string_utils.h"

#include <util/datetime/base.h>
#include <util/string/cast.h>

#include <algorithm>
#include <time.h>

namespace NPassport::NUtils {
    TString FormatTimestamp(const TStringBuf ts) {
        try {
            return FormatTimestamp(IntFromString<time_t, 10>(ts));
        } catch (const std::exception&) {
            return TString();
        }
    }

    TString FormatTimestamp(const time_t ts) {
        return TInstant::Seconds(ts).FormatLocalTime("%Y-%m-%d %H:%M:%S");
    }

    // TODO Probably all escapings are not efficient. Mey be need optimizing
    static const char EOL_CHARS[] = "\\\n\r\v\t\0";

    TString& EscapeEol(TString& src) {
        TString dst;
        TString::size_type pos = src.find_first_of(TStringBuf(EOL_CHARS, sizeof(EOL_CHARS)));
        if (pos == TString::npos) {
            return src;
        }
        dst.reserve(src.size() * 2);
        dst.assign(src, 0, pos);
        for (TString::const_iterator it = src.cbegin() + pos; it != src.cend(); ++it) {
            switch (*it) {
                case '\n':
                    dst.push_back('\\');
                    dst.push_back('n');
                    break;
                case '\r':
                    dst.push_back('\\');
                    dst.push_back('r');
                    break;
                case '\\':
                    dst.push_back('\\');
                    dst.push_back('\\');
                    break;
                case '\v':
                    dst.push_back('\\');
                    dst.push_back('v');
                    break;
                case '\t':
                    dst.push_back('\\');
                    dst.push_back('t');
                    break;
                case '\0':
                    dst.push_back('\\');
                    dst.push_back('0');
                    break;
                default:
                    dst.push_back(*it);
                    break;
            }
        }
        src = std::move(dst);
        return src;
    }

    static const char HEX_DIGITS[] = "0123456789ABCDEF";
    TString& EscapeUnprintable(TString& src) {
        auto it = std::find_if_not(src.cbegin(), src.cend(), &isprint);
        if (it == src.cend()) {
            return src;
        }

        TString dst;
        dst.reserve(src.size() * 2);
        dst.assign(src.cbegin(), it);
        for (const char c : TStringBuf(it, src.cend())) {
            switch (c) {
                case '\n':
                    dst.push_back('\\');
                    dst.push_back('n');
                    break;
                case '\r':
                    dst.push_back('\\');
                    dst.push_back('r');
                    break;
                case '\\':
                    dst.push_back('\\');
                    dst.push_back('\\');
                    break;
                case '\v':
                    dst.push_back('\\');
                    dst.push_back('v');
                    break;
                case '\t':
                    dst.push_back('\\');
                    dst.push_back('t');
                    break;
                default:
                    if (isprint(c)) {
                        dst.push_back(c);
                    } else {
                        dst.push_back('<');
                        dst.push_back(HEX_DIGITS[0xf & (c >> 4)]);
                        dst.push_back(HEX_DIGITS[0xf & c]);
                        dst.push_back('>');
                    }
                    break;
            }
        }
        src = std::move(dst);
        return src;
    }

    static char EolToSpaceChar(char c) {
        return (c == '\n' || c == '\r') ? ' ' : c;
    }

    TString EolToSpace(const TStringBuf src) {
        TString buf;
        buf.reserve(src.size() + 10);
        std::transform(src.begin(), src.end(), std::back_inserter(buf), EolToSpaceChar);
        return buf;
    }

    TString Unquote(const TStringBuf src) {
        if (src.size() > 1 && src[0] == '"' && src.back() == '"') {
            return TString(src.substr(1, src.size() - 2));
        }
        return TString(src);
    }

    TString TolowerCopy(const TStringBuf str) {
        TString result;
        result.reserve(str.size());
        int (*func)(int) = &std::tolower;
        std::transform(str.begin(), str.end(), std::back_inserter(result), func);
        return result;
    }

    TString& Tolower(TString& str) {
        int (*func)(int) = &std::tolower;
        std::transform(str.begin(), str.vend(), str.begin(), func);
        return str;
    }

    TString ToupperCopy(const TStringBuf str) {
        TString result;
        result.reserve(str.size());
        int (*func)(int) = &std::toupper;
        std::transform(str.begin(), str.end(), std::back_inserter(result), func);
        return result;
    }

    TString& Toupper(TString& str) {
        int (*func)(int) = &std::toupper;
        std::transform(str.begin(), str.vend(), str.begin(), func);
        return str;
    }

    TString RemoveSpaces(const TStringBuf str) {
        TString result;
        result.reserve(str.size());
        for (char c : str) {
            if (c != ' ' && c != '\t') {
                result.push_back(c);
            }
        }
        return result;
    }

    TString& Trim(TString& str) {
        if (!str.empty() && isspace(str.back())) {
            size_t len = 1;
            while (len < str.size() && isspace(str.at(str.size() - len - 1))) {
                ++len;
            }
            str.erase(str.size() - len);
        }

        if (!str.empty() && isspace(str.front())) {
            auto it = std::find_if_not(str.cbegin(), str.cend(), isspace);
            str.erase(str.cbegin(), it);
        }

        return str;
    }

    TStringBuf& Trim(TStringBuf& str) {
        if (str && isspace(str.back())) {
            size_t len = 1;
            while (len < str.size() && isspace(str.at(str.size() - len - 1))) {
                ++len;
            }
            str.Chop(len);
        }

        if (str && isspace(str[0])) {
            auto it = std::find_if_not(str.begin(), str.end(), isspace);
            str.Skip(it - str.begin());
        }

        return str;
    }

    TString TrimCopy(const TString& str) {
        TString res(str);
        return Trim(res);
    }
}
