#pragma once

#include "format.h"

#include <util/generic/strbuf.h>
#include <util/generic/string.h>
#include <util/string/split.h>

#include <algorithm>
#include <vector>

namespace NPassport::NUtils {
    template <class Str = TString>
    static std::vector<Str> ToVector(const TStringBuf str, const char* delim, size_t reserve = 16);
    template <class Str = TString>
    static std::vector<Str> ToVector(TStringBuf str, const char delim, size_t reserve = 16);

    template <class Str = TString>
    static std::vector<Str> ToVectorWithEmpty(TStringBuf str, const char delim, size_t reserve = 16);

    template <class Str = TString>
    static std::vector<Str> NormalizeListValue(const TStringBuf str, const char* splitter);

    template <typename Func>
    static void Transform(const TStringBuf str, char delim, Func func);
    template <typename Func>
    static void Transform(const TStringBuf str, const char* delim, Func func);

    namespace NPrivate {
        template <typename Iter, typename Eq, typename Op>
        void SplitImpl(Iter startit, Iter endit, Eq isDelimiter, Op oper) {
            Iter curit = startit;
            Iter lastit = curit;
            while (true) {
                if (curit == endit) {
                    oper(TStringBuf(lastit, curit));
                    break;
                }
                if (isDelimiter(*curit)) {
                    oper(TStringBuf(lastit, curit));
                    ++curit;
                    lastit = curit;
                } else {
                    ++curit;
                }
            }
        }
    }

    template <class Str>
    std::vector<Str> ToVector(const TStringBuf str, const char* delim, size_t reserve) {
        if (str.empty()) {
            return {};
        }

        std::vector<Str> res;
        res.reserve(reserve);

        NPrivate::SplitImpl(
            str.begin(),
            str.end(),
            [delim](const char other) -> bool {
                if (delim == nullptr || *delim == 0) {
                    return false;
                }
                const char* it = delim;
                do {
                    if (*it == other) {
                        return true;
                    }
                } while (*++it != 0);
                return false;
            },
            [&res](TStringBuf buf) {
                if (buf) {
                    res.push_back(Str(buf));
                }
            });

        return res;
    }

    template <class Str>
    std::vector<Str> ToVector(TStringBuf str, const char delim, size_t reserve) {
        if (str.empty()) {
            return {};
        }

        std::vector<Str> res;
        res.reserve(reserve);

        while (str) {
            TStringBuf s = str.NextTok(delim);
            if (s) {
                res.push_back(Str(s));
            }
        }

        return res;
    }

    template <class Str>
    std::vector<Str> ToVectorWithEmpty(TStringBuf str, const char delim, size_t reserve) {
        if (str.empty()) {
            return {};
        }

        std::vector<Str> res;
        res.reserve(reserve);

        NPrivate::SplitImpl(
            str.begin(),
            str.end(),
            [delim](const char other) -> bool { return delim == other; },
            [&res](const TStringBuf buf) {
                res.push_back(Str(buf));
            });

        return res;
    }

    /**
     * Mutations:
     * 0. split
     * 1. trim
     * 2. drop empty
     * 3. sort
     * 4. deduplicate
     */
    template <class Str>
    std::vector<Str> NormalizeListValue(const TStringBuf str, const char* splitter) {
        std::vector<Str> res = ToVector<Str>(str, splitter);

        // TODO use splitter like ", ". Then we can drop this code
        std::for_each(res.begin(), res.end(), [](auto& s) { Trim(s); });
        res.erase(std::remove_if(res.begin(), res.end(), [](const auto& s) { return s.empty(); }), res.end());
        //

        std::sort(res.begin(), res.end());
        res.erase(std::unique(res.begin(), res.end()), res.end());
        return res;
    }

    // Expecting signature for Func
    // [](const TStringBuf buf) -> void {}
    template <typename Func>
    void Transform(const TStringBuf str, char delim, Func func) {
        NPrivate::SplitImpl(
            str.begin(),
            str.end(),
            [delim](char c) -> bool { return c == delim; },
            func);
    }

    template <typename Func>
    void Transform(const TStringBuf str, const char* delim, Func func) {
        NPrivate::SplitImpl(
            str.begin(),
            str.end(),
            [delim](const char other) -> bool {
                if (delim == nullptr || *delim == 0) {
                    return false;
                }
                const char* it = delim;
                do {
                    if (*it == other) {
                        return true;
                    }
                } while (*++it != 0);
                return false;
            },
            func);
    }

}
