#include "string_utils.h"

#include "format.h"

#include <util/string/ascii.h>
#include <util/string/cast.h>

#include <algorithm>
#include <string>

namespace NPassport::NUtils {
    bool NotDigit(char c) {
        return c < '0' || c > '9';
    }

    bool DigitsOnly(const TStringBuf str) {
        return std::find_if(str.begin(), str.end(), &NotDigit) == str.end();
    }

    bool HexDigitsOnly(const TStringBuf str) {
        return std::find_if_not(str.begin(), str.end(), [](char c) {
                   return (c >= '0' && c <= '9') || (c >= 'a' && c <= 'f') || (c >= 'A' && c <= 'F');
               }) == str.end();
    }

    bool LowercaseLettersOnly(const TStringBuf str) {
        return std::find_if_not(str.begin(), str.end(), [](char c) {
                   return c >= 'a' && c <= 'z';
               }) == str.end();
    }

    // Equivalent to regex: ^(http(s)?://)?([[:alnum:]._-]+)(:[0-9]+)?([/?]|$)
    TString HostFromUri(TStringBuf uri) {
        if (uri.StartsWith("http")) {
            uri.Skip(4);
            if (uri && uri[0] == 's') {
                uri.Skip(1);
            }
            if (!uri.StartsWith("://")) {
                return {};
            }
            uri.Skip(3);
        }

        if (!uri.empty() && ((uri.back() == '?') || uri.back() == '/')) {
            uri.Chop(1);
        }

        const size_t tail = 6;
        if (uri.size() > tail) {
            size_t pos = uri.Last(tail).rfind(':');
            if (pos != TStringBuf::npos) {
                if (!DigitsOnly(uri.Last(tail - pos - 1))) {
                    return {};
                }
                uri.Chop(tail - pos);
            }
        }

        for (const char c : uri) {
            if ((c >= 'A' && c <= 'Z') ||
                (c >= 'a' && c <= 'z') ||
                (c >= '0' && c <= '9') ||
                c == '.' || c == '-' || c == '_') {
                continue;
            }
            return {};
        }

        TString res(uri);
        Tolower(res);
        return res;
    }

    bool SecureCompare(const TStringBuf actual, const TStringBuf expected) {
        return SecureCompare(actual.data(), actual.size(), expected.data(), expected.size());
    }

    bool SecureCompare(const char* actual, size_t actualLen, const char* expected, size_t expectedLen) {
        bool result = true;

        size_t len = 0;
        if (actual) {
            if (actualLen == expectedLen) {
                len = actualLen;
            } else {
                result = false;
                len = std::min(actualLen, expectedLen);
            }
        } else {
            len = expectedLen;
            actual = expected;
            result = false;
        }

        for (; len > 0; --len, ++actual, ++expected) {
            if (*actual != *expected) {
                result = false;
            }
        }

        return result;
    }

    bool ToBoolean(const TString& str) {
        return AsciiEqualsIgnoreCase(str.c_str(), "yes") ||
               AsciiEqualsIgnoreCase(str.c_str(), "true") ||
               AsciiEqualsIgnoreCase(str.c_str(), "1");
    }

    template <typename T>
    static T ToIntImpl(TStringBuf strval, TStringBuf name) {
        T value;
        if (!TryIntFromString<10>(strval, value)) {
            throw std::invalid_argument(CreateStr("invalid ", name, " value: ", strval).c_str());
        }
        return value;
    }

    i64 ToInt(TStringBuf strval, TStringBuf name) {
        return ToIntImpl<i64>(strval, name);
    }

    ui64 ToUInt(TStringBuf strval, TStringBuf name) {
        return ToIntImpl<ui64>(strval, name);
    }

    static size_t CheckRegularUtfBytes(TStringBuf::const_iterator it,
                                       TStringBuf::const_iterator end,
                                       size_t len,
                                       const std::pair<ui8, ui8> bordersForSecondByte = {0x80, 0xBF}) {
        if (++it == end ||
            (ui8)*it < bordersForSecondByte.first ||
            (ui8)*it > bordersForSecondByte.second) {
            return 0;
        }

        ++it;
        for (size_t idx = 2; idx < len; ++idx, ++it) {
            if (it == end || (ui8)*it < 0x80 || (ui8)*it > 0xBF) {
                return 0;
            }
        }

        return len;
    }

    size_t Utf8CharLength(TStringBuf::const_iterator it, TStringBuf::const_iterator end) {
        if (it == end) {
            return 0;
        }

        // https://docs.oracle.com/cd/E18283_01/server.112/e10729/appunicode.htm#CACHBDGH

        // ASCII char
        if ((unsigned char)*it <= 0x7F) {
            return 1;
        }

        // two-byte char
        // should be >= 0xC0, but 0xC0 and 0xC1 are forbidden chars - they lead to ascii symbols
        if ((unsigned char)*it >= 0xC2 && (unsigned char)*it <= 0xDF) {
            return CheckRegularUtfBytes(it, end, 2);
        }

        // three-byte char
        if ((unsigned char)*it == 0xE0) {
            return CheckRegularUtfBytes(it, end, 3, {0xA0, 0xBF});
        }
        if ((unsigned char)*it == 0xED) {
            return CheckRegularUtfBytes(it, end, 3, {0x80, 0x9F});
        }
        if ((unsigned char)*it >= 0xE1 && (unsigned char)*it <= 0xEF) {
            return CheckRegularUtfBytes(it, end, 3);
        }

        // four-byte char. max utf8 symbol: F4 8F BF BF (codepoint 10FFFF)
        if ((unsigned char)*it == 0xF0) {
            return CheckRegularUtfBytes(it, end, 4, {0x90, 0xBF});
        }
        if ((unsigned char)*it >= 0xF1 && (unsigned char)*it <= 0xF3) {
            return CheckRegularUtfBytes(it, end, 4);
        }
        if ((unsigned char)*it == 0xF4) {
            return CheckRegularUtfBytes(it, end, 4, {0x80, 0x8F});
        }

        return 0;
    }

    TString ReplaceAny(const TStringBuf source, const TStringBuf chars, const TStringBuf str) {
        TString res;
        res.reserve(source.size());

        TString::size_type left = 0;
        TString::size_type right;
        while ((right = source.find_first_of(chars, left)) != TStringBuf::npos) {
            TStringBuf sub = source.substr(left, right - left);
            res.append(sub.begin(), sub.end()).append(str);
            if (right == source.size() - 1) {
                return res;
            }
            left = right + 1;
        }
        TStringBuf sub = source.substr(left, TStringBuf::npos);
        res.append(sub.begin(), sub.end());
        return res;
    }
}
