#pragma once

#include <util/generic/strbuf.h>
#include <util/generic/string.h>
#include <util/string/cast.h>

#include <list>
#include <vector>

namespace NPassport::NUtils::NPrivate {
    template <typename T>
    inline size_t CalcSize(const T& s) {
        using TActual = std::remove_cv_t<T>;

        if constexpr (std::is_integral_v<TActual> || std::is_enum_v<T>) {
            static_assert(sizeof(TActual) <= 8);
            return 16;
        } else {
            TStringBuf tmp(s);
            return tmp.size();
        }
    }

    template <typename T, typename... Str>
    size_t CalcSize(const T& s, const Str&... values) {
        return CalcSize(s) + CalcSize(values...);
    }

    template <typename T>
    void Append(TStringOutput& target, const T& s) {
        static_assert(!std::is_same_v<T, ui8>, "ui8 serialized as char, not number");
        static_assert(!std::is_same_v<T, i8>, "i8 serialized as char, not number");
        target << s;
    }

    template <typename T, typename... Str>
    void Append(TStringOutput& target, const T& s, const Str&... values) {
        Append(target, s);
        Append(target, values...);
    }
}

namespace NPassport::NUtils {
    bool NotDigit(char c);
    bool DigitsOnly(const TStringBuf str);
    bool HexDigitsOnly(const TStringBuf str);
    bool LowercaseLettersOnly(const TStringBuf str);

    TString HostFromUri(const TStringBuf uri);

    bool SecureCompare(const TStringBuf actual, const TStringBuf expected);
    bool SecureCompare(const char* actual, size_t actualLen, const char* expected, size_t expectedLen);

    bool ToBoolean(const TString& str);

    i64 ToInt(TStringBuf strval, TStringBuf name);
    ui64 ToUInt(TStringBuf strval, TStringBuf name);

    size_t Utf8CharLength(TStringBuf::const_iterator it, TStringBuf::const_iterator end);

    TString ReplaceAny(const TStringBuf source, const TStringBuf chars, const TStringBuf str);

    template <typename... Str>
    TString& AppendExt(TString& str, size_t extSize, const Str&... arg) {
        TStringOutput out(str);
        str.reserve(str.size() + NPrivate::CalcSize(arg...) + extSize);
        NPrivate::Append(out, arg...);
        return str;
    }

    template <typename... Str>
    TString& Append(TString& str, const Str&... arg) {
        return AppendExt(str, 0, arg...);
    }

    template <typename... Str>
    TString CreateStrExt(size_t extSize, const Str&... arg) {
        TString str;
        AppendExt(str, extSize, arg...);
        return str;
    }

    template <typename... Str>
    TString CreateStr(const Str&... arg) {
        return CreateStrExt(0, arg...);
    }

    template <typename Str>
    TString& AppendSeparated(TString& buffer, Str separator, TStringBuf str) {
        if (buffer && str) {
            return Append(buffer, separator, str);
        }
        return Append(buffer, str);
    }

    inline TString& AddMessage(TString& buffer, const TString& msg) {
        return AppendSeparated(buffer, ", ", msg);
    }
}
