#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport::NUtils;

Y_UNIT_TEST_SUITE(PasspUtilsCoder) {
    Y_UNIT_TEST(simpleBase64Test) {
        UNIT_ASSERT_VALUES_EQUAL("", BinToBase64(""));
        UNIT_ASSERT_VALUES_EQUAL("+g", BinToBase64("\xfa", false));
        UNIT_ASSERT_VALUES_EQUAL("+g==", BinToBase64("\xfa", true));
        UNIT_ASSERT_VALUES_EQUAL("+hE", BinToBase64("\xfa\x11", false));
        UNIT_ASSERT_VALUES_EQUAL("+hE=", BinToBase64("\xfa\x11", true));
        UNIT_ASSERT_VALUES_EQUAL("+hHx", BinToBase64("\xfa\x11\xf1", false));
        UNIT_ASSERT_VALUES_EQUAL("+hHx", BinToBase64("\xfa\x11\xf1", true));
        UNIT_ASSERT_VALUES_EQUAL("+hHx/g", BinToBase64("\xfa\x11\xf1\xfe", false));
        UNIT_ASSERT_VALUES_EQUAL("+hHx/g==", BinToBase64("\xfa\x11\xf1\xfe", true));
        UNIT_ASSERT_VALUES_EQUAL("+hHx/v8", BinToBase64("\xfa\x11\xf1\xfe\xff", false));
        UNIT_ASSERT_VALUES_EQUAL("+hHx/v8=", BinToBase64("\xfa\x11\xf1\xfe\xff", true));

        UNIT_ASSERT_VALUES_EQUAL("", Base64ToBin("hHx--"));
        UNIT_ASSERT_VALUES_EQUAL("", Base64ToBin("&*^"));

        UNIT_ASSERT_VALUES_EQUAL("", Base64ToBin(""));
        UNIT_ASSERT_VALUES_EQUAL("He", Base64ToBin("SGV"));
        UNIT_ASSERT_VALUES_EQUAL("Hello, ", Base64ToBin("SGVsbG8sIG"));
        UNIT_ASSERT_VALUES_EQUAL("Hello, ev", Base64ToBin("SGVsbG8sIGV2"));
        UNIT_ASSERT_VALUES_EQUAL("", Base64ToBin("SGVsbG8sIGV2Z"));
        UNIT_ASSERT_VALUES_EQUAL("+hE=", BinToBase64("\xfa\x11"));
        UNIT_ASSERT_VALUES_EQUAL("+hHx", BinToBase64("\xfa\x11\xf1"));
        UNIT_ASSERT_VALUES_EQUAL("+hHx/v//", BinToBase64("\xfa\x11\xf1\xfe\xff\xff"));

        UNIT_ASSERT_VALUES_EQUAL("", Base64ToBin("+"));
        UNIT_ASSERT_VALUES_EQUAL("\xfa", Base64ToBin("+g"));
        UNIT_ASSERT_VALUES_EQUAL("\xfa\x11", Base64ToBin("+hE"));
        UNIT_ASSERT_VALUES_EQUAL("\xfa\x11\xf1", Base64ToBin("+hHx"));
        UNIT_ASSERT_VALUES_EQUAL("\xfa\x11\xf1\xfe", Base64ToBin("+hHx/g"));
        UNIT_ASSERT_VALUES_EQUAL("\xfa\x11\xf1\xfe", Base64ToBin("+hHx/g="));
        UNIT_ASSERT_VALUES_EQUAL("\xfa\x11\xf1\xfe", Base64ToBin("+hHx/g=="));
        UNIT_ASSERT_VALUES_EQUAL("\xfa\x11\xf1\xfe", Base64ToBin("+hHx/g======="));
        UNIT_ASSERT_VALUES_EQUAL("", Base64ToBin("+hHx/g==a"));
        UNIT_ASSERT_VALUES_EQUAL("\xfa\x11\xf1\xfe\xff", Base64ToBin("+hHx/v8"));
        UNIT_ASSERT_VALUES_EQUAL("\xfa\x11\xf1\xfe\xff", Base64ToBin("+hHx/v8============"));
        UNIT_ASSERT_VALUES_EQUAL("", Base64ToBin("+hHx/v8=n==="));
        UNIT_ASSERT_VALUES_EQUAL("", Base64ToBin("+hHx/v8;"));

        UNIT_ASSERT(IsBase64(""));
        UNIT_ASSERT(IsBase64("+"));
        UNIT_ASSERT(IsBase64("/"));
        UNIT_ASSERT(IsBase64("A=="));
        UNIT_ASSERT(IsBase64("ABC="));
        UNIT_ASSERT(IsBase64("AB+C=========="));
        UNIT_ASSERT(IsBase64("ab+/"));
        UNIT_ASSERT(!IsBase64("a b"));
        UNIT_ASSERT(!IsBase64("a-b"));
        UNIT_ASSERT(!IsBase64("a_b"));
        UNIT_ASSERT(!IsBase64("ABC;"));
        UNIT_ASSERT(!IsBase64("ABC===D"));
        UNIT_ASSERT(!IsBase64("ABC======D===="));
    }

    Y_UNIT_TEST(simpleBase64urlTest) {
        UNIT_ASSERT_VALUES_EQUAL("", Bin2base64url(""));
        UNIT_ASSERT_VALUES_EQUAL("-hHx", Bin2base64url("\xfa\x11\xf1", false));
        UNIT_ASSERT_VALUES_EQUAL("-hHx", Bin2base64url("\xfa\x11\xf1", true));
        UNIT_ASSERT_VALUES_EQUAL("-hHx_g", Bin2base64url("\xfa\x11\xf1\xfe", false));
        UNIT_ASSERT_VALUES_EQUAL("-hHx_g==", Bin2base64url("\xfa\x11\xf1\xfe", true));
        UNIT_ASSERT_VALUES_EQUAL("-hHx_v8", Bin2base64url("\xfa\x11\xf1\xfe\xff", false));
        UNIT_ASSERT_VALUES_EQUAL("-hHx_v8=", Bin2base64url("\xfa\x11\xf1\xfe\xff", true));

        UNIT_ASSERT_VALUES_EQUAL("", Base64url2bin("hHx++"));
        UNIT_ASSERT_VALUES_EQUAL("", Base64url2bin("&*^"));

        UNIT_ASSERT_VALUES_EQUAL("", Base64url2bin(""));
        UNIT_ASSERT_VALUES_EQUAL("He", Base64url2bin("SGV"));
        UNIT_ASSERT_VALUES_EQUAL("Hello, ", Base64url2bin("SGVsbG8sIG"));
        UNIT_ASSERT_VALUES_EQUAL("Hello, ev", Base64url2bin("SGVsbG8sIGV2"));
        UNIT_ASSERT_VALUES_EQUAL("", Base64url2bin("SGVsbG8sIGV2Z"));
        UNIT_ASSERT_VALUES_EQUAL("-hE", Bin2base64url("\xfa\x11"));
        UNIT_ASSERT_VALUES_EQUAL("-hHx", Bin2base64url("\xfa\x11\xf1"));
        UNIT_ASSERT_VALUES_EQUAL("-hHx_v__", Bin2base64url("\xfa\x11\xf1\xfe\xff\xff"));

        UNIT_ASSERT_VALUES_EQUAL("\xfa\x11\xf1", Base64url2bin("-hHx"));
        UNIT_ASSERT_VALUES_EQUAL("\xfa\x11\xf1\xfe", Base64url2bin("-hHx_g"));
        UNIT_ASSERT_VALUES_EQUAL("\xfa\x11\xf1\xfe", Base64url2bin("-hHx_g="));
        UNIT_ASSERT_VALUES_EQUAL("\xfa\x11\xf1\xfe", Base64url2bin("-hHx_g=="));
        UNIT_ASSERT_VALUES_EQUAL("\xfa\x11\xf1\xfe", Base64url2bin("-hHx_g======="));
        UNIT_ASSERT_VALUES_EQUAL("", Base64url2bin("-hHx_g==+"));
        UNIT_ASSERT_VALUES_EQUAL("\xfa\x11\xf1\xfe\xff", Base64url2bin("-hHx_v8"));
        UNIT_ASSERT_VALUES_EQUAL("\xfa\x11\xf1\xfe\xff", Base64url2bin("-hHx_v8============"));
        UNIT_ASSERT_VALUES_EQUAL("", Base64url2bin("-hHx_v8=/==="));
        UNIT_ASSERT_VALUES_EQUAL("", Base64url2bin("-hHx_v8=;"));

        UNIT_ASSERT(IsBase64url(""));
        UNIT_ASSERT(IsBase64url("-"));
        UNIT_ASSERT(IsBase64url("_"));
        UNIT_ASSERT(IsBase64url("A=="));
        UNIT_ASSERT(IsBase64url("ABC="));
        UNIT_ASSERT(IsBase64url("AB-C=========="));
        UNIT_ASSERT(IsBase64url("ab-_"));
        UNIT_ASSERT(!IsBase64url("a b"));
        UNIT_ASSERT(!IsBase64url("a+b"));
        UNIT_ASSERT(!IsBase64url("a/b"));
        UNIT_ASSERT(!IsBase64url("ABC;"));
        UNIT_ASSERT(!IsBase64url("ABC===;"));
        UNIT_ASSERT(!IsBase64url("ABC======-===="));
    }

    Y_UNIT_TEST(urlTest) {
        UNIT_ASSERT_VALUES_EQUAL("2%3A_-.%2B%40%24%25", Urlencode("2:_-.+@$%"));
        UNIT_ASSERT_VALUES_EQUAL("2++++++", Urlencode("2      "));
        UNIT_ASSERT_VALUES_EQUAL("http%3A%3A%2Fwww.yandex.ru%2F", Urlencode("http::/www.yandex.ru/"));
        UNIT_ASSERT_VALUES_EQUAL("", Urlencode(""));

        UNIT_ASSERT_VALUES_EQUAL("2:_-.+@$%", Urldecode("2%3A_-.%2B%40%24%25"));
        UNIT_ASSERT_VALUES_EQUAL("2      ", Urldecode("2++++++"));
        UNIT_ASSERT_VALUES_EQUAL("http::/www.yandex.ru/", Urldecode("http%3A%3A%2Fwww.yandex.ru%2F"));
        UNIT_ASSERT_VALUES_EQUAL("", Urldecode(""));
        UNIT_ASSERT_VALUES_EQUAL("2:_-.+@$%", Urldecode("2%3A_-.%2B%40%24%"));
        UNIT_ASSERT_NO_EXCEPTION(Urldecode("2++%00-.-"));
    }

    Y_UNIT_TEST(hexTest) {
        TString nullStr(1, 0);
        UNIT_ASSERT_VALUES_EQUAL("323a5f2b402425", Bin2hex("2:_+@$%"));
        UNIT_ASSERT_VALUES_EQUAL("32202020202020", Bin2hex("2      "));
        UNIT_ASSERT_VALUES_EQUAL("00", Bin2hex(nullStr));
        UNIT_ASSERT_VALUES_EQUAL("", Bin2hex(""));
        UNIT_ASSERT_VALUES_EQUAL("07323a5f2b402425", Bin2hex("\007"
                                                             "2:_+@$%"));
        UNIT_ASSERT_VALUES_EQUAL("323a5f2b07402425", Bin2hex("2:_+"
                                                             "\007"
                                                             "@$%"));
        TString str = "2:_+ @$%";
        str[4] = '\0';
        UNIT_ASSERT_VALUES_EQUAL("323a5f2b00402425", Bin2hex(str));

        UNIT_ASSERT_VALUES_EQUAL("2:_+@$%", Hex2bin("323a5f2b402425"));
        UNIT_ASSERT_VALUES_EQUAL("2      ", Hex2bin("32202020202020"));
        UNIT_ASSERT_VALUES_EQUAL(nullStr, Hex2bin("00"));
        UNIT_ASSERT_VALUES_EQUAL("", Hex2bin(""));
        UNIT_ASSERT_VALUES_EQUAL("\0072:_+@$%", Hex2bin("07323a5f2b402425"));
        UNIT_ASSERT_VALUES_EQUAL("\0072:_+@$%", Hex2bin("7323a5f2b402425"));
        UNIT_ASSERT_VALUES_EQUAL("2:_+\x07@$%", Hex2bin("323a5f2b07402425"));
        UNIT_ASSERT_VALUES_EQUAL(str, Hex2bin("323a5f2b00402425"));
        UNIT_ASSERT_VALUES_EQUAL(str, Hex2bin("323a5f2bxx402425"));

        UNIT_ASSERT_VALUES_EQUAL("2:_+@$%", Hex2bin(Bin2hex("2:_+@$%")));
        UNIT_ASSERT_VALUES_EQUAL("2      ", Hex2bin(Bin2hex("2      ")));
        UNIT_ASSERT_VALUES_EQUAL(nullStr, Hex2bin(Bin2hex(nullStr)));
    }

    Y_UNIT_TEST(simpleBase32Test) {
        UNIT_ASSERT_VALUES_EQUAL("", BinToBase32(""));
        UNIT_ASSERT_VALUES_EQUAL("00", BinToBase32(Hex2bin("00")));
        UNIT_ASSERT_VALUES_EQUAL("zw", BinToBase32("\xff"));
        UNIT_ASSERT_VALUES_EQUAL("xyzg", BinToBase32("\xef\xbf"));
        UNIT_ASSERT_VALUES_EQUAL("d1jprv3f", BinToBase32("hello"));
        UNIT_ASSERT_VALUES_EQUAL("d1jprv3f5g", BinToBase32("hello,"));
        UNIT_ASSERT_VALUES_EQUAL("d1jprv3f5gg0", BinToBase32("hello, "));
        UNIT_ASSERT_VALUES_EQUAL("d1jprv3f5gg20", BinToBase32("hello,  "));

        UNIT_ASSERT_VALUES_EQUAL("", Base32ToBin(""));
        UNIT_ASSERT_VALUES_EQUAL("", Base32ToBin("abc+abca"));
        UNIT_ASSERT_VALUES_EQUAL("", Base32ToBin("abcdabc="));
        UNIT_ASSERT_VALUES_EQUAL("", Base32ToBin("abcdabcda"));
        UNIT_ASSERT_VALUES_EQUAL("", Base32ToBin("abcdabcdabc"));
        UNIT_ASSERT_VALUES_EQUAL("", Base32ToBin("abcdabcdabcabc"));
        UNIT_ASSERT_VALUES_EQUAL("", Base32ToBin("X"));
        UNIT_ASSERT_VALUES_EQUAL("", Base32ToBin("XYZ"));
        UNIT_ASSERT_VALUES_EQUAL("", Base32ToBin("no"));
        UNIT_ASSERT_VALUES_EQUAL("", Base32ToBin("it"));

        UNIT_ASSERT_VALUES_EQUAL("T", Base32ToBin("ah"));
        UNIT_ASSERT_VALUES_EQUAL("[\x95", Base32ToBin("bear"));
        UNIT_ASSERT_VALUES_EQUAL("[\x95\x86", Base32ToBin("Beard"));
        UNIT_ASSERT_VALUES_EQUAL("0044352d8defbf", Bin2hex(Base32ToBin("0123abcdxyzt")));
        UNIT_ASSERT_VALUES_EQUAL("hello", Base32ToBin("d1jprv3f"));
        UNIT_ASSERT_VALUES_EQUAL("hello", Base32ToBin("D1JPRV3F"));
        UNIT_ASSERT_VALUES_EQUAL("hello,", Base32ToBin("d1jPRV3f5g"));
        UNIT_ASSERT_VALUES_EQUAL("hello, ", Base32ToBin("d1jprv3f5GG0"));
        UNIT_ASSERT_VALUES_EQUAL("hello,  ", Base32ToBin("d1jprv3f5gg20"));
    }

    Y_UNIT_TEST(singleCharBase32Test) {
        UNIT_ASSERT_VALUES_EQUAL('0', BinToBase32(0));
        UNIT_ASSERT_VALUES_EQUAL('1', BinToBase32(1));
        UNIT_ASSERT_VALUES_EQUAL('a', BinToBase32(10));
        UNIT_ASSERT_VALUES_EQUAL('p', BinToBase32(22));
        UNIT_ASSERT_VALUES_EQUAL('z', BinToBase32(31));
        UNIT_ASSERT_VALUES_EQUAL(0xff, BinToBase32(32));
        UNIT_ASSERT_VALUES_EQUAL(0xff, BinToBase32(255));

        UNIT_ASSERT_VALUES_EQUAL(0, Base32ToBin('0'));
        UNIT_ASSERT_VALUES_EQUAL(1, Base32ToBin('1'));
        UNIT_ASSERT_VALUES_EQUAL(10, Base32ToBin('a'));
        UNIT_ASSERT_VALUES_EQUAL(20, Base32ToBin('m'));
        UNIT_ASSERT_VALUES_EQUAL(31, Base32ToBin('z'));
        UNIT_ASSERT_VALUES_EQUAL(10, Base32ToBin('A'));
        UNIT_ASSERT_VALUES_EQUAL(20, Base32ToBin('M'));
        UNIT_ASSERT_VALUES_EQUAL(31, Base32ToBin('Z'));
        UNIT_ASSERT_VALUES_EQUAL(0xff, Base32ToBin('\x0'));
        UNIT_ASSERT_VALUES_EQUAL(0xff, Base32ToBin('+'));
        UNIT_ASSERT_VALUES_EQUAL(0xff, Base32ToBin('='));
        UNIT_ASSERT_VALUES_EQUAL(0xff, Base32ToBin('i'));
        UNIT_ASSERT_VALUES_EQUAL(0xff, Base32ToBin('l'));
        UNIT_ASSERT_VALUES_EQUAL(0xff, Base32ToBin('o'));
        UNIT_ASSERT_VALUES_EQUAL(0xff, Base32ToBin('s'));
        UNIT_ASSERT_VALUES_EQUAL(0xff, Base32ToBin('\xff'));
    }

    Y_UNIT_TEST(fromToBytesTestLSB) {
        // toBytesLSB
        TString buf(16, 0);

        UNIT_ASSERT_EXCEPTION_CONTAINS(ToBytesLsb<i8>(0, buf, 16), yexception, "Buffer position out of range");
        UNIT_ASSERT_EXCEPTION_CONTAINS(ToBytesLsb<i16>(0, buf, 15), yexception, "Buffer position out of range");
        UNIT_ASSERT_EXCEPTION_CONTAINS(ToBytesLsb<i32>(0, buf, 13), yexception, "Buffer position out of range");
        UNIT_ASSERT_EXCEPTION_CONTAINS(ToBytesLsb<ui64>(0, buf, 9), yexception, "Buffer position out of range");

        ToBytesLsb<ui8>(0xcc, buf);
        UNIT_ASSERT_STRINGS_EQUAL("cc000000000000000000000000000000", Bin2hex(buf));

        ToBytesLsb<i8>(-1, buf, 2);
        UNIT_ASSERT_STRINGS_EQUAL("cc00ff00000000000000000000000000", Bin2hex(buf));

        ToBytesLsb<i16>(0xee, buf, 3);
        UNIT_ASSERT_STRINGS_EQUAL("cc00ffee000000000000000000000000", Bin2hex(buf));

        ToBytesLsb<ui32>(0xdeadbeaf, buf, 5);
        UNIT_ASSERT_STRINGS_EQUAL("cc00ffee00afbeadde00000000000000", Bin2hex(buf));

        ToBytesLsb<i32>(-2, buf, 11);
        UNIT_ASSERT_STRINGS_EQUAL("cc00ffee00afbeadde0000feffffff00", Bin2hex(buf));

        ToBytesLsb<ui64>(80, buf, 8);
        UNIT_ASSERT_STRINGS_EQUAL("cc00ffee00afbead5000000000000000", Bin2hex(buf));

        ToBytesLsb<i64>(-65536, buf, 3);
        UNIT_ASSERT_STRINGS_EQUAL("cc00ff0000ffffffffffff0000000000", Bin2hex(buf));

        // fromBytesLSB
        UNIT_ASSERT_EXCEPTION_CONTAINS(FromBytesLsb<i8>(Hex2bin("")), yexception, "Buffer position out of range");

        UNIT_ASSERT_VALUES_EQUAL(0xcc, FromBytesLsb<ui8>(Hex2bin("cc00ffee")));
        UNIT_ASSERT_VALUES_EQUAL(0x00, FromBytesLsb<ui8>(Hex2bin("cc00ffee"), 1));
        UNIT_ASSERT_VALUES_EQUAL(0xff, FromBytesLsb<ui8>(Hex2bin("cc00ffee"), 2));
        UNIT_ASSERT_VALUES_EQUAL(0xee, FromBytesLsb<ui8>(Hex2bin("cc00ffee"), 3));
        UNIT_ASSERT_EXCEPTION_CONTAINS(FromBytesLsb<ui8>(Hex2bin("cc00ffee"), 4), yexception, "Buffer position out of range");

        UNIT_ASSERT_VALUES_EQUAL(-0x34, FromBytesLsb<i8>(Hex2bin("cc00ffee")));
        UNIT_ASSERT_VALUES_EQUAL(-0x00, FromBytesLsb<i8>(Hex2bin("cc00ffee"), 1));
        UNIT_ASSERT_VALUES_EQUAL(-0x01, FromBytesLsb<i8>(Hex2bin("cc00ffee"), 2));
        UNIT_ASSERT_VALUES_EQUAL(-0x12, FromBytesLsb<i8>(Hex2bin("cc00ffee"), 3));
        UNIT_ASSERT_EXCEPTION_CONTAINS(FromBytesLsb<i8>(Hex2bin("cc00ffee"), 4), yexception, "Buffer position out of range");

        UNIT_ASSERT_VALUES_EQUAL(0xcc, FromBytesLsb<ui16>(Hex2bin("cc00ffee")));
        UNIT_ASSERT_VALUES_EQUAL(0xff00, FromBytesLsb<ui16>(Hex2bin("cc00ffee"), 1));
        UNIT_ASSERT_VALUES_EQUAL(0xeeff, FromBytesLsb<ui16>(Hex2bin("cc00ffee"), 2));
        UNIT_ASSERT_EXCEPTION_CONTAINS(FromBytesLsb<ui16>(Hex2bin("cc00ffee"), 3), yexception, "Buffer position out of range");
        UNIT_ASSERT_EXCEPTION_CONTAINS(FromBytesLsb<ui16>(Hex2bin("cc00ffee"), 4), yexception, "Buffer position out of range");

        UNIT_ASSERT_VALUES_EQUAL(0xddccbbaa, FromBytesLsb<ui32>(Hex2bin("00112233445566778899aabbccddeeff"), 10));
        UNIT_ASSERT_VALUES_EQUAL(-0x22334456, FromBytesLsb<i32>(Hex2bin("00112233445566778899aabbccddeeff"), 10));
        UNIT_ASSERT_VALUES_EQUAL(0xffeeddcc, FromBytesLsb<ui32>(Hex2bin("00112233445566778899aabbccddeeff"), 12));
        UNIT_ASSERT_EXCEPTION_CONTAINS(FromBytesLsb<ui32>(Hex2bin("00112233445566778899aabbccddeeff"), 13), yexception, "Buffer position out of range");

        UNIT_ASSERT_VALUES_EQUAL(0x8877665544332211, FromBytesLsb<ui64>(Hex2bin("00112233445566778899aabbccddeeff"), 1));
        UNIT_ASSERT_VALUES_EQUAL(-0x778899aabbccddef, FromBytesLsb<i64>(Hex2bin("00112233445566778899aabbccddeeff"), 1));
        UNIT_ASSERT_VALUES_EQUAL(0xffeeddccbbaa9988, FromBytesLsb<ui64>(Hex2bin("00112233445566778899aabbccddeeff"), 8));
        UNIT_ASSERT_EXCEPTION_CONTAINS(FromBytesLsb<ui64>(Hex2bin("00112233445566778899aabbccddeeff"), 9), yexception, "Buffer position out of range");
    }

    Y_UNIT_TEST(fromToBytesTestMSB) {
        // toBytesMSB
        TString buf(16, 0);

        UNIT_ASSERT_EXCEPTION_CONTAINS(ToBytesMsb<i8>(0, buf, 16), yexception, "Buffer position out of range");
        UNIT_ASSERT_EXCEPTION_CONTAINS(ToBytesMsb<i16>(0, buf, 15), yexception, "Buffer position out of range");
        UNIT_ASSERT_EXCEPTION_CONTAINS(ToBytesMsb<i32>(0, buf, 13), yexception, "Buffer position out of range");
        UNIT_ASSERT_EXCEPTION_CONTAINS(ToBytesMsb<ui64>(0, buf, 9), yexception, "Buffer position out of range");

        ToBytesMsb<ui8>(0xcc, buf);
        UNIT_ASSERT_STRINGS_EQUAL("cc000000000000000000000000000000", Bin2hex(buf));

        ToBytesMsb<i8>(-1, buf, 2);
        UNIT_ASSERT_STRINGS_EQUAL("cc00ff00000000000000000000000000", Bin2hex(buf));

        ToBytesMsb<i16>(0xee, buf, 3);
        UNIT_ASSERT_STRINGS_EQUAL("cc00ff00ee0000000000000000000000", Bin2hex(buf));

        ToBytesMsb<ui32>(0xdeadbeaf, buf, 5);
        UNIT_ASSERT_STRINGS_EQUAL("cc00ff00eedeadbeaf00000000000000", Bin2hex(buf));

        ToBytesMsb<i32>(-2, buf, 11);
        UNIT_ASSERT_STRINGS_EQUAL("cc00ff00eedeadbeaf0000fffffffe00", Bin2hex(buf));

        ToBytesMsb<ui64>(80, buf, 8);
        UNIT_ASSERT_STRINGS_EQUAL("cc00ff00eedeadbe0000000000000050", Bin2hex(buf));

        ToBytesMsb<i64>(-65536, buf, 3);
        UNIT_ASSERT_STRINGS_EQUAL("cc00ffffffffffffff00000000000050", Bin2hex(buf));

        // fromBytesMSB
        UNIT_ASSERT_EXCEPTION_CONTAINS(FromBytesMsb<i8>(Hex2bin("")), yexception, "Buffer position out of range");

        UNIT_ASSERT_VALUES_EQUAL(0xcc, FromBytesMsb<ui8>(Hex2bin("cc00ffee")));
        UNIT_ASSERT_VALUES_EQUAL(0x00, FromBytesMsb<ui8>(Hex2bin("cc00ffee"), 1));
        UNIT_ASSERT_VALUES_EQUAL(0xff, FromBytesMsb<ui8>(Hex2bin("cc00ffee"), 2));
        UNIT_ASSERT_VALUES_EQUAL(0xee, FromBytesMsb<ui8>(Hex2bin("cc00ffee"), 3));
        UNIT_ASSERT_EXCEPTION_CONTAINS(FromBytesMsb<ui8>(Hex2bin("cc00ffee"), 4), yexception, "Buffer position out of range");

        UNIT_ASSERT_VALUES_EQUAL(-0x34, FromBytesMsb<i8>(Hex2bin("cc00ffee")));
        UNIT_ASSERT_VALUES_EQUAL(-0x00, FromBytesMsb<i8>(Hex2bin("cc00ffee"), 1));
        UNIT_ASSERT_VALUES_EQUAL(-0x01, FromBytesMsb<i8>(Hex2bin("cc00ffee"), 2));
        UNIT_ASSERT_VALUES_EQUAL(-0x12, FromBytesMsb<i8>(Hex2bin("cc00ffee"), 3));
        UNIT_ASSERT_EXCEPTION_CONTAINS(FromBytesMsb<i8>(Hex2bin("cc00ffee"), 4), yexception, "Buffer position out of range");

        UNIT_ASSERT_VALUES_EQUAL(0xcc00, FromBytesMsb<ui16>(Hex2bin("cc00ffee")));
        UNIT_ASSERT_VALUES_EQUAL(0xff, FromBytesMsb<ui16>(Hex2bin("cc00ffee"), 1));
        UNIT_ASSERT_VALUES_EQUAL(0xffee, FromBytesMsb<ui16>(Hex2bin("cc00ffee"), 2));
        UNIT_ASSERT_EXCEPTION_CONTAINS(FromBytesMsb<ui16>(Hex2bin("cc00ffee"), 3), yexception, "Buffer position out of range");
        UNIT_ASSERT_EXCEPTION_CONTAINS(FromBytesMsb<ui16>(Hex2bin("cc00ffee"), 4), yexception, "Buffer position out of range");

        UNIT_ASSERT_VALUES_EQUAL(0xaabbccdd, FromBytesMsb<ui32>(Hex2bin("00112233445566778899aabbccddeeff"), 10));
        UNIT_ASSERT_VALUES_EQUAL(-0x55443323, FromBytesMsb<i32>(Hex2bin("00112233445566778899aabbccddeeff"), 10));
        UNIT_ASSERT_VALUES_EQUAL(0xccddeeff, FromBytesMsb<ui32>(Hex2bin("00112233445566778899aabbccddeeff"), 12));
        UNIT_ASSERT_EXCEPTION_CONTAINS(FromBytesMsb<ui32>(Hex2bin("00112233445566778899aabbccddeeff"), 13), yexception, "Buffer position out of range");

        UNIT_ASSERT_VALUES_EQUAL(0x1122334455667788, FromBytesMsb<ui64>(Hex2bin("00112233445566778899aabbccddeeff"), 1));
        UNIT_ASSERT_VALUES_EQUAL(-0xeeddccbbaa998878, FromBytesMsb<i64>(Hex2bin("00112233445566778899aabbccddeeff"), 1));
        UNIT_ASSERT_VALUES_EQUAL(0x8899aabbccddeeff, FromBytesMsb<ui64>(Hex2bin("00112233445566778899aabbccddeeff"), 8));
        UNIT_ASSERT_EXCEPTION_CONTAINS(FromBytesMsb<ui64>(Hex2bin("00112233445566778899aabbccddeeff"), 9), yexception, "Buffer position out of range");
    }
}
