#include <passport/infra/libs/cpp/utils/crypto/hash.h>
#include <passport/infra/libs/cpp/utils/crypto/rsa.h>
#include <passport/infra/libs/cpp/utils/crypto/sshkey.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <library/cpp/ssh/sign_key.h>
#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>

#include <util/stream/file.h>

#include <array>

using namespace NPassport::NUtils;

Y_UNIT_TEST_SUITE(PasspUtilsCrypto) {
    Y_UNIT_TEST(md5Test) {
        UNIT_ASSERT_VALUES_EQUAL("phnZdGWPPnSbLYiyFbrqRg==",
                                 BinToBase64(TCrypto::Md5("ololo")));
        UNIT_ASSERT_VALUES_EQUAL("RLkg1av3M4FA0xV8kXDWJA==",
                                 BinToBase64(TCrypto::Md5("alala")));
        UNIT_ASSERT_VALUES_EQUAL("yk+3ytRJ4HutY7IdB2vMyA==",
                                 BinToBase64(TCrypto::Md5("ololo alala")));
        UNIT_ASSERT_VALUES_EQUAL("giaIGW3lTmn4Kj0sH+m+FQ==",
                                 BinToBase64(TCrypto::Md5("This magic wonderful world!!")));
        UNIT_ASSERT_VALUES_EQUAL("1B2M2Y8AsgTpgAmY7PhCfg==",
                                 BinToBase64(TCrypto::Md5("")));
        UNIT_ASSERT_VALUES_EQUAL("oc0dH8ZJEGjZEAcoPthEiQ==",
                                 BinToBase64(TCrypto::Md5(Hex2bin("0123456789abcdef"))));
        UNIT_ASSERT_VALUES_EQUAL("WOLYCeDeNMYvU6PEpPxE0Q==",
                                 BinToBase64(TCrypto::Md5(Hex2bin("3189a08a7e8fd90f88a88bc8e0123456789abcdef"))));
        UNIT_ASSERT_VALUES_EQUAL("k7iFrf4NoInN9jSQT9WfcQ==",
                                 BinToBase64(TCrypto::Md5(Hex2bin("00"))));

        TString str = "This magic wonderful world!!";
        str[5] = '\0';
        str.back() = '\0';
        str[0] = '\0';
        UNIT_ASSERT_VALUES_EQUAL("hWgXeBL6mmuef02+P0m5Bg==",
                                 BinToBase64(TCrypto::Md5(str)));
    }

    Y_UNIT_TEST(sha256Test) {
        UNIT_ASSERT_VALUES_EQUAL("DLKsi89gA3K1c7+fgH3j6ws87aUcCiBF1pAkEsU0UfI=",
                                 BinToBase64(TCrypto::Sha256("ololo")));
        UNIT_ASSERT_VALUES_EQUAL("NUEjccywB+bU36IDArF7ahj2q+0oy5o+XWLqn8620rE=",
                                 BinToBase64(TCrypto::Sha256("alala")));
        UNIT_ASSERT_VALUES_EQUAL("qcg82W+cKs58I0YvkP4o7j3ZD4bt9BzrFTp8Ft2sgPc=",
                                 BinToBase64(TCrypto::Sha256("ololo alala")));
        UNIT_ASSERT_VALUES_EQUAL("ppMLi27CMgQP1nNCnnPClrkIkDnE366ClXY19fwmjV4=",
                                 BinToBase64(TCrypto::Sha256("This magic wonderful world!!")));
        UNIT_ASSERT_VALUES_EQUAL("47DEQpj8HBSa+/TImW+5JCeuQeRkm5NMpJWZG3hSuFU=",
                                 BinToBase64(TCrypto::Sha256("")));
        UNIT_ASSERT_VALUES_EQUAL("VcU/XUkCl5AM76gl0Mjo6VMu6KEYq+fYVwdizTi+mBg=",
                                 BinToBase64(TCrypto::Sha256(Hex2bin("0123456789abcdef"))));
        UNIT_ASSERT_VALUES_EQUAL("m591+spAlPa58P0KoAkM1Gtn05S9nc0KNr6lM9/UqX8=",
                                 BinToBase64(TCrypto::Sha256(Hex2bin("3189a08a7e8fd90f88a88bc8e0123456789abcdef"))));
        UNIT_ASSERT_VALUES_EQUAL("bjQLnP+zepicpUTmu3gKLHiQHT+zNzh2hRGjBhevoB0=",
                                 BinToBase64(TCrypto::Sha256(Hex2bin("00"))));

        TString str = "This magic wonderful world!!";
        str[5] = '\0';
        str.back() = '\0';
        str[0] = '\0';
        UNIT_ASSERT_VALUES_EQUAL("wd72RWHmcT6FlXJgfO3Olcn71Cy7URLPJJdl+aHBjh4=",
                                 BinToBase64(TCrypto::Sha256(str)));
    }

    Y_UNIT_TEST(hmacSha256Test) {
        UNIT_ASSERT_VALUES_EQUAL("94u82eFuGEgLuCnrdoEel7mm7ktsbh7xc5CcvdR3hro=",
                                 BinToBase64(TCrypto::HmacSha256("123", "ololo")));
        UNIT_ASSERT_VALUES_EQUAL("vZDoQ+CSO3hXaYEnR8suQUY8+InYsdLP3QCsrzhawPw=",
                                 BinToBase64(TCrypto::HmacSha256("0123456789abcdef", "alala")));
        UNIT_ASSERT_VALUES_EQUAL("NdYg/Vp/qzFAdcr9hAK8GPwvA0WeohopNX+AC+kpakk=",
                                 BinToBase64(TCrypto::HmacSha256("0123456789abcdef0123456789abcdef", "ololo alala")));
        UNIT_ASSERT_VALUES_EQUAL("5+/TD0LV7R7oI8VDlz9Kn49LUjzSHA0khR1tzCv/GEU=",
                                 BinToBase64(TCrypto::HmacSha256("magic key for", "This magic wonderful world!!")));
        UNIT_ASSERT_VALUES_EQUAL("dt78puJ2Yq5xruCcsFKRB35bX/mxoiZJHH0to6HKRUA=",
                                 BinToBase64(TCrypto::HmacSha256("magic key for", "")));
        UNIT_ASSERT_VALUES_EQUAL("o3mROD0QQ6T7WJnD64FqVBg+DFHtD2JDbrgCgEM+4sg=",
                                 BinToBase64(TCrypto::HmacSha256("", "magic key for")));
        UNIT_ASSERT_VALUES_EQUAL("thNnmggU2ex3L5XXeMNfxf8Wl8STcVZTxscSFEKSxa0=",
                                 BinToBase64(TCrypto::HmacSha256("", "")));
        UNIT_ASSERT_VALUES_EQUAL("iIWn6OUKcH5uQN1BPUWty3mvGTE14tyzLk6l/JtpGQ8=",
                                 BinToBase64(TCrypto::HmacSha256(Hex2bin("0123456789abcdef"),
                                                                 Hex2bin("3189a08a7e8fd90f88a88bc8e0123456789abcdef"))));
        UNIT_ASSERT_VALUES_EQUAL("wr/MaeM/bkniITAt11RtasRDsolyyk3EDMVmIsVVKSg=",
                                 BinToBase64(TCrypto::HmacSha256(Hex2bin("3189a08a7e8fd90f88a88bc8e0123456789abcdef"),
                                                                 Hex2bin("0123456789abcdef"))));
        UNIT_ASSERT_VALUES_EQUAL("Aj7J8adx6jSGVLTJOody7pYa9f3IoD5kco1O8OeIy4M=",
                                 BinToBase64(TCrypto::HmacSha256(Hex2bin("0123456789abcdef"),
                                                                 Hex2bin("00"))));
        UNIT_ASSERT_VALUES_EQUAL("AE+D1TtRyJLmvktp4tJhghLPhD/w5eC5Xru58Vwk0Ys=",
                                 BinToBase64(TCrypto::HmacSha256(Hex2bin("00"),
                                                                 Hex2bin("0123456789abcdef"))));

        TString str = "This magic wonderful world!!";
        str[5] = '\0';
        str.back() = '\0';
        str[0] = '\0';
        UNIT_ASSERT_VALUES_EQUAL("MrQK74bZIIGCavXl/b/EVx2HN4Odu8HqEvfmaGF7LU4=",
                                 BinToBase64(TCrypto::HmacSha256("magic key for", str)));

        TString key = "0123456789abcdef0123456789abcdef";
        str[8] = '\0';
        str.back() = '\0';
        str[1] = '\0';
        UNIT_ASSERT_VALUES_EQUAL("W0pSDn1nqCpIaalqjrtp2W83xTBj9vW1tmLDuNa3Y9c=",
                                 BinToBase64(TCrypto::HmacSha256(key, str)));
    }

    Y_UNIT_TEST(hmacSha1Test) {
        UNIT_ASSERT_VALUES_EQUAL("SFLDyy85ShvPelOMY25klsIdKT4=",
                                 BinToBase64(TCrypto::HmacSha1("ololo", "123")));
        UNIT_ASSERT_VALUES_EQUAL("EzF5MtFxyWEL6J3Vundb38/1urY=",
                                 BinToBase64(TCrypto::HmacSha1("alala", "0123456789abcdef")));
        UNIT_ASSERT_VALUES_EQUAL("ln8I40/nP7eYByKxxgvE9YNYn2M=",
                                 BinToBase64(TCrypto::HmacSha1("ololo alala", "0123456789abcdef0123456789abcdef")));
        UNIT_ASSERT_VALUES_EQUAL("6aoWr/WghkmaYiWj8nSl2g70XAE=",
                                 BinToBase64(TCrypto::HmacSha1("This magic wonderful world!!", "magic key for")));
        UNIT_ASSERT_VALUES_EQUAL("ibkDnjid5oezCztduxf53OWK+S4=",
                                 BinToBase64(TCrypto::HmacSha1("This magic wonderful world!!", "")));
        UNIT_ASSERT_VALUES_EQUAL("tl8kSvLpk/zsKHnS2FovuFh+FqM=",
                                 BinToBase64(TCrypto::HmacSha1("", "magic key for")));
        UNIT_ASSERT_VALUES_EQUAL("+9sdGxiqbAgyS31ktx+3Y3BpDh0=",
                                 BinToBase64(TCrypto::HmacSha1("", "")));
        UNIT_ASSERT_VALUES_EQUAL("FBHYcIVobw86Q4HCGV2lkj5dch0=",
                                 BinToBase64(TCrypto::HmacSha1(Hex2bin("0123456789abcdef"),
                                                               Hex2bin("3189a08a7e8fd90f88a88bc8e0123456789abcdef"))));
        UNIT_ASSERT_VALUES_EQUAL("M2+FqYuNKpT45gVU/17u/X2gomw=",
                                 BinToBase64(TCrypto::HmacSha1(Hex2bin("3189a08a7e8fd90f88a88bc8e0123456789abcdef"),
                                                               Hex2bin("0123456789abcdef"))));
        UNIT_ASSERT_VALUES_EQUAL("//NwhDdNn2pA0Miw2k6zKAfws94=",
                                 BinToBase64(TCrypto::HmacSha1(Hex2bin("0123456789abcdef"),
                                                               Hex2bin("00"))));
        UNIT_ASSERT_VALUES_EQUAL("QMMKocmgKKuZXhkE0XLvuIxe5Ag=",
                                 BinToBase64(TCrypto::HmacSha1(Hex2bin("00"),
                                                               Hex2bin("0123456789abcdef"))));

        TString str = "This magic wonderful world!!";
        str[5] = '\0';
        str.back() = '\0';
        str[0] = '\0';
        UNIT_ASSERT_VALUES_EQUAL("2Lg5BhV2+2qIn4qgh834VxB3Nqk=",
                                 BinToBase64(TCrypto::HmacSha1("magic key for", str)));

        TString key = "0123456789abcdef0123456789abcdef";
        str[8] = '\0';
        str.back() = '\0';
        str[1] = '\0';
        UNIT_ASSERT_VALUES_EQUAL("Fg2Hrf71z2SSyInTps9P8hsx470=",
                                 BinToBase64(TCrypto::HmacSha1(key, str)));
    }

    Y_UNIT_TEST(GCMTest) {
        TString err_msg;
        TString key = Hex2bin("0123456789abcdef0123456789abcdef0123456789abcdef0123456789abcdef");
        TString text = "Original ;.&\\ \a\b\t\n message \xfe\xff";
        text[18] = '\0';
        text.push_back('\0');
        TString out;
        TCrypto::TCiphertext c;
        UNIT_ASSERT_VALUES_EQUAL(30, text.size());

        // check we don't crash with no err_msg
        UNIT_ASSERT(TCrypto::EncryptGcm(key, text, c));
        UNIT_ASSERT(TCrypto::DecryptGcm(key, c, out));
        UNIT_ASSERT_VALUES_EQUAL(out, text);

        *(c.Text.begin() + 2) ^= 'a';
        UNIT_ASSERT(!TCrypto::DecryptGcm(key, c, out));

        c.Iv = Hex2bin("050ed6c1b72b2a5330525587");
        c.Text = Hex2bin("2351600e6c725ca2dd51c75733bd793b85dc1da9c657d6661cf78038525b");
        c.Tag = Hex2bin("4285ce63d5dac134008f183f96553c99");
        UNIT_ASSERT(TCrypto::DecryptGcm(key, c, out));
        UNIT_ASSERT_VALUES_EQUAL(out, text);

        // tag with some garbage
        c.Tag.append("abcdef");
        UNIT_ASSERT(!TCrypto::DecryptGcm(key, c, out));

        // shortened tag
        c.Tag.resize(8);
        UNIT_ASSERT(!TCrypto::DecryptGcm(key, c, out));

        // check bad key size
        UNIT_ASSERT(!TCrypto::EncryptGcm(TString(), TString(), c, &err_msg));
        UNIT_ASSERT_VALUES_EQUAL("too short key given, 256 bit needed", err_msg);
        UNIT_ASSERT(!TCrypto::DecryptGcm(TString(), c, out, &err_msg));
        UNIT_ASSERT_VALUES_EQUAL("too short key given, 256 bit needed", err_msg);

        // check encrypt-decrypt with no aadata
        UNIT_ASSERT(TCrypto::EncryptGcm(key, text, c, &err_msg));
        UNIT_ASSERT_VALUES_EQUAL("OK", err_msg);
        UNIT_ASSERT(TCrypto::DecryptGcm(key, c, out, &err_msg));
        UNIT_ASSERT_VALUES_EQUAL("OK", err_msg);
        UNIT_ASSERT_VALUES_EQUAL(out, text);

        // checck that modified text does not decrypt
        *(c.Text.begin() + 2) ^= 'a';
        UNIT_ASSERT(!TCrypto::DecryptGcm(key, c, out, &err_msg));
        UNIT_ASSERT_VALUES_EQUAL("unable to perform EVP_DecryptFinal_ex()", err_msg);

        // check that truncated IV is not accepted
        c.Iv.resize(8);
        UNIT_ASSERT(!TCrypto::DecryptGcm(key, c, out, &err_msg));
        UNIT_ASSERT_VALUES_EQUAL("too short IV given, at least 96 bit needed", err_msg);

        c.Iv = Hex2bin("050ed6c1b72b2a5330525587");
        c.Text = Hex2bin("2351600e6c725ca2dd51c75733bd793b85dc1da9c657d6661cf78038525b");
        c.Tag = Hex2bin("4285ce63d5dac134008f183f96553c99");
        UNIT_ASSERT(TCrypto::DecryptGcm(key, c, out, &err_msg));
        UNIT_ASSERT_VALUES_EQUAL("OK", err_msg);
        UNIT_ASSERT_VALUES_EQUAL(out, text);

        c.Tag.append("abcdef");
        UNIT_ASSERT(!TCrypto::DecryptGcm(key, c, out, &err_msg));
        UNIT_ASSERT_VALUES_EQUAL("unable to perform EVP_CIPHER_CTX_ctrl(gcm_set_tag)", err_msg);

        // check encrypt-decrypt with empty aadata
        UNIT_ASSERT(TCrypto::EncryptGcm(key, text, c, TString(), &err_msg));
        UNIT_ASSERT_VALUES_EQUAL("OK", err_msg);
        UNIT_ASSERT(TCrypto::DecryptGcm(key, c, out, TString(), &err_msg));
        UNIT_ASSERT_VALUES_EQUAL("OK", err_msg);
        UNIT_ASSERT_VALUES_EQUAL(out, text);

        // corrupted text
        *(c.Text.begin() + 2) ^= 'a';
        UNIT_ASSERT(!TCrypto::DecryptGcm(key, c, out, TString(), &err_msg));
        UNIT_ASSERT_VALUES_EQUAL("unable to perform EVP_DecryptFinal_ex()", err_msg);
        *(c.Text.begin() + 2) ^= 'a';

        // shortened tag
        c.Tag.resize(12);
        UNIT_ASSERT(!TCrypto::DecryptGcm(key, c, out, TString(), &err_msg));
        UNIT_ASSERT_VALUES_EQUAL("too short tag given, at least 128 bit needed", err_msg);

        c.Iv = Hex2bin("050ed6c1b72b2a5330525587");
        c.Text = Hex2bin("2351600e6c725ca2dd51c75733bd793b85dc1da9c657d6661cf78038525b");
        c.Tag = Hex2bin("4285ce63d5dac134008f183f96553c99");
        UNIT_ASSERT(TCrypto::DecryptGcm(key, c, out, TString(), &err_msg));
        UNIT_ASSERT_VALUES_EQUAL("OK", err_msg);
        UNIT_ASSERT_VALUES_EQUAL(out, text);

        c.Tag.append("abcdef");
        UNIT_ASSERT(!TCrypto::DecryptGcm(key, c, out, TString(), &err_msg));
        UNIT_ASSERT_VALUES_EQUAL("unable to perform EVP_CIPHER_CTX_ctrl(gcm_set_tag)", err_msg);

        UNIT_ASSERT(TCrypto::EncryptGcm(key, text, c, &err_msg));
        UNIT_ASSERT_VALUES_EQUAL("OK", err_msg);
        UNIT_ASSERT(TCrypto::DecryptGcm(key, c, out, &err_msg));
        UNIT_ASSERT_VALUES_EQUAL("OK", err_msg);
        UNIT_ASSERT_VALUES_EQUAL(out, text);

        *(c.Text.begin() + 2) ^= 'a';
        UNIT_ASSERT(!TCrypto::DecryptGcm(key, c, out, &err_msg));
        UNIT_ASSERT_VALUES_EQUAL("unable to perform EVP_DecryptFinal_ex()", err_msg);

        c.Iv = Hex2bin("050ed6c1b72b2a5330525587");
        c.Text = Hex2bin("2351600e6c725ca2dd51c75733bd793b85dc1da9c657d6661cf78038525b");
        c.Tag = Hex2bin("4285ce63d5dac134008f183f96553c99");
        UNIT_ASSERT(TCrypto::DecryptGcm(key, c, out, &err_msg));
        UNIT_ASSERT_VALUES_EQUAL("OK", err_msg);
        UNIT_ASSERT_VALUES_EQUAL(out, text);

        c.Tag.append("abcdef");
        UNIT_ASSERT(!TCrypto::DecryptGcm(key, c, out, &err_msg));
        UNIT_ASSERT_VALUES_EQUAL("unable to perform EVP_CIPHER_CTX_ctrl(gcm_set_tag)", err_msg);

        // check encrypt-decrypt with aadata
        TString aa_data("Additional data for\a\b authentification");
        aa_data.push_back('\0');
        aa_data.push_back('\0');
        aa_data[10] = '\0';
        aa_data[20] = '\0';

        UNIT_ASSERT_VALUES_EQUAL(40, aa_data.size());

        UNIT_ASSERT(TCrypto::EncryptGcm(key, text, c, aa_data, &err_msg));
        UNIT_ASSERT_VALUES_EQUAL("OK", err_msg);
        UNIT_ASSERT(TCrypto::DecryptGcm(key, c, out, aa_data, &err_msg));
        UNIT_ASSERT_VALUES_EQUAL("OK", err_msg);
        UNIT_ASSERT_VALUES_EQUAL(out, text);

        UNIT_ASSERT(!TCrypto::DecryptGcm(key, c, out, aa_data.substr(0, 2), &err_msg));
        UNIT_ASSERT_VALUES_EQUAL("unable to perform EVP_DecryptFinal_ex()", err_msg);

        *(c.Text.begin() + 2) ^= 'a';
        UNIT_ASSERT(!TCrypto::DecryptGcm(key, c, out, aa_data, &err_msg));
        UNIT_ASSERT_VALUES_EQUAL("unable to perform EVP_DecryptFinal_ex()", err_msg);

        c.Iv = Hex2bin("875c080c41034ff630b67c12");
        c.Text = Hex2bin("a27625531b95e0d43668fb74042ce288d5d3d4147b27db0e8b03b9787554");
        c.Tag = Hex2bin("6d77c115d3d7646499ef359fbfe47a2a");
        UNIT_ASSERT(TCrypto::DecryptGcm(key, c, out, aa_data, &err_msg));
        UNIT_ASSERT_VALUES_EQUAL("OK", err_msg);
        UNIT_ASSERT_VALUES_EQUAL(out, text);

        UNIT_ASSERT(!TCrypto::DecryptGcm(key, c, out, aa_data.substr(0, aa_data.size() - 2), &err_msg));
        UNIT_ASSERT_VALUES_EQUAL("unable to perform EVP_DecryptFinal_ex()", err_msg);

        c.Tag.append("abcdef");
        UNIT_ASSERT(!TCrypto::DecryptGcm(key, c, out, aa_data, &err_msg));
        UNIT_ASSERT_VALUES_EQUAL("unable to perform EVP_CIPHER_CTX_ctrl(gcm_set_tag)", err_msg);

        UNIT_ASSERT(TCrypto::EncryptGcm(key, text, c, aa_data, &err_msg));
        UNIT_ASSERT_VALUES_EQUAL("OK", err_msg);
        UNIT_ASSERT(TCrypto::DecryptGcm(key, c, out, aa_data, &err_msg));
        UNIT_ASSERT_VALUES_EQUAL("OK", err_msg);
        UNIT_ASSERT_VALUES_EQUAL(out, text);

        UNIT_ASSERT(!TCrypto::DecryptGcm(key, c, out, TStringBuf(aa_data).Trunc(2), &err_msg));
        UNIT_ASSERT_VALUES_EQUAL("unable to perform EVP_DecryptFinal_ex()", err_msg);

        *(c.Text.begin() + 2) ^= 'a';
        UNIT_ASSERT(!TCrypto::DecryptGcm(key, c, out, TStringBuf(aa_data).Trunc(30), &err_msg));
        UNIT_ASSERT_VALUES_EQUAL("unable to perform EVP_DecryptFinal_ex()", err_msg);

        c.Iv = Hex2bin("875c080c41034ff630b67c12");
        c.Text = Hex2bin("a27625531b95e0d43668fb74042ce288d5d3d4147b27db0e8b03b9787554");
        c.Tag = Hex2bin("6d77c115d3d7646499ef359fbfe47a2a");
        UNIT_ASSERT(TCrypto::DecryptGcm(key, c, out, aa_data, &err_msg));
        UNIT_ASSERT_VALUES_EQUAL("OK", err_msg);
        UNIT_ASSERT_VALUES_EQUAL(out, text);

        UNIT_ASSERT(!TCrypto::DecryptGcm(key, c, out, TStringBuf(aa_data).Trunc(38), &err_msg));
        UNIT_ASSERT_VALUES_EQUAL("unable to perform EVP_DecryptFinal_ex()", err_msg);

        c.Tag.append("abcdef");
        UNIT_ASSERT(!TCrypto::DecryptGcm(key, c, out, aa_data, &err_msg));
        UNIT_ASSERT_VALUES_EQUAL("unable to perform EVP_CIPHER_CTX_ctrl(gcm_set_tag)", err_msg);

        // check with different key and message (paranoic mode: on)
        key = Hex2bin("012345678900fffe00010d0a0b0c0dff012345678900fffe00010d0a0b0c0dff");
        text = "Lorem ipsum dolor sit amet, consectetur adipiscing elit, "
               "sed do eiusmod tempor incididunt ut labore et dolore magna aliqua. "
               "Ut enim ad minim veniam, quis nostrud exercitation ullamco laboris "
               "nisi ut aliquip ex ea commodo consequat. Duis aute irure dolor "
               "in reprehenderit in voluptate velit esse cillum dolore eu fugiat nulla pariatur. "
               "Excepteur sint occaecat cupidatat non proident, "
               "sunt in culpa qui officia deserunt mollit anim id est laborum.";

        UNIT_ASSERT(TCrypto::EncryptGcm(key, text, c));
        UNIT_ASSERT(TCrypto::DecryptGcm(key, c, out));
        UNIT_ASSERT_VALUES_EQUAL(out, text);

        c.Iv = Hex2bin("49da51c1d8f02cd93822dfa5");
        c.Text = Hex2bin("1f9dc07012ee017eecbc58e2936313485f54fcb5bfa1e53c9749bef2b90f70c787437cc398e32c23c1"
                         "a1da199aa608b4c7244897ca51c789cdae8488290d2df5cc2d916fe498f18dc1fc740bdb64afab2ef57994d137f7d6054ab503"
                         "2bd122ecd9a40068280c762ed7124b7669efbbdd4f1d009c5861fd437245b302fcf7918534db2e5efb53b89cd3cda61a2c5aff"
                         "bed8a447f71b041925b03c84269eaae6d33c207d0600ba161e0d0c981200af38beee653f572d65f8c20d3729e805c65b98150b"
                         "7ccdd577486e9549dc6c3a4bf0bc447eee970c0f00ec6c4676e2a0caf58ec6742e7c6601df348722c9f501818e68874ce512a2"
                         "d8b5f00e6d765c8ff48aee900acccc82bb1c7b262f749e65453155d1ab1b3158800131c1eb9356419ade3ffc4d527b2fd58dbc"
                         "9fc862d849bb4a2a8800b75b78d6139c4b31c0ab16f206958a9d4db6778a5f35b46bf830bc0a81c0943b2d30d526032094ae8f"
                         "66220e6801e7f5464f1b3f569f4b142af6c57845b85acbc51e99f65d4492841bed18624c80d70bb752839146358393dd65a874"
                         "84c78804448e8cddd3f0c17e6bf361561c96946cb029df18e43aace2956ca9fae6a95cf44d70eb7447ed16466af1f6");
        c.Tag = Hex2bin("e972972ecd235ce77594b0ce5793e128");
        UNIT_ASSERT(TCrypto::DecryptGcm(key, c, out));
        UNIT_ASSERT_VALUES_EQUAL(out, text);

        *(c.Tag.begin() + 2) ^= 1;
        UNIT_ASSERT(!TCrypto::DecryptGcm(key, c, out));
    }

    Y_UNIT_TEST(AES_CBC_Test) {
        TString out;
        TString err;
        UNIT_ASSERT(!TCrypto::DecryptCbc(
            "qqq",
            "aaa",
            "zzz",
            out,
            &err));
        UNIT_ASSERT_VALUES_EQUAL(err, "key required to be 32 bytes");

        UNIT_ASSERT(!TCrypto::DecryptCbc(
            "qqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqq",
            "aaa",
            "zzz",
            out,
            &err));
        UNIT_ASSERT_VALUES_EQUAL(err, "iv required to be 16 bytes");

        UNIT_ASSERT(!TCrypto::DecryptCbc(
            "qqqqqqqqqqqqqqqqqqqqqqqqqqqqqqqq",
            "aaaaaaaaaaaaaaaa",
            "zzz",
            out,
            &err));
        UNIT_ASSERT_VALUES_EQUAL(err, "unable to perform EVP_DecryptFinal_ex()");

        UNIT_ASSERT(TCrypto::DecryptCbc(
            Hex2bin("5d8a36c1533b1bc9b964745b39542b48a5d99f4cfba50f1c7b2d546646798d1a"),
            Hex2bin("8b33490e79b103b9cb60919ea0dd223e"),
            Hex2bin("b90a148a8c494905a890d77249159a0dc9f39d30a68fe8ff7c7155a16372e468de52befef886e595ca0bd1976bbd74b3"),
            out,
            &err));
        UNIT_ASSERT_VALUES_EQUAL(out, R"({"1": "dGVzdGluZyAweERFQURCRUVG"})");
    }

    Y_UNIT_TEST(sshkeyParse) {
        TSshPublicKey key1("ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAABAQC5WKP9ik5EuemL+HF2IXkZlg09cLsMA0hXALwnpCrq07p8tZSpz8VFBp/u9ZuZa46VwJN3u/0F60WGNW/ROJ22uZ1orasD8vIRtTwNffswNqgp4K4cU08G3C/C3GEJZr8GvkRejJiOnsICu7l9kQ8Ufrlqz5PH3qkTw8SpdWnOpw7AwWSQCs+nE8SZ2lMMhanZxLQ9LL0qrh9PRvvwefNwaRStMaIimr+q3Afgj23fqk5ewkGUX7yb5xuq/ok984K6M3pO3X+4oCyEFRl099ik5XCh4Rz1YgsT3Zc245siYwNktSf4cOyrjt5hO8K8jNrR5n7bxxu+SzwywU2+VPbN");
        TSshPublicKey key2("ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAABAQDJyaQn/c7FldMrq+sFxu+9PaN9wJl2SIbF6Bl8gmU3eiD1l1AuC21ag8cSBSZuVC3X5kDuIZ18XS5TE8d2nm3gC4ZH7hUeE1H6p26U5myvo+P+mrsxSSCuXpvsnt49PwTly4+5lh2pHMrYGU5ESl6rSZexgwV14+yVqzk2c4jAMZ0DEH+GKPQ6SN9A3EbxYdTuCqBiS9zQoPkfh3/Tj9XVrBEVEg1uUtM7O0SmNlomRuS5ZQsBrokkWU5VIHuwIrnVmYXr0rUbmZ6DAkYujTEgHrCvqWDuZgP8KeABvToAzbUidzx+af1Z1nGGr1vPrMSXG2ti+aS4t7dPaZlcO6Db disruptor@disruptor");
        TSshPublicKey key3("ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAABAQDJyaQn\n"
                           "/c7FldMrq+sFxu+9PaN9wJl2SIbF6Bl8gmU3eiD1l1Au\n"
                           "C21ag8cSBSZuVC3X5kDuIZ18XS5TE8d2nm3gC4ZH7hUe\n"
                           "E1H6p26U5myvo+P+mrsxSSCuXpvsnt49PwTly4+5lh2p\n"
                           "HMrYGU5ESl6rSZexgwV14+yVqzk2c4jAMZ0DEH+GKPQ6\n"
                           "SN9A3EbxYdTuCqBiS9zQoPkfh3/Tj9XVrBEVEg1uUtM7\n"
                           "O0SmNlomRuS5ZQsBrokkWU5VIHuwIrnVmYXr0rUbmZ6D\n"
                           "AkYujTEgHrCvqWDuZgP8KeABvToAzbUidzx+af1Z1nGG\n"
                           "r1vPrMSXG2ti+aS4t7dPaZlcO6Db disruptor@disruptor");
        UNIT_ASSERT_EXCEPTION_CONTAINS(TSshPublicKey("ssh-rsa"), TSshPublicKey::TMalformedException, "SshPublicKey ill-formed: tokens: ssh-rsa");
        UNIT_ASSERT_EXCEPTION_CONTAINS(TSshPublicKey(""), TSshPublicKey::TMalformedException, "SshPublicKey ill-formed: tokens: ");
        UNIT_ASSERT_EXCEPTION_CONTAINS(TSshPublicKey("ssh-dsa AAAAB3NzaC1yc2EAAAADAQABAAABAQDJyaQn/c7FldMrq+sFxu+9PaN9wJl2SIbF6Bl8gmU3eiD1l1AuC21ag8cSBSZuVC3X5kDuIZ18XS5TE8d2nm3gC4ZH7hUeE1H6p26U5myvo+P+mrsxSSCuXpvsnt49PwTly4+5lh2pHMrYGU5ESl6rSZexgwV14+yVqzk2c4jAMZ0DEH+GKPQ6SN9A3EbxYdTuCqBiS9zQoPkfh3/Tj9XVrBEVEg1uUtM7O0SmNlomRuS5ZQsBrokkWU5VIHuwIrnVmYXr0rUbmZ6DAkYujTEgHrCvqWDuZgP8KeABvToAzbUidzx+af1Z1nGGr1vPrMSXG2ti+aS4t7dPaZlcO6Db disruptor@disruptor"),
                                       TSshPublicKey::TUnsupportedException,
                                       "");
        UNIT_ASSERT_EXCEPTION_CONTAINS(TSshPublicKey("ssh-rsa AAB3NzaC1yc2EAAAADAQABAAABAQDJyaQn/c7FldMrq+sFxu+9PaN9wJl2SIbF6Bl8gmU3eiD1l1AuC21ag8cSBSZuVC3X5kDuIZ18XS5TE8d2nm3gC4ZH7hUeE1H6p26U5myvo+P+mrsxSSCuXpvsnt49PwTly4+5lh2pHMrYGU5ESl6rSZexgwV14+yVqzk2c4jAMZ0DEH+GKPQ6SN9A3EbxYdTuCqBiS9zQoPkfh3/Tj9XVrBEVEg1uUtM7O0SmNlomRuS5ZQsBrokkWU5VIHuwIrnVmYXr0rUbmZ6DAkYujTEgHrCvqWDuZgP8KeABvToAzbUidzx+af1Z1nGGr1vPrMSXG2ti+aS4t7dPaZlcO6Db disruptor@disruptor"),
                                       TSshPublicKey::TMalformedException,
                                       "SshPublicKey ill-formed: encoded prefix len doesn't match to open prefix: ssh-rsa AAB3NzaC1yc2EAAAADAQABAAABAQDJyaQn/c7FldMrq+sFxu+9PaN9wJl2SIbF6Bl8gmU3eiD1l1AuC21ag8cSBSZuVC3X5kDuIZ18XS5TE8d2nm3gC4ZH7hUeE1H6p26U5myvo+P+mrsxSSCuXpvsnt49PwTly4+5lh2pHMrYGU5ESl6rSZexgwV14+yVqzk2c4jAMZ0DEH+GKPQ6SN9A3EbxYdTuCqBiS9zQoPkfh3/Tj9XVrBEVEg1uUtM7O0SmNlomRuS5ZQsBrokkWU5VIHuwIrnVmYXr0rUbmZ6DAkYujTEgHrCvqWDuZgP8KeABvToAzbUidzx+af1Z1nGGr1vPrMSXG2ti+aS4t7dPaZlcO6Db disruptor@disruptor");
        UNIT_ASSERT_EXCEPTION_CONTAINS(TSshPublicKey("ssh-rsa AAAAB3NzaC1yc2AAADAQABAAABAQDJyaQn/c7FldMrq+sFxu+9PaN9wJl2SIbF6Bl8gmU3eiD1l1AuC21ag8cSBSZuVC3X5kDuIZ18XS5TE8d2nm3gC4ZH7hUeE1H6p26U5myvo+P+mrsxSSCuXpvsnt49PwTly4+5lh2pHMrYGU5ESl6rSZexgwV14+yVqzk2c4jAMZ0DEH+GKPQ6SN9A3EbxYdTuCqBiS9zQoPkfh3/Tj9XVrBEVEg1uUtM7O0SmNlomRuS5ZQsBrokkWU5VIHuwIrnVmYXr0rUbmZ6DAkYujTEgHrCvqWDuZgP8KeABvToAzbUidzx+af1Z1nGGr1vPrMSXG2ti+aS4t7dPaZlcO6Db disruptor@disruptor"),
                                       TSshPublicKey::TMalformedException,
                                       "SshPublicKey ill-formed: encoded prefix doesn't match to open prefix");
    }

    TString PrivateKeyName() {
        return ArcadiaSourceRoot() + "/passport/infra/libs/cpp/utils/ut/test_key";
    }

    TString PublicKeyBody() {
        TFileInput ifs(ArcadiaSourceRoot() + "/passport/infra/libs/cpp/utils/ut/test_key.pub");
        return ifs.ReadLine();
    }

    Y_UNIT_TEST(sshkeySimple) {
        TSshPublicKey key(PublicKeyBody());
        TString err;

        const TString sign1 = Base64ToBin(
            "fIHIitqCNc4mKvqbhax25DsIHE1YXmRH5plzKBsiKqJt7P4sNLzNhYssLjHwg25p+n6cQz/Hx+v1mYBlmKf/jO7ObWTW2UF4t0LGn411MmHx/NAvRthNet7yoBkkxUbulBDAHe0S6kbmCPdQoCSowwr/1Ft6sJp96dX8gAaYfx21gxneRQ5pXeDcALwy3B3/SObwv5+4S7Iu27PYQMUkOhOJkW6rQy1yZwZygs+9+zb94Iafbw9KK926S44IlUJByCKmYRXDhS4czSC8f4V7bJsfRjVW0MBnmxYvSR9Zd1qhTQgaaX10BZ6NZsxpWiZwHAcBbgAdK5guBSNdsc+3yw==");
        const TString data1 = "very_random_string";
        const TString sign2 = Base64ToBin(
            "F+dNBfQe5sKLWQLWI/GXK4ZzI46iXUD+FNHugq3+Gha5Wpc8Kr8seIOK6Kuq35IUK0XY/mHB2ObTXAjPb/W7cACT967ja41ZUYJluIxjU9Y2arHtigHz356OPBF9QNGJENHMmfuSWoiGVRs/RNt7Fs+3marT45gWOvJYRJ3HptMoAy5Cxdzr37+ImytcPKRbl+DYoPoFDFVKpeq0RLz5L5gZ0cNphbmMOn0j58HMu5kpOM4h5PU79iWnq/dCzlNj1YhkupT2/I8WtegP5xuXYLa38Fh/rXYlsSGTBv7NCbZbtpwhh16k8FE/auAWESoe2aCPE388ZAUTrdCfSGfy5A==");
        const TString data2 = "very_poor_string";

        UNIT_ASSERT(key.Verify(sign1, data1, TSshPublicKey::EMode::RSA, err));
        UNIT_ASSERT(key.Verify(sign2, data2, TSshPublicKey::EMode::RSA, err));
        UNIT_ASSERT(!key.Verify(sign2, data1, TSshPublicKey::EMode::RSA, err));
        UNIT_ASSERT(!key.Verify(sign1, data2, TSshPublicKey::EMode::RSA, err));

        UNIT_ASSERT(!key.Verify(sign1, data1, TSshPublicKey::EMode::RSA_PSS, err));
        UNIT_ASSERT(!key.Verify(sign2, data2, TSshPublicKey::EMode::RSA_PSS, err));
        UNIT_ASSERT(!key.Verify(sign2, data1, TSshPublicKey::EMode::RSA_PSS, err));
        UNIT_ASSERT(!key.Verify(sign1, data2, TSshPublicKey::EMode::RSA_PSS, err));
    }

    Y_UNIT_TEST(sshkeyAgent) {
        TSSHPrivateKey priv(PrivateKeyName());
        TSshPublicKey key(PublicKeyBody());
        TString err;

        const TString sign1 = TSshPublicKey::GetSshAgentSign(Base64ToBin(
            "AAAAB3NzaC1yc2EAAAEAfIHIitqCNc4mKvqbhax25DsIHE1YXmRH5plzKBsiKqJt7P4sNLzNhYssLjHwg25p+n6cQz/Hx+v1mYBlmKf/jO7ObWTW2UF4t0LGn411MmHx/NAvRthNet7yoBkkxUbulBDAHe0S6kbmCPdQoCSowwr/1Ft6sJp96dX8gAaYfx21gxneRQ5pXeDcALwy3B3/SObwv5+4S7Iu27PYQMUkOhOJkW6rQy1yZwZygs+9+zb94Iafbw9KK926S44IlUJByCKmYRXDhS4czSC8f4V7bJsfRjVW0MBnmxYvSR9Zd1qhTQgaaX10BZ6NZsxpWiZwHAcBbgAdK5guBSNdsc+3yw=="));
        const TString data1 = "very_random_string";
        const TString sign2 = TSshPublicKey::GetSshAgentSign(Base64ToBin(
            "AAAAB3NzaC1yc2EAAAEAF+dNBfQe5sKLWQLWI/GXK4ZzI46iXUD+FNHugq3+Gha5Wpc8Kr8seIOK6Kuq35IUK0XY/mHB2ObTXAjPb/W7cACT967ja41ZUYJluIxjU9Y2arHtigHz356OPBF9QNGJENHMmfuSWoiGVRs/RNt7Fs+3marT45gWOvJYRJ3HptMoAy5Cxdzr37+ImytcPKRbl+DYoPoFDFVKpeq0RLz5L5gZ0cNphbmMOn0j58HMu5kpOM4h5PU79iWnq/dCzlNj1YhkupT2/I8WtegP5xuXYLa38Fh/rXYlsSGTBv7NCbZbtpwhh16k8FE/auAWESoe2aCPE388ZAUTrdCfSGfy5A=="));
        const TString data2 = "very_poor_string";

        UNIT_ASSERT(!sign1.empty());
        UNIT_ASSERT(!sign2.empty());

        UNIT_ASSERT(key.Verify(sign1, data1, TSshPublicKey::EMode::RSA, err));
        UNIT_ASSERT(key.Verify(sign2, data2, TSshPublicKey::EMode::RSA, err));
        UNIT_ASSERT(!key.Verify(sign2, data1, TSshPublicKey::EMode::RSA, err));
        UNIT_ASSERT(!key.Verify(sign1, data2, TSshPublicKey::EMode::RSA, err));

        UNIT_ASSERT(!key.Verify(sign1, data1, TSshPublicKey::EMode::RSA_PSS, err));
        UNIT_ASSERT(!key.Verify(sign1, data1, TSshPublicKey::EMode::RSA_PSS, err));
        UNIT_ASSERT(!key.Verify(sign2, data1, TSshPublicKey::EMode::RSA_PSS, err));
        UNIT_ASSERT(!key.Verify(sign1, data2, TSshPublicKey::EMode::RSA_PSS, err));

        const TString sign03 = Base64ToBin(
            "fIHIitqCNc4mKvqbhax25DsIHE1YXmRH5plzKBsiKqJt7P4sNLzNhYssLjHwg25p+n6cQz/Hx+v1mYBlmKf/jO7ObWTW2UF4t0LGn411MmHx/NAvRthNet7yoBkkxUbulBDAHe0S6kbmCPdQoCSowwr/1Ft6sJp96dX8gAaYfx21gxneRQ5pXeDcALwy3B3/SObwv5+4S7Iu27PYQMUkOhOJkW6rQy1yZwZygs+9+zb94Iafbw9KK926S44IlUJByCKmYRXDhS4czSC8f4V7bJsfRjVW0MBnmxYvSR9Zd1qhTQgaaX10BZ6NZsxpWiZwHAcBbgAdK5guBSNdsc+3yw==");
        const TString sign04 = Base64ToBin(
            "F+dNBfQe5sKLWQLWI/GXK4ZzI46iXUD+FNHugq3+Gha5Wpc8Kr8seIOK6Kuq35IUK0XY/mHB2ObTXAjPb/W7cACT967ja41ZUYJluIxjU9Y2arHtigHz356OPBF9QNGJENHMmfuSWoiGVRs/RNt7Fs+3marT45gWOvJYRJ3HptMoAy5Cxdzr37+ImytcPKRbl+DYoPoFDFVKpeq0RLz5L5gZ0cNphbmMOn0j58HMu5kpOM4h5PU79iWnq/dCzlNj1YhkupT2/I8WtegP5xuXYLa38Fh/rXYlsSGTBv7NCbZbtpwhh16k8FE/auAWESoe2aCPE388ZAUTrdCfSGfy5A==");
        UNIT_ASSERT(TSshPublicKey::GetSshAgentSign(sign03).empty());
        UNIT_ASSERT(TSshPublicKey::GetSshAgentSign(sign04).empty());
    }

    Y_UNIT_TEST(sshkeyPss) {
        TSSHPrivateKey priv(PrivateKeyName());
        TSshPublicKey key(PublicKeyBody());
        TString err;

        const TString sign1 = Base64ToBin(
            "FqT+oNU77e01cfHGNCLESR6Uq1ywHEbIQc0IbCVH7MUEA/64XVQ011FCbhbCpHFtMLQD3hWHb92Ig2PONh+5l4SmJjRQb68GWm6ttcQlzT/5clKX86nSreJmRW14Vgnxa8VW8ttG0AEanwwDqDOO756xBxMd7xFiHhm+oKC+X6/LXeYbGc5OGURZL2A5hNF8MUS8Ps3ThhAwE9ak5Ul34y3EehpUBkigwxKb3n3EIW/RuQ4nqHTTy8vlRcKSAQ5xL2DjBHOkPZwZIiDCBHvD2Frg4005WGMijdrhG6EBuF9/ETRn7VzqyskGi5sfKaNloTq4cKCX6LHGbLOfz/EaUg==");
        const TString data1 = "very_random_string";
        const TString sign2 = Base64ToBin(
            "Fje6T5og9+rEV6myjCL3lMFItfwiCYieyDaDHbBrW3tu2e/63DHOrIwADySMECz9GO5HU1JKu0yinhtWBSRlZi9X9WXi4DkLY30zMt/kN1gc6vHKmtZYtWpH4Ixha5KnA5gYyF+ItPZrXuzmTZv1p7caeOlrTIeR2NaCkI9n5ObT7IXogvZNyygIs9Gt2eFis4QghUCKxisKYWDgDuVoDrQdPc1jRFoZNn+lEKS6WRcpUPJAc8v0Z7doLqYhrXeD84pBP/5k/vYFC9R8EqeBq6L30TRR+wpyr9wkzK8FsVUsDz4BMdSAm4oXgAUALBXmWYaWXXWpzlME0lH14VxbHQ==");
        const TString data2 = "very_poor_string";

        UNIT_ASSERT(!key.Verify(sign1, data1, TSshPublicKey::EMode::RSA, err));
        UNIT_ASSERT(!key.Verify(sign2, data2, TSshPublicKey::EMode::RSA, err));
        UNIT_ASSERT(!key.Verify(sign2, data1, TSshPublicKey::EMode::RSA, err));
        UNIT_ASSERT(!key.Verify(sign1, data2, TSshPublicKey::EMode::RSA, err));

        UNIT_ASSERT(key.Verify(sign1, data1, TSshPublicKey::EMode::RSA_PSS, err));
        UNIT_ASSERT(key.Verify(sign2, data2, TSshPublicKey::EMode::RSA_PSS, err));
        UNIT_ASSERT(!key.Verify(sign2, data1, TSshPublicKey::EMode::RSA_PSS, err));
        UNIT_ASSERT(!key.Verify(sign1, data2, TSshPublicKey::EMode::RSA_PSS, err));

        const TString sign3 = priv.Sign(data1, TSSHPrivateKey::EPss::Use);
        const TString sign4 = priv.Sign(data2, TSSHPrivateKey::EPss::Use);
        UNIT_ASSERT(!key.Verify(sign3, data1, TSshPublicKey::EMode::RSA, err));
        UNIT_ASSERT(!key.Verify(sign4, data2, TSshPublicKey::EMode::RSA, err));
        UNIT_ASSERT(!key.Verify(sign4, data1, TSshPublicKey::EMode::RSA, err));
        UNIT_ASSERT(!key.Verify(sign3, data2, TSshPublicKey::EMode::RSA, err));

        UNIT_ASSERT(key.Verify(sign3, data1, TSshPublicKey::EMode::RSA_PSS, err));
        UNIT_ASSERT(key.Verify(sign4, data2, TSshPublicKey::EMode::RSA_PSS, err));
        UNIT_ASSERT(!key.Verify(sign4, data1, TSshPublicKey::EMode::RSA_PSS, err));
        UNIT_ASSERT(!key.Verify(sign3, data2, TSshPublicKey::EMode::RSA_PSS, err));
    }

    Y_UNIT_TEST(sshkeyAgentPss) {
        TSSHPrivateKey priv(PrivateKeyName());
        TSshPublicKey key(PublicKeyBody());
        TString err;

        const TString sign01 = Base64ToBin(
            "AAAAB3NzaC1yc2EAAAEAYW6pudfcfLZgJLfJzjVXzeA6YSSWokG3SjI0O+z/kjgVrcJaVWJPR457IGD0Ps0OaMSUEDq6ARnlPXTZJCEnZlpn9o2Qs3aSkdFp9dOmDsvOK0oPKwFL8moJDD1KsF7H+/9+TolTslY4d583BvjB7mn9/q4/m4Erazhh06MTkad/CmKxm6rRiHY6PVfeOOjqUTpRbJe3FezB4RHBH/TMxrxW7F5WtFBtXhnMmg5TMx0HbfbnBO1tN7COfY/r3+Zq+uvj/hBOkOemUnZ9u9Xc5AXOinSuliU45kyK1qBSHWOqZBkbd46XC8Z46FatGYpXnZeHmBzRcQgRekMkjZsXtg==");
        const TString sign1 = TSshPublicKey::GetSshAgentSign(sign01);
        const TString data1 = "very_random_string";
        const TString sign02 = Base64ToBin(
            "AAAAB3NzaC1yc2EAAAEAeVQR74OxP/bLEO0j6VxmXwB1SihbOTiB+xkcFUXigiJlnHzSs/2TQ3JEXZXVXrnJqA9m9hSLuLI4vZlmNIWZqAaEQ2D8E4vyX05y6LIPO1GhhTPGOHQbwaQPqQ5dZrcDAM8C+ZQyBixA90AyiSuK7JmshvdZVxfniwWJ3TJoLRit6xiyVUDgUvJygTswi+Z3GJeLUaCj95JK2qQkuN7zblhjlXceXF4IjexmghLGxJ2unRFT6q/6lbFUVmqgXDwncDOZDUe2vYY+zxb7iiwlK6wUKmMBVsbFL4DBvoo+7l8U6Twrl3CcxUzp3JlByGAkkYsibpXgHMhWQeEu7xqF8g==");
        const TString sign2 = TSshPublicKey::GetSshAgentSign(sign02);
        const TString data2 = "very_poor_string";

        UNIT_ASSERT(!sign1.empty());
        UNIT_ASSERT(!sign2.empty());

        UNIT_ASSERT(!key.Verify(sign1, data1, TSshPublicKey::EMode::RSA, err));
        UNIT_ASSERT(!key.Verify(sign2, data2, TSshPublicKey::EMode::RSA, err));
        UNIT_ASSERT(!key.Verify(sign2, data1, TSshPublicKey::EMode::RSA, err));
        UNIT_ASSERT(!key.Verify(sign1, data2, TSshPublicKey::EMode::RSA, err));

        UNIT_ASSERT(key.Verify(sign1, data1, TSshPublicKey::EMode::RSA_PSS, err));
        UNIT_ASSERT(key.Verify(sign1, data1, TSshPublicKey::EMode::RSA_PSS, err));
        UNIT_ASSERT(!key.Verify(sign2, data1, TSshPublicKey::EMode::RSA_PSS, err));
        UNIT_ASSERT(!key.Verify(sign1, data2, TSshPublicKey::EMode::RSA_PSS, err));

        const TString sign3 = TSshPublicKey::GetSshAgentSign(priv.Sign(data1, TSSHPrivateKey::EPss::Use, TSSHPrivateKey::EAgent::Simulate));
        const TString sign4 = TSshPublicKey::GetSshAgentSign(priv.Sign(data2, TSSHPrivateKey::EPss::Use, TSSHPrivateKey::EAgent::Simulate));
        UNIT_ASSERT(!key.Verify(sign3, data1, TSshPublicKey::EMode::RSA, err));
        UNIT_ASSERT(!key.Verify(sign4, data2, TSshPublicKey::EMode::RSA, err));
        UNIT_ASSERT(!key.Verify(sign4, data1, TSshPublicKey::EMode::RSA, err));
        UNIT_ASSERT(!key.Verify(sign3, data2, TSshPublicKey::EMode::RSA, err));

        UNIT_ASSERT(key.Verify(sign3, data1, TSshPublicKey::EMode::RSA_PSS, err));
        UNIT_ASSERT(key.Verify(sign4, data2, TSshPublicKey::EMode::RSA_PSS, err));
        UNIT_ASSERT(!key.Verify(sign4, data1, TSshPublicKey::EMode::RSA_PSS, err));
        UNIT_ASSERT(!key.Verify(sign3, data2, TSshPublicKey::EMode::RSA_PSS, err));

        const TString sign03 = Base64ToBin(
            "FqT+oNU77e01cfHGNCLESR6Uq1ywHEbIQc0IbCVH7MUEA/64XVQ011FCbhbCpHFtMLQD3hWHb92Ig2PONh+5l4SmJjRQb68GWm6ttcQlzT/5clKX86nSreJmRW14Vgnxa8VW8ttG0AEanwwDqDOO756xBxMd7xFiHhm+oKC+X6/LXeYbGc5OGURZL2A5hNF8MUS8Ps3ThhAwE9ak5Ul34y3EehpUBkigwxKb3n3EIW/RuQ4nqHTTy8vlRcKSAQ5xL2DjBHOkPZwZIiDCBHvD2Frg4005WGMijdrhG6EBuF9/ETRn7VzqyskGi5sfKaNloTq4cKCX6LHGbLOfz/EaUg==");
        const TString sign04 = Base64ToBin(
            "Fje6T5og9+rEV6myjCL3lMFItfwiCYieyDaDHbBrW3tu2e/63DHOrIwADySMECz9GO5HU1JKu0yinhtWBSRlZi9X9WXi4DkLY30zMt/kN1gc6vHKmtZYtWpH4Ixha5KnA5gYyF+ItPZrXuzmTZv1p7caeOlrTIeR2NaCkI9n5ObT7IXogvZNyygIs9Gt2eFis4QghUCKxisKYWDgDuVoDrQdPc1jRFoZNn+lEKS6WRcpUPJAc8v0Z7doLqYhrXeD84pBP/5k/vYFC9R8EqeBq6L30TRR+wpyr9wkzK8FsVUsDz4BMdSAm4oXgAUALBXmWYaWXXWpzlME0lH14VxbHQ==");
        UNIT_ASSERT(TSshPublicKey::GetSshAgentSign(sign03).empty());
        UNIT_ASSERT(TSshPublicKey::GetSshAgentSign(sign04).empty());

        UNIT_ASSERT(TSshPublicKey::GetSshAgentSign("abc").empty());
        std::array<char, 11> prefix = {0x00, 0x00, 0x00, 0x07, 's', 's', 'h', '-', 'r', 's', 'a'};
        UNIT_ASSERT(TSshPublicKey::GetSshAgentSign(TString(prefix.begin(), prefix.end()) + "abcdef").empty());
    }

    Y_UNIT_TEST(rsaFromPem) {
        UNIT_ASSERT_NO_EXCEPTION(TRsaPublicEvp::FromPem(
            R"(
-----BEGIN PUBLIC KEY-----
MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAsjV1xEmZLw4lTDUDaCd0
CdPYX/2chWGZ+GguJ1UwejbhCBXA+/gFYy2ibAbhVfivW89bWQCi3/LTnJHoejrZ
1LQEEc9Yexap4tQdqXYxbLtHztATuFi5Ellr55Z9/2PC/RV1JY4DC8T879CSrEx6
ccgSy2SGXdJMPu1ml2VED3UO436O/IdGcO+9Lanf1dBcT3C+hjVrVngEzp3rREoV
6pZ5eQvRJ8J4JuIzPsHafC1ZfnJMprjux78H3W9gzIwPSxVWJENplr2RKMOZUMgK
qUTekPu0gopJAAG/Iedk4QyUul43ZCEbce5lJxgbCBGZQhT2ObKju+ft8rv3Rrhh
AwIDAQAB
-----END PUBLIC KEY-----
)"));

        UNIT_ASSERT_NO_EXCEPTION(TRsaPublicEvp::FromPem(
            R"(
-----BEGIN PUBLIC KEY-----
    MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAsjV1xEmZLw4lTDUDaCd0
    CdPYX/2chWGZ+GguJ1UwejbhCBXA+/gFYy2ibAbhVfivW89bWQCi3/LTnJHoejrZ
    1LQEEc9Yexap4tQdqXYxbLtHztATuFi5Ellr55Z9/2PC/RV1JY4DC8T879CSrEx6
    ccgSy2SGXdJMPu1ml2VED3UO436O/IdGcO+9Lanf1dBcT3C+hjVrVngEzp3rREoV
    6pZ5eQvRJ8J4JuIzPsHafC1ZfnJMprjux78H3W9gzIwPSxVWJENplr2RKMOZUMgK
    qUTekPu0gopJAAG/Iedk4QyUul43ZCEbce5lJxgbCBGZQhT2ObKju+ft8rv3Rrhh
    AwIDAQAB
-----END PUBLIC KEY-----
)"));

        UNIT_ASSERT_NO_EXCEPTION(TRsaPublicEvp::FromPem(
            R"(
-----BEGIN PUBLIC KEY-----
MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAsjV1xEmZLw4lTDUDaCd0
        CdPYX/2chWGZ+GguJ1UwejbhCBXA+/gFYy2ibAbhVfivW89bWQCi3/LTnJHoejrZ
        1LQEEc9Yexap4tQdqXYxbLtHztATuFi5Ellr55Z9/2PC/RV1JY4DC8T879CSrEx6
    ccgSy2SGXdJMPu1ml2VED3UO436O/IdGcO+9Lanf1dBcT3C+hjVrVngEzp3rREoV
    6pZ5eQvRJ8J4JuIzPsHafC1ZfnJMprjux78H3W9gzIwPSxVWJENplr2RKMOZUMgK
    qUTekPu0gopJAAG/Iedk4QyUul43ZCEbce5lJxgbCBGZQhT2ObKju+ft8rv3Rrhh
        AwIDAQAB
-----END PUBLIC KEY-----
)"));

        UNIT_ASSERT_NO_EXCEPTION(TRsaPublicEvp::FromPem(
            R"(-----BEGIN PUBLIC KEY-----
MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAsjV1xEmZLw4lTDUDaCd0
CdPYX/2chWGZ+GguJ1UwejbhCBXA+/gFYy2ibAbhVfivW89bWQCi3/LTnJHoejrZ
1LQEEc9Yexap4tQdqXYxbLtHztATuFi5Ellr55Z9/2PC/RV1JY4DC8T879CSrEx6
ccgSy2SGXdJMPu1ml2VED3UO436O/IdGcO+9Lanf1dBcT3C+hjVrVngEzp3rREoV
6pZ5eQvRJ8J4JuIzPsHafC1ZfnJMprjux78H3W9gzIwPSxVWJENplr2RKMOZUMgK
qUTekPu0gopJAAG/Iedk4QyUul43ZCEbce5lJxgbCBGZQhT2ObKju+ft8rv3Rrhh
AwIDAQAB
-----END PUBLIC KEY-----)"));

        UNIT_ASSERT_NO_EXCEPTION(TRsaPublicEvp::FromPem(
            R"(

-----BEGIN PUBLIC KEY-----
MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAsjV1xEmZLw4lTDUDaCd0
CdPYX/2chWGZ+GguJ1UwejbhCBXA+/gFYy2ibAbhVfivW89bWQCi3/LTnJHoejrZ
1LQEEc9Yexap4tQdqXYxbLtHztATuFi5Ellr55Z9/2PC/RV1JY4DC8T879CSrEx6
ccgSy2SGXdJMPu1ml2VED3UO436O/IdGcO+9Lanf1dBcT3C+hjVrVngEzp3rREoV
6pZ5eQvRJ8J4JuIzPsHafC1ZfnJMprjux78H3W9gzIwPSxVWJENplr2RKMOZUMgK
qUTekPu0gopJAAG/Iedk4QyUul43ZCEbce5lJxgbCBGZQhT2ObKju+ft8rv3Rrhh
AwIDAQAB
-----END PUBLIC KEY-----

            )"));

        UNIT_ASSERT_NO_EXCEPTION(TRsaPublicEvp::FromPem(
            R"(-----BEGIN PUBLIC KEY-----
MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAsjV1xEmZLw4lTDUDaCd0CdPYX/2chWGZ+GguJ1UwejbhCBXA+/gFYy2ibAbhVfivW89bWQCi3/LTnJHoejrZ1LQEEc9Yexap4tQdqXYxbLtHztATuFi5Ellr55Z9/2PC/RV1JY4DC8T879CSrEx6ccgSy2SGXdJMPu1ml2VED3UO436O/IdGcO+9Lanf1dBcT3C+hjVrVngEzp3rREoV6pZ5eQvRJ8J4JuIzPsHafC1ZfnJMprjux78H3W9gzIwPSxVWJENplr2RKMOZUMgKqUTekPu0gopJAAG/Iedk4QyUul43ZCEbce5lJxgbCBGZQhT2ObKju+ft8rv3RrhhAwIDAQAB
-----END PUBLIC KEY-----)"));

        UNIT_ASSERT_NO_EXCEPTION(TRsaPublicEvp::FromPem(
            R"(-----BEGIN PUBLIC KEY-----
MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAsjV1xEmZLw4lTDUDaCd0CdPYX/2chWGZ+GguJ1UwejbhCBXA+/gFYy2ibAbhVfivW89bWQCi3/LTnJHoejrZ
1LQEEc9Yexap4tQdqXYxbLtHztATuFi5Ellr55Z9/2PC/RV1JY4DC8T879CSrEx6
ccgSy2SGXdJMPu1ml2VED3UO436O/IdGcO+9Lanf1dBcT3C+hjVrVngEzp3rREoV
6pZ5eQvRJ8J4JuIzPsHafC1ZfnJMprjux78H3W9gzIwPSxVWJENplr2RKMOZUMgK
qUTekPu0gopJAAG/Iedk4QyUul43ZCEbce5lJxgbCBGZQhT2ObKju+ft8rv3RrhhAwIDAQAB
-----END PUBLIC KEY-----)"));

        UNIT_ASSERT_EXCEPTION_CONTAINS(TRsaPublicEvp::FromPem(
                                           R"(
 -----BEGIN PUBLIC KEY-----
MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAsjV1xEmZLw4lTDUDaCd0
CdPYX/2chWGZ+GguJ1UwejbhCBXA+/gFYy2ibAbhVfivW89bWQCi3/LTnJHoejrZ
1LQEEc9Yexap4tQdqXYxbLtHztATuFi5Ellr55Z9/2PC/RV1JY4DC8T879CSrEx6
ccgSy2SGXdJMPu1ml2VED3UO436O/IdGcO+9Lanf1dBcT3C+hjVrVngEzp3rREoV
6pZ5eQvRJ8J4JuIzPsHafC1ZfnJMprjux78H3W9gzIwPSxVWJENplr2RKMOZUMgK
qUTekPu0gopJAAG/Iedk4QyUul43ZCEbce5lJxgbCBGZQhT2ObKju+ft8rv3Rrhh
AwIDAQAB
-----END PUBLIC KEY-----
)"), // missing minus
                                       yexception,
                                       "Failed PEM_read_bio_PUBKEY: no start line");

        UNIT_ASSERT_EXCEPTION_CONTAINS(TRsaPublicEvp::FromPem(
                                           R"(
-----BEGIN PUBLIC KEY----
MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAsjV1xEmZLw4lTDUDaCd0
CdPYX/2chWGZ+GguJ1UwejbhCBXA+/gFYy2ibAbhVfivW89bWQCi3/LTnJHoejrZ
1LQEEc9Yexap4tQdqXYxbLtHztATuFi5Ellr55Z9/2PC/RV1JY4DC8T879CSrEx6
ccgSy2SGXdJMPu1ml2VED3UO436O/IdGcO+9Lanf1dBcT3C+hjVrVngEzp3rREoV
6pZ5eQvRJ8J4JuIzPsHafC1ZfnJMprjux78H3W9gzIwPSxVWJENplr2RKMOZUMgK
qUTekPu0gopJAAG/Iedk4QyUul43ZCEbce5lJxgbCBGZQhT2ObKju+ft8rv3Rrhh
AwIDAQAB
-----END PUBLIC KEY-----
)"), // missing minus
                                       yexception,
                                       "Failed PEM_read_bio_PUBKEY: no start line");

        UNIT_ASSERT_EXCEPTION_CONTAINS(TRsaPublicEvp::FromPem(
                                           R"(
-----BEGIN PUBLIC KEY-----
MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAsjV1xEmZLw4lTDUDaCd0
CdPYX/2chWGZ+GguJ1UwejbhCBXA+/gFYy2ibAbhVfivW89bWQCi3/LTnJHoejrZ
1LQEEc9Yexap4tQdqXYxbLtHztATuFi5Ellr55Z9/2PC/RV1JY4DC8T879CSrEx6
ccgSy2SGXdJMPu1ml2VED3UO436O/IdGcO+9Lanf1dBcT3C+hjVrVngEzp3rREoV
6pZ5eQvRJ8J4JuIzPsHafC1ZfnJMprjux78H3W9gzIwPSxVWJENplr2RKMOZUMgK
qUTekPu0gopJAAG/Iedk4QyUul43ZCEbce5lJxgbCBGZQhT2ObKju+ft8rv3Rrhh
AwIDAQAB

-----END PUBLIC KEY-----
)"), // extra \n
                                       yexception,
                                       "Failed PEM_read_bio_PUBKEY: ");

        UNIT_ASSERT_EXCEPTION_CONTAINS(TRsaPublicEvp::FromPem(
                                           R"(
-----BEGIN PUBLIC KEY-----
MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAsjV1xEmZLw4lTDUDaCd0
CdPYX/2chWGZ+GguJ1UwejbhCBXA+/gFYy2ibAbhVfivW89bWQCi3/LTnJHoejrZ
1LQEEc9Yexap4tQdqXYxbLtHztATuFi5Ellr55Z9/2PC/RV1JY4DC8T879CSrEx6
ccgSy2SGXdJMPu1ml2VED3UO436O/IdGcO+9Lanf1dBcT3C+hjVrVngEzp3rREoV
6pZ5eQvRJ8J4JuIzPsHafC1ZfnJMprjux78H3W9gzIwPSxVWJENplr2RKMOZUMgK
qUTekPu0gopJAAG/Iedk4QyUul43ZCEbce5lJxgbCBGZQhT2ObKju+ft8rv3Rrhh
AwIDAQA
-----END PUBLIC KEY-----
)"), // missing last char of key
                                       yexception,
                                       "Failed PEM_read_bio_PUBKEY: bad base64 decode");

        UNIT_ASSERT_EXCEPTION_CONTAINS(TRsaPublicEvp::FromPem(
                                           R"(-----BEGIN PUBLIC KEY-----
MIIDRzCCAjkGByqGSM44BAEwggIsAoIBAQDEJWPq9m1/p5VayE5mzaixAUVoHWPK
qh76dzGwqqvALRv+aq4RHNXLo9Jk4mJKi2gFVydWsARc2KxhZzCsP9tN3co0AKOs
H7FbUfgtp5/ge15ycVE2B/gy4SuMM9LY/HWVCkAPuiu/UiJEiN3P9Hf9XllWHVGH
AWQsjPPSpIdzw4C9WfyTOoXxIa8oYy3plWTLMYawqAjYjeBXu5rEAJ78PfwQ9FsQ
CYmDh1UYEqvD7k57+UE3vRNAhdTPP8oh/AyppMYXlCIs3Kc9IjP1/AeHfdTUeN6C
iIdfkq4pj0KraAWvDs0BrD1uE2W3D1RX2fziIWR1gr9t5/QKpntrx9r/AiEAiWSG
5QJK3S9e3gm3jjXoy7NDO36MIH0VOIbql2+Z7iECggEAXqJ7d2rS7D+vLueAUgNK
vIwidYtvia+pAQXT8Md9PlxSQ1f92WK+dRdJDptI7pp+kIb+Nlh9qZyM1NlMTHdX
UsMKpFWuholRYLIhp1/9GnNUHtbyN29dW7X+iRUiVxTLlPFLynBGAzNKI4Mva+TW
Zlt/yQDVRTl/D5mVeQpxQ2G6DCWitBQnHPksIO2xymyZRM5zsTz1D+74o9tSahFi
4RMO28AeDVN863xSST4bNC/7TBH4dfr/st+lku2m6fnAh3pKVGSsWcTpowC3zsV/
Anujwmn6Xtah7D/vVxtao5NSgsbTVF8iZ5Li1EzPVlKrJccNRoG0j4O4qqBA+1n+
jgOCAQYAAoIBAQC8jlsMQtRKkKZtjWOuhJ47lKZn2ceL6m2E8iNaW5nrlS5YHFOg
CZN0yJoMna8oziXUK7QJCFtIUzXtunTe6e4PSpEhUzwrwnkhkbyOt2SFahJDLzZg
zalAMuJrdLrf6x3BYETnvgJsByMwuUxdbNO0ppG9fqrh7hWSW2VR1G6OLmR+39Qo
hiVSA7prOPhaw2DhDO7Zvj92nlPWoy9+HOM3jrgeW3nTACWiQT7okivMxIiRzFCm
wzfq+X1BUKj+On7XG0ftUvAqoaOOsFKFsFFPMtwdMMCMhjNYOSF0++74l2gl96x+
5CrKskBBACekwc7jl121RCHCufcdBywG35cU
-----END PUBLIC KEY-----)"), // DSA key
                                       yexception,
                                       "Wrong type of key: expected==6(rsaEncryption), actual==116(DSA)");
    }

    Y_UNIT_TEST(RsaVerifyWithSha256) {
        //// Success
        TRsaPublicEvp rsa1 = TRsaPublicEvp::FromPem(
            R"(-----BEGIN PUBLIC KEY-----
MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAu+tp41NZteINpO4rEu/JoPeA5im3L0cJ
Ztuo+tTvqJbf5QbGVSnPF5Gryc0Rn8jIJCT8m0Y1dC8KOXO16HFoT97dqMEe6ssG7HSDNzmy3mMJ
6rOqeWDZ3ztpmYNIq1FQU2KuJ7UyucoFhJ6uYoL6SbbIv4ev7z3rvZsV1hl5G0SwxSC+DrRkjFa3
Aqy0/94E+TOV3ioqORrTStz0xc1VwJ21vP8yN3RNXsl/jjErU70jTegAl5ormTGQ9f9jzu2Hqz0c
ALqEyLR80VygpA+dZAAyznsiyfMvbT4Bt/fvKWozmI9TqEDxr4utVMbp3iX+G+5imJcSauzR9cqV
EDGRFQIDAQAB
-----END PUBLIC KEY-----)");

        UNIT_ASSERT_VALUES_EQUAL(TRsaPublicEvp::TResult({true, ""}),
                                 rsa1.VerifyWithSha256(
                                     "0#test",
                                     Base64ToBin("U6/pHMzAvp6DEz0BjFghXVafO6UTqVfW4imX1inVqCHHgVMEHnX/2d0N1S7hpSdETWRP5o96vnLLbQWZs9Zo+l9fLIaSN5ilOJxNvRFkkIH8c9eg/c4Q0M8+63Ke8Lnsm+8VUAzwrXSOOb932OZDz6o/u4fk5e0oWQ3MxfFsFwuosApwzuFXFTnPREPC0PFYwWSB5G4UcY2Vxqf+j9LREMQ2ZYyYOwNVUkTMRFmCmIhAZzgGUdv367FEsciDoz1fBzLCq5eTr/ueRUjQ6K0R2lrLdJ0njnQ9J9R+vN4ICNMhiSbSGSYtss/IIeQXz4ytQt9EfaSc/LT5hwLUt4pbjw==")));
        UNIT_ASSERT_VALUES_EQUAL(TRsaPublicEvp::TResult({true, ""}),
                                 rsa1.VerifyWithSha256(
                                     "1#test",
                                     Base64ToBin("c1BDJMHQQhS19IWW5lNY3vzJsKOIavx99nHzKnbw8LwKqRbfL+np90ai9bzBpdq0kWQIK/au1/NmmplhNyGJUJ8oQzQbowH4MKPTn9vySeLtS1Y6Mjg0Z5DrSAJGo2j4rFOHJxLrBKsB19ccra6Scq3DAX5hSeNtD7C88ZNRtTi1GltNr2mOKU8hqXb3FHkQJNc68Sy34oESQ/CApLotyHu9POYkOvaEGVVZjgFYSQIaLy+0yD/iuyqmNgYKphMKuxPBdktygGU4L5BYtzCuGuzn2K38y4nX5sqUtr3NWiGllHsav9CsQpGPV6Xvy6E/Nk1OGDRYMgA3hzqK/pj7nA==")));
        UNIT_ASSERT_VALUES_EQUAL(TRsaPublicEvp::TResult({true, ""}),
                                 rsa1.VerifyWithSha256(
                                     "2#test",
                                     Base64ToBin("afX4J3to2poV3VvZnYjsybDorjGem8x9zhvS77cdUN2re61Ktt+oNDgdRkkzn/GWOdA3PZds4xKtidVj2c3gMc6PhP+WtxxP5yl0M2W9mFHlvq8GzLOcujILxBD1xv7sn3vP1d62GX+UdU/QmwNoTwG3wUnnFDKCu78904CxUw282LSpns9OicQ+ANP/XSTgoZmUZXvCz6/6K93xoilxHc5m/mk521CxvIDwspaVSiPifwW5AMN7y/BoiNk2rEJjWLUqSEbcxTWer6FQ8NxWA8H8ZNg71UIZie6DqZ8/KU5OZBnnXhPz9S+HqfUVfbe1d9Bs0O/IJdjfQeJVjFra5w==")));
        UNIT_ASSERT_VALUES_EQUAL(TRsaPublicEvp::TResult({true, ""}),
                                 rsa1.VerifyWithSha256(
                                     "3#test",
                                     Base64ToBin("QqAODaaGBNHUMK2vqfSNlwIuD095u1DnBzeMfmEY2zBp743YkT7PwQEO4Uo55bwb0uLW/nBq5zmhlN+euwBbcmg9zPbfGdS1SZhJbxSbw5pIqS1IwEIo9XTEbeYxan3tHoZeRycN9JeFL3Bkwb0ZPxzrwlzKgU3BHUSL2ahFeUa6V0K9DwG8z6tKfMpDaAhNg+i2dho9Lp+LJG2FxAQRWzXFwXvwUbXkFU0boByuiMcpCSGgEncWBDiee4lR/6zRN8ihLihabqKr8KAvip9PfJ6sA1e0XFZvi68sd0aVzatnJG+hEb1GMzgNPA3RrB1Qndv1Uk1c/ABJgk9+B8aoIw==")));
        UNIT_ASSERT_VALUES_EQUAL(TRsaPublicEvp::TResult({true, ""}),
                                 rsa1.VerifyWithSha256(
                                     "4#test",
                                     Base64ToBin("haiN3z5o7moltZvytHOL1MxGyIIBSnFN/+WVIBgHYjswH2c4pHVx5Y8w87+PTnUSYIlG5eIqKpRyhwQHx19Zk0zJ8KL6XhAzQX3N8H8cBkAbTTcIRoo1pzwdaxChNf5x52d2TTc8aEJLtc1pZSKfZUjvibP6BvJf1m7vJQbE8n68AI1xD5sSm34SWcWpDl8GqNp5p3OzT8hi3UVCxwgS6IDC649T5X0olILBlYuFZzVuUw/Ikf1XBp6WOUvuIprZoHiKJT4TI8G7sUwbW/ApWRL19dNpZfvkbwoFdEw4SXwCGFMUbGBlqq0gywcl8acla7Vdv5yTSaeXGfy2gB4Vhw==")));

        TRsaPublicEvp rsa2 = TRsaPublicEvp::FromPem(
            R"(-----BEGIN PUBLIC KEY-----
MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAv69kTN0r+RejKHyzFyl5meiTh6Ul9L+D
05kL7b0MDE8cjtZeX8VeWjk1gShJGoMNTMQ431mI4YpmB3DbiNGqVlnocQ9bCeJiHHknKJn81KG8
zB5UOoogM6+q+bVPOX+x+Opy0AUmS+OgXh/60tL4C/I9zrP+FwykbWmt25EPsDQKXgWzyNK4hAJR
xtWZpPk+oxREDYwFitmNTUWaxq1YpayCJOqb7aBJEFcsd6tlOSAQ88a7BtIB1r/ueySyQmY5RShz
VR6f3dWxHfJshcCvIx8ly5hShi1LLXdwG7KJJ6RhW5t+Z2PNBrABFkBWaAQX39KoizNMeYl/wGib
yDPFswIDAQAB
-----END PUBLIC KEY-----)");

        UNIT_ASSERT_VALUES_EQUAL(TRsaPublicEvp::TResult({true, ""}),
                                 rsa2.VerifyWithSha256(
                                     "5#test",
                                     Base64ToBin("FmcWs0nHbBQKeIqvkb4wjU6YJWvut16X3Jpoi9GAmH1yJ2vywEpWZfAocQmkGwoGblC3m2RHAY2Vg7749qGMAzt99LfufmDI2za6OHpOuA71nNy8a7LC3QfKRhJ6rQ9k8J/yGtVjxYqicvtjdb7kMCvA6B5SPqZRDQ9bQoaZc9zjFVvAjtR9r9/ddijRiSEHZjRYd4LEqTfshzvn+S7XFn+ww/xGC+MN+MJPJdf8DcK37lBoQTMss4TccrvZ4aPu40ybroxNQxrXF/POctc+3eIwtqvt2CcwM7euVLTUNH4L9nQH/JDu4AUrh5OeVmaxr6QkGqPZ0hdFwHgt090Xww==")));
        UNIT_ASSERT_VALUES_EQUAL(TRsaPublicEvp::TResult({true, ""}),
                                 rsa2.VerifyWithSha256(
                                     "6#test",
                                     Base64ToBin("WVQ2xLDdXJlWodyH1lIrcNbOssrMJgaMJheANOYjDlTZ5FkzfGpsE5CliWwVIcRPjcPYljyqkZhU8n2pm9YV4dHKqOBUJSieWzAuagYvISG5COWeqXmakWYf4rVERdFa29Th4m28Vb1QW7rU3uSpKltfBdFSVX7zqOm/pDzDRb4O2kLVN3gIz3/YSVnnNBJLmt5twCbq9Hm+kUsYRAiovhRY3vAnXH+Gy5yjZzLrjHkF9Y/fCv5rk3TYfqPDuN31Gal5sXuwnctV3eB92Il5x2eUEkH49s3h5Akt3yF3gNyX9jmGPp8V73IS6iDMaOuUH4RWHTz4p5F+zgTupKF6Lw==")));
        UNIT_ASSERT_VALUES_EQUAL(TRsaPublicEvp::TResult({true, ""}),
                                 rsa2.VerifyWithSha256(
                                     "7#test",
                                     Base64ToBin("Ex5YLU6I9Vw3/u/ihYB1U3igCqZN0OgOA7ZlimD6arjbl+9assMe+KJkxxI2CxOjBPnYDu9pSvf+PRia3Zeww7tEfjOtT4LlEBYVKiXScZyZ/nLkR7cipziSPD5NeAeBcNHZvWxu0tivsj5BKca5XLOlTWpzyFnp7GQwdLnMjVJLvKvBR1FHKHD368pfpWMtx+gYw935HpVB9SUFakDwyDuO5eUTqU8Y2I2Y2WBL9SFnGpmuSg3wnCCEcb2DeqtK61GwqWyixa7a3sGFUbD8zk1Cz8pnJIC4mp108ICERYSlI5TnxaxVbqNTCEXdA5xpRYSYWB/aHtpzFCSsN75Siw==")));
        UNIT_ASSERT_VALUES_EQUAL(TRsaPublicEvp::TResult({true, ""}),
                                 rsa2.VerifyWithSha256(
                                     "8#test",
                                     Base64ToBin("hKt7tnrvTSeK4C6OHiA9b+J5G+DvpJl1LuCJhbzlLO62ZPY4izaMR1ENYGmUupSYd/p3lPITBGFNog65/C6Px9azq5yWTtxmfgRn82cbOqaUbG1RlmYORfMGXdqOWEK1O09aEx0I6g0tEeppYPEBBVCQEXZ+fROLdbPTn0L5fFQKGvp9x71FZy6XUXImvj/oznMI2Km9JO//wXnNCI0ahKhFMt+PSqXg0dXFyq0aIsyWmzBPYgj7WegwT0gCyaY2HaXKnHn5AagXz3XHnYC4bbtlSM/1IQ7knh6TsapS+MdaeQqTdd5uV3HFtq1wSimNG/Dnt5/7rBJitztGA1ZWjQ==")));
        UNIT_ASSERT_VALUES_EQUAL(TRsaPublicEvp::TResult({true, ""}),
                                 rsa2.VerifyWithSha256(
                                     "9#test",
                                     Base64ToBin("sqw/ynx1et2E+bT/CwZXoNN5JzaEEjtJTtGY+YAAnKYXEB9Wfhfy1YfkTaa0wFoM2vmt7KKttO10c4w43i+tB7ZjGm9jg08w/7L4rXOjpauFROxCtszIi1VbKWiNOxd3vX10cBde2uqYxU+N8NIDor5Nb7C18GXk/vQ/5xdxlBFXgS4do9PfGHCbaDYV/ab/X+0rGpxtg1IB6KfDDaF17kvuiCLQW53A8Pk8rhEYz9g6cjTft9oUKcpBZBbd/2jo4M8ZmN0JbelQqqwgXMqU8kLV6dxk118Q27kH/CPuli5/gyYZ8up4T+3cV44rKaifExtWpOsxCUKNTfJIdmkgtw==")));

        //// Fail
        // wrong data
        UNIT_ASSERT_VALUES_EQUAL(TRsaPublicEvp::TResult({false, "bad signature"}),
                                 rsa1.VerifyWithSha256(
                                     "1#test",
                                     Base64ToBin("U6/pHMzAvp6DEz0BjFghXVafO6UTqVfW4imX1inVqCHHgVMEHnX/2d0N1S7hpSdETWRP5o96vnLLbQWZs9Zo+l9fLIaSN5ilOJxNvRFkkIH8c9eg/c4Q0M8+63Ke8Lnsm+8VUAzwrXSOOb932OZDz6o/u4fk5e0oWQ3MxfFsFwuosApwzuFXFTnPREPC0PFYwWSB5G4UcY2Vxqf+j9LREMQ2ZYyYOwNVUkTMRFmCmIhAZzgGUdv367FEsciDoz1fBzLCq5eTr/ueRUjQ6K0R2lrLdJ0njnQ9J9R+vN4ICNMhiSbSGSYtss/IIeQXz4ytQt9EfaSc/LT5hwLUt4pbjw==")));

        // wrong key
        UNIT_ASSERT_VALUES_EQUAL(TRsaPublicEvp::TResult({false, "invalid padding"}),
                                 rsa2.VerifyWithSha256(
                                     "0#test",
                                     Base64ToBin("U6/pHMzAvp6DEz0BjFghXVafO6UTqVfW4imX1inVqCHHgVMEHnX/2d0N1S7hpSdETWRP5o96vnLLbQWZs9Zo+l9fLIaSN5ilOJxNvRFkkIH8c9eg/c4Q0M8+63Ke8Lnsm+8VUAzwrXSOOb932OZDz6o/u4fk5e0oWQ3MxfFsFwuosApwzuFXFTnPREPC0PFYwWSB5G4UcY2Vxqf+j9LREMQ2ZYyYOwNVUkTMRFmCmIhAZzgGUdv367FEsciDoz1fBzLCq5eTr/ueRUjQ6K0R2lrLdJ0njnQ9J9R+vN4ICNMhiSbSGSYtss/IIeQXz4ytQt9EfaSc/LT5hwLUt4pbjw==")));

        // too short signature
        UNIT_ASSERT_VALUES_EQUAL(TRsaPublicEvp::TResult({false, "wrong signature length"}),
                                 rsa1.VerifyWithSha256(
                                     "0#test",
                                     Base64ToBin("U6/pHMzAvp6DEz0BjFghXVafO6UTqVfW4imX1inVqCHHgVMEHnX/2d0N1S7hpSdETWRP5o96vnLLbQWZs9Zo+l9fLIaSN5ilOJxNvRFkkIH8c9eg/c4Q0M8+63Ke8Lnsm+8VUAzwrXSOOb932OZDz6o/u4fk5e0oWQ3MxfFsFwuosApwzuFXFTnPREPC0PFYwWSB5G4UcY2Vxqf+j9LREMQ2ZYyYOwNVUkTMRFmCmIhAZzgGUdv367FEsciDoz1fBzLCq5eTr/ueRUjQ6K0R2lrLdJ0njnQ9J9R+vN4ICNMhiSbSGSYtss/IIeQXz4ytQt9EfaSc/LT5hwLUt4pb")));

        // too long signature
        UNIT_ASSERT_VALUES_EQUAL(TRsaPublicEvp::TResult({false, "wrong signature length"}),
                                 rsa1.VerifyWithSha256(
                                     "0#test",
                                     Base64ToBin("U6/pHMzAvp6DEz0BjFghXVafO6UTqVfW4imX1inVqCHHgVMEHnX/2d0N1S7hpSdETWRP5o96vnLLbQWZs9Zo+l9fLIaSN5ilOJxNvRFkkIH8c9eg/c4Q0M8+63Ke8Lnsm+8VUAzwrXSOOb932OZDz6o/u4fk5e0oWQ3MxfFsFwuosApwzuFXFTnPREPC0PFYwWSB5G4UcY2Vxqf+j9LREMQ2ZYyYOwNVUkTMRFmCmIhAZzgGUdv367FEsciDoz1fBzLCq5eTr/ueRUjQ6K0R2lrLdJ0njnQ9J9R+vN4ICNMhiSbSGSYtss/IIeQXz4ytQt9EfaSc/LT5hwLUt4pbjwqqqqqq")));

        // corrupted signature
        UNIT_ASSERT_VALUES_EQUAL(TRsaPublicEvp::TResult({false, "invalid padding"}),
                                 rsa1.VerifyWithSha256(
                                     "0#test",
                                     Base64ToBin("U6/pHMzAvp6DEz0BjFghXVafO6UTqVfW4imX1inVqCHHgVMEHnX/3d0N1S7hpSdETWRP5o96vnLLbQWZs9Zo+l9fLIaSN5ilOJxNvRFkkIH8c9eg/c4Q0M8+63Ke8Lnsm+8VUAzwrXSOOb932OZDz6o/u4fk5e0oWQ3MxfFsFwuosApwzuFXFTnPREPC0PFYwWSB5G4UcY2Vxqf+j9LREMQ2ZYyYOwNVUkTMRFmCmIhAZzgGUdv367FEsciDoz1fBzLCq5eTr/ueRUjQ6K0R2lrLdJ0njnQ9J9R+vN4ICNMhiSbSGSYtss/IIeQXz4ytQt9EfaSc/LT5hwLUt4pbjw==")));

        // strange signature
        UNIT_ASSERT_VALUES_EQUAL(TRsaPublicEvp::TResult({false, "wrong signature length"}),
                                 rsa1.VerifyWithSha256(
                                     "1#test",
                                     "\nkek"));
    }

    Y_UNIT_TEST(RsaSignWithSha256) {
        TRsaPrivateEvp priv = TRsaPrivateEvp::FromPem(
            R"(-----BEGIN RSA PRIVATE KEY-----
MIIEowIBAAKCAQEAo6kuQneY+KPoD+KhxO4EDvWZVKXxT3CHcNTJNWYmNfJ4ufGa
iy1xANq3WTRTXqmEXyteWYsW7dcIaYbjjdORllebSsxnsxTlA8SwOKIZfLuy2Es6
eXk0ryTShppB43mLtmWXYcHdphCCAKdvU1VK8yFJ9BXl6e7doGvKJEby6KK73LFS
FeEp5Ah+ySSLwGOTDZbDvX3XNNBUafkm/IVguPI6Xv+LMdqYIZ4SitSEWNMeJieg
nm5Q4y0IXmcsfkgFYR0on5knnM73d9UL1iFkX+iZQjOcLwOIrVnprc+OzZopU4zV
UfX8lIVGVljTFy39NRpqDDbFJCY4BT7CMJIGtQIDAQABAoIBAQCJhbonhBP4qtJv
tsK0lkS2xU3IwYBafnZxK3y/8EwffNZReWGhndedHix/OubrXoqq5ehsWeI0jUK2
WfCQ1r05lfgaDFT/OImQdxCVJjFAjQClm+FQ/bZ1jf2RucGwAgySgh+It70mtCxp
nEiBv+QxFUHtZxFBV8TH7ot3nF61d+Eg+ft6JvtzKlBmLC/DBW2og+8/yQJigQLT
tc8CQ2smmrINvUw3xOJlcB8eyNdyyoZxIL1gFochmHHAkPYYMn0VpS5G2UmwJf1C
KJYrGDlLkrGNqEWQAampG9syaJbC035CMOjmpktE7bdwE8yvwOBNrbpHpBOj99hB
GxpXqxhBAoGBANSfRQLt4VgCKiZPYuMHU+0ZWZLBrWJOIDvYdL1JygnOp+fawCUB
/JMN1tOqkODlzeMLdED8mDroCZFJtih0XSx596olIdAE0YuxCmcipFxCm4UexDYN
wMzuWmWPUy2vzlNdeeJRBm+E0Euiwql7QnknAqvQziDawzcaDbpsM0XpAoGBAMUM
yLGmmRGDFYbgPym+n+kgg7PtcuXFqFOaSBbEK+UUV7BQihDOKEH2qCmLPa/oyfbk
VYi0H9qbr4Dk31KFU5MuocCU7Z6P8wUnF+Iqsi6rzzwruEZEVWKedb92Vi5IHb1V
ke0VVnG0AOYlb17uUHcXk2f3OvVXBzhTcdW1AR7tAoGAKCbLiXoSi+QcwY9MRUMk
/ilDWQtqCp5mR+bgZL6CgMedvZL5pgNyBvMeFmqnYVEESFPpaDxLIxSeeelJlOok
jjmaI29Pq1C9oXuVU1SfZSzO4ZylwO9n8Usxluwx8Pa6J9QHKE3UwUp7a00ZJZg5
PVe4kWmlA19MbiNlZEb6h8ECgYA5BN3+KgsY+il8gd9FIRf8yZ/SrN8Whf+TUY2S
qOlavGwzZNrOyhj0HX/DpbsP7/ihLKljiVu+UlaSoafNpS7t5AyCQZYQz+6uiwVq
OJie0LCC5NPJ4XQjuV6xLHj7o3qXku9K/2WoOUIZVrPHcIZwL65D41J4M9qu/dZu
1JWicQKBgDPSSfvlfIyhxjdhz3t45sf1HoksWhq/kwi0b65AbeAK3VmPezquKhsb
m7ot3K3GR26YeeVNo/uHJ2mlj2UCQkRVA4Sbnw14uqivqoQROarS4IacxEuGzEwQ
rjmvfrwOKR3fmy3hcMqrl0/WcVVkhjobX04i+ToNm0bRkqo5YlLE
-----END RSA PRIVATE KEY-----)");

        TRsaPublicEvp pub = TRsaPublicEvp::FromPem(
            R"(-----BEGIN PUBLIC KEY-----
MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAo6kuQneY+KPoD+KhxO4E
DvWZVKXxT3CHcNTJNWYmNfJ4ufGaiy1xANq3WTRTXqmEXyteWYsW7dcIaYbjjdOR
llebSsxnsxTlA8SwOKIZfLuy2Es6eXk0ryTShppB43mLtmWXYcHdphCCAKdvU1VK
8yFJ9BXl6e7doGvKJEby6KK73LFSFeEp5Ah+ySSLwGOTDZbDvX3XNNBUafkm/IVg
uPI6Xv+LMdqYIZ4SitSEWNMeJiegnm5Q4y0IXmcsfkgFYR0on5knnM73d9UL1iFk
X+iZQjOcLwOIrVnprc+OzZopU4zVUfX8lIVGVljTFy39NRpqDDbFJCY4BT7CMJIG
tQIDAQAB
-----END PUBLIC KEY-----)");

        const TString data = "avada kedavra, madman";
        UNIT_ASSERT_VALUES_EQUAL(TRsaPublicEvp::TResult({true, ""}),
                                 pub.VerifyWithSha256(data, priv.SignWithSha256(data)));
    }

    Y_UNIT_TEST(RsaDecrypt) {
        TRsaPrivateEvp priv = TRsaPrivateEvp::FromPem(
            R"(-----BEGIN RSA PRIVATE KEY-----
MIIEpAIBAAKCAQEA1Jeo8gMXIzb4LkgXlWnk1nrgcxm+CngNzGBRNMcAnSyD25rC
lU56s2I2tXWU4m9WusGcqVo+2qgt6FDOudhdb10AgvdIVGN26kpLzL/C91O3CeWa
A6LA9d3g33kP7IK9X6xT+aLyrjPB0HkCUwpAb8PyWqFXNu+aL9au7F/4QtX9Azdn
YQ1QjZWTN+oW+aqQvZRVMkE1R9+MfdIwwTAZ5VKOmNyDeo2I1pTLwTWczuwJJstd
1adpIHI2BXVvjHcQBq+sf0ZEAVzNrLE9eI8m4vLBufQOTQzeQdl/icob6i9m8ifl
8KTPI7jlGZs6qCw+jWmJsAIvm95avqel0cEAuwIDAQABAoIBACyZiJsFJPSBUr26
6B+zcIvCiZX8H49oslRfHIo6+Zj/vsXyiMH0De/WOe4Fte9vNj3F4ozw3uDWiZME
aOo6RxfW5gWTxTvXBhqO7aoNzORATtUnsxDyMxMhskyPxFR09S++InvrwIflWdLX
oDI1o2EPNafn0zk2OR2yJ051r8qtYddwrLjGEH2pK8EO4Yk4vIPShepoG973kNc+
6jdovZunnVTS320O/dVFRZkdLwSEMpinvS6/ImwLsFr4aGXJFVSPy6uEGZNNfLVV
OfnZsiICKlniTxyE1z50H3Vfvl3QGdQ+iIJHqZBAEJx6R9vpJPt9sY/2NhYyP3Zj
g9LdGCECgYEA+2vjIWaquouDPDqPhsKkhyJQts0b90uZ9nKv6T0fsei/9iHp+Kyy
qxoroMBmZj5Bwgua/guzxMszK+poK6aLyrF0zVcr7J35xUsLdUVejB+i3bHElN9b
r3v+0lf3sK4wCD0TxsWvMyYAm6Cd2RpAbKDxJx+GXYUs3G7zg3e56YsCgYEA2HbB
BuyJQPpQR35zJhAkE2PVQ/+hnY7fuOJX0g7Mjmjz644zDeMOCN9gWEKT44vR7233
FbrdLwIIkeK7bO6WNvJW2kEAYk92eKkcTuku7dlL0zLxNe+VOHfZoq8XJf8fH3gO
PhkssFgIEwTcmrs9tfIBnCk5VFMgSTcWy48YS5ECgYAnUI+LwYog8X/sAxw+bmFb
4DnO2/Prj57bssNfLXevUj3yNwtnH1ow9z6rPGrGwiV2OmpaH2pU85Fk5Jq1eK2T
XHxvP5pWvb045Ks+A0LtlmTZVdrvjJ3BqxoFuku5DQPlxec5xd/Hl8GwlEUalchN
ND2BaKSSKykArzjU7fvcUQKBgQDJMBbYSKCJNB3JVU3Q4s+k9fUR7lZYCqMFQ9o7
8mVNN17+YHwzPAWfWX5Cih7KLvpAfwRXvgY12r38Wa+530beav8ue5vPw+to2kTl
UvW7O2uuMHrgln0qrroo6nOpbSWQKhWSwfM3M5rNkoc3iTAiG3xDmE0gguJPKlSV
fbonAQKBgQC0twv36cvY9JAJmjOWu1HspFI5ltfGX5gqIz/OniPQcIh0SyW0ubSi
J2UhZeefvESKlbkGcLj+OSIS3+j/jxR+WSKJfRHqQWUmF/1yZ/0zY/iMP1VucogB
1L3sQfs+Htbzrj9GOdNn6+AIByTfUqD5k166RJu+Xne5yM0xOL0KWw==
-----END RSA PRIVATE KEY-----)");

        UNIT_ASSERT_EXCEPTION_CONTAINS(
            priv.DecryptWithModes(
                "qqq",
                {
                    {"qwe", "oaep"},
                }),
            yexception,
            "Failed to init set control 'qwe'");
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            priv.DecryptWithModes(
                "qqq",
                {
                    {"rsa_padding_mode", "oaep"},
                    {"rsa_oaep_md", "sha256"},
                    {"rsa_mgf1_md", "sha256"},
                }),
            yexception,
            "Failed to decrypt:");

        TString val;
        UNIT_ASSERT_NO_EXCEPTION(
            val = priv.DecryptWithModes(
                Hex2bin("5e7df1024356e0d8691d7480fe11c58f9598f8bd71b1ca9a10c4e034867484cb16914ac7bfb4ce2e21df46d9f3d084904bb88ca7452db1ad954444d3d6a8c41ba262d087c68328764e50bb18c21feffb77643ed6e2447130d0fd617aab39b801498fb60b96d09b82285c70fdd94d7b12bf833c6557b866d9aa2f60e24632be39b993c07de96fc07217084e0dc33188c46d383e640ca2ec0b831e8fca76948eebf6bd1fdde6fd63c78845b1aef6725cbd26ddb06f1fd4d300aeeae02f43f5266a88bfafd5e3d02564ea1db2bf5784a5ce7d291aa21072176009505edc11829374cbe1e6254df2b431d9330b440ed425cfe167fcea0d6bbf45bd5476116ce6eaf5"),
                {
                    {"rsa_padding_mode", "oaep"},
                    {"rsa_oaep_md", "sha256"},
                    {"rsa_mgf1_md", "sha256"},
                }));
        UNIT_ASSERT_VALUES_EQUAL("5d8a36c1533b1bc9b964745b39542b48a5d99f4cfba50f1c7b2d546646798d1a",
                                 Bin2hex(val));
    }
}
