#include "common.h"

#include <passport/infra/libs/cpp/utils/string/format.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport::NUtils;

Y_UNIT_TEST_SUITE(PasspUtilsFormat) {
#ifdef _unix_
    Y_UNIT_TEST(timestamp) {
        NPassport::SetTZ();

        UNIT_ASSERT_VALUES_EQUAL("1970-04-26 22:10:00",
                                 FormatTimestamp("10005000"));
        UNIT_ASSERT_VALUES_EQUAL("1973-03-04 02:40:00",
                                 FormatTimestamp("100050000"));
        UNIT_ASSERT_VALUES_EQUAL("2001-09-15 00:40:00",
                                 FormatTimestamp("1000500000"));
        UNIT_ASSERT_VALUES_EQUAL("2287-01-17 17:40:00",
                                 FormatTimestamp("10005000000"));
        UNIT_ASSERT_VALUES_EQUAL("",
                                 FormatTimestamp(""));
    }
#endif // _unix_

    Y_UNIT_TEST(escapeEOL) {
        TString str = "a1bc\\\n\r\v\tdef";
        str[1] = '\0';
        UNIT_ASSERT_VALUES_EQUAL("a\\0bc\\\\\\n\\r\\v\\tdef",
                                 EscapeEol(str));
        str.clear();
        UNIT_ASSERT_VALUES_EQUAL("", EscapeEol(str));
    }

    Y_UNIT_TEST(eolToSpace) {
        TString str = "abc\\\n\v\r\tdef";
        str.push_back('\0');
        TString res("abc\\ \v \tdef");
        res.push_back('\0');
        UNIT_ASSERT_VALUES_EQUAL(res, EolToSpace(str));
        UNIT_ASSERT_VALUES_EQUAL("", EolToSpace(""));
    }

    Y_UNIT_TEST(unquote) {
        UNIT_ASSERT_VALUES_EQUAL("\'olo\'lolo\'", Unquote("\'olo\'lolo\'"));
        UNIT_ASSERT_VALUES_EQUAL("olo\"lolo", Unquote("\"olo\"lolo\""));
        UNIT_ASSERT_VALUES_EQUAL("olololo", Unquote("\"olololo\""));
        UNIT_ASSERT_VALUES_EQUAL("olololo", Unquote("olololo"));
        UNIT_ASSERT_VALUES_EQUAL("\"", Unquote("\""));
        UNIT_ASSERT_VALUES_EQUAL("olololo\"", Unquote("olololo\""));
        UNIT_ASSERT_VALUES_EQUAL("\"olololo", Unquote("\"olololo"));
        UNIT_ASSERT_VALUES_EQUAL("", Unquote(""));

        TString str = "\"olololo\"";
        str[4] = '\0';
        TString res = "olololo";
        res[3] = '\0';
        UNIT_ASSERT_VALUES_EQUAL(res, Unquote(str));
    }

    Y_UNIT_TEST(toLower) {
        UNIT_ASSERT_VALUES_EQUAL("olololo", TolowerCopy("olololo"));
        UNIT_ASSERT_VALUES_EQUAL("olololo", TolowerCopy("oLolOlO"));
        UNIT_ASSERT_VALUES_EQUAL("olololo", TolowerCopy("OLOLOLO"));
        UNIT_ASSERT_VALUES_EQUAL("0lo1olo", TolowerCopy("0Lo1OlO"));
        UNIT_ASSERT_VALUES_EQUAL("", TolowerCopy(""));

        TString str = "olololo";
        Tolower(str);
        UNIT_ASSERT_VALUES_EQUAL("olololo", str);
        str = "oLolOlO";
        Tolower(str);
        UNIT_ASSERT_VALUES_EQUAL("olololo", str);
        str = "OLOLOLO";
        Tolower(str);
        UNIT_ASSERT_VALUES_EQUAL("olololo", TolowerCopy("OLOLOLO"));
        str = "0Lo1OlO";
        Tolower(str);
        UNIT_ASSERT_VALUES_EQUAL("0lo1olo", TolowerCopy("0Lo1OlO"));

        str = "oLolOlO";
        str[4] = '\0';
        TString res = "olololo";
        res[4] = '\0';
        Tolower(str);
        UNIT_ASSERT_VALUES_EQUAL(res, str);
    }

    Y_UNIT_TEST(toUpper) {
        UNIT_ASSERT_VALUES_EQUAL("OLOLOLO", ToupperCopy("olololo"));
        UNIT_ASSERT_VALUES_EQUAL("OLOLOLO", ToupperCopy("oLolOlO"));
        UNIT_ASSERT_VALUES_EQUAL("OLOLOLO", ToupperCopy("OLOLOLO"));
        UNIT_ASSERT_VALUES_EQUAL("0LO1OLO", ToupperCopy("0Lo1OlO"));
        UNIT_ASSERT_VALUES_EQUAL("", ToupperCopy(""));

        TString str = "OLOLOLO";
        Toupper(str);
        UNIT_ASSERT_VALUES_EQUAL("OLOLOLO", str);
        str = "oLolOlO";
        Toupper(str);
        UNIT_ASSERT_VALUES_EQUAL("OLOLOLO", str);
        str = "OLOLOLO";
        Toupper(str);
        UNIT_ASSERT_VALUES_EQUAL("OLOLOLO", ToupperCopy("OLOLOLO"));
        str = "0Lo1OlO";
        Toupper(str);
        UNIT_ASSERT_VALUES_EQUAL("0LO1OLO", ToupperCopy("0Lo1OlO"));

        str = "oLolOlO";
        str[4] = '\0';
        TString res = "OLOLOLO";
        res[4] = '\0';
        Toupper(str);
        UNIT_ASSERT_VALUES_EQUAL(res, str);
    }

    Y_UNIT_TEST(removeSpaces) {
        UNIT_ASSERT_VALUES_EQUAL("magiccruelfuel", RemoveSpaces("magic cruel fuel"));
        UNIT_ASSERT_VALUES_EQUAL("", RemoveSpaces(""));
        UNIT_ASSERT_VALUES_EQUAL("", RemoveSpaces("   "));
        UNIT_ASSERT_VALUES_EQUAL("mybatle", RemoveSpaces("my batle"));
        UNIT_ASSERT_VALUES_EQUAL("mybatle", RemoveSpaces("my ba\ttle"));
        UNIT_ASSERT_VALUES_EQUAL("m\nybatle", RemoveSpaces("m\ny ba\ttle"));
        UNIT_ASSERT_VALUES_EQUAL("m\nybatl\ve", RemoveSpaces("m\ny ba\ttl\ve"));
        UNIT_ASSERT_VALUES_EQUAL("m\nybat\fl\ve", RemoveSpaces("m\ny ba\tt\fl\ve"));

        TString str = "m\ny ba\ttle";
        str[2] = '\0';
        TString res = "m\nybatle";
        res[2] = '\0';
        UNIT_ASSERT_VALUES_EQUAL(res, RemoveSpaces(str));
    }

    Y_UNIT_TEST(trim) {
        UNIT_ASSERT_VALUES_EQUAL("magic cruel fuel", TrimCopy("magic cruel fuel"));
        UNIT_ASSERT_VALUES_EQUAL("magiccruelfuel", TrimCopy("magiccruelfuel"));
        UNIT_ASSERT_VALUES_EQUAL("", TrimCopy("                "));
        UNIT_ASSERT_VALUES_EQUAL("", TrimCopy(""));

        UNIT_ASSERT_VALUES_EQUAL("magic cruel", TrimCopy("magic cruel "));
        UNIT_ASSERT_VALUES_EQUAL("magic cruel", TrimCopy("magic cruel \v  "));
        UNIT_ASSERT_VALUES_EQUAL("magic cruel", TrimCopy(" magic cruel  \n       "));
        UNIT_ASSERT_VALUES_EQUAL("magic cruel", TrimCopy("    \f    magic cruel       \t  "));
        UNIT_ASSERT_VALUES_EQUAL("magic cruel", TrimCopy("      \v   magic cruel"));
        UNIT_ASSERT_VALUES_EQUAL("magic cruel", TrimCopy(" magic cruel"));

        TString str = "magic cruel fuel";
        UNIT_ASSERT_VALUES_EQUAL("magic cruel fuel", Trim(str));
        str = "magiccruelfuel";
        UNIT_ASSERT_VALUES_EQUAL("magiccruelfuel", Trim(str));
        str = "          ";
        UNIT_ASSERT_VALUES_EQUAL("", Trim(str));
        str = "";
        UNIT_ASSERT_VALUES_EQUAL("", Trim(str));

        str = "magic cruel ";
        UNIT_ASSERT_VALUES_EQUAL("magic cruel", Trim(str));
        str = "magic cruel       \n   ";
        UNIT_ASSERT_VALUES_EQUAL("magic cruel", Trim(str));
        str = " magic cruel     \t     ";
        UNIT_ASSERT_VALUES_EQUAL("magic cruel", Trim(str));
        str = "     \f    magic cruel  \v        ";
        UNIT_ASSERT_VALUES_EQUAL("magic cruel", Trim(str));
        str = "     \t    magic cruel";
        UNIT_ASSERT_VALUES_EQUAL("magic cruel", Trim(str));
        str = " magic cruel";
        UNIT_ASSERT_VALUES_EQUAL("magic cruel", Trim(str));

        str = "     \f    magic cruel  \v        ";
        str[3] = '\0';
        TString res = "  \f    magic cruel";
        res[0] = '\0';
        UNIT_ASSERT_VALUES_EQUAL(res, TrimCopy(str));

        TStringBuf strbuf = "magic cruel fuel";
        UNIT_ASSERT_VALUES_EQUAL("magic cruel fuel", Trim(strbuf));
        strbuf = "magiccruelfuel";
        UNIT_ASSERT_VALUES_EQUAL("magiccruelfuel", Trim(strbuf));
        strbuf = "          ";
        UNIT_ASSERT_VALUES_EQUAL("", Trim(strbuf));
        strbuf = "";
        UNIT_ASSERT_VALUES_EQUAL("", Trim(strbuf));

        strbuf = "magic cruel ";
        UNIT_ASSERT_VALUES_EQUAL("magic cruel", Trim(strbuf));
        strbuf = "magic cruel       \n   ";
        UNIT_ASSERT_VALUES_EQUAL("magic cruel", Trim(strbuf));
        strbuf = " magic cruel     \t     ";
        UNIT_ASSERT_VALUES_EQUAL("magic cruel", Trim(strbuf));
        strbuf = "     \f    magic cruel  \v        ";
        UNIT_ASSERT_VALUES_EQUAL("magic cruel", Trim(strbuf));
        strbuf = "     \t    magic cruel";
        UNIT_ASSERT_VALUES_EQUAL("magic cruel", Trim(strbuf));
        strbuf = " magic cruel";
        UNIT_ASSERT_VALUES_EQUAL("magic cruel", Trim(strbuf));
    }

    Y_UNIT_TEST(escapeUnprintable) {
        TString str;
        EscapeUnprintable(str);
        UNIT_ASSERT_VALUES_EQUAL("", str);

        str.assign("abc de");
        EscapeUnprintable(str);
        UNIT_ASSERT_VALUES_EQUAL("abc de", str);

        str.assign("abc\r\n\t\v\\de");
        EscapeUnprintable(str);
        UNIT_ASSERT_VALUES_EQUAL("abc\\r\\n\\t\\v\\\\de", str);

        str.assign("abc de")[2] = '\0';
        EscapeUnprintable(str);
        UNIT_ASSERT_VALUES_EQUAL("ab<00> de", str);

        str.assign("\nabc de \t")[2] = '\0';
        EscapeUnprintable(str);
        UNIT_ASSERT_VALUES_EQUAL("\\na<00>c de \\t", str);

        const TString req = "attributes=13&dbfields=account_info.country.uid,account_info.reg_date.uid,"
                            "accounts.ena.uid,subscription.born_date.2,subscription.login.-,subscription.suid.-,"
                            "userphones.confirmed.uid&login=fbl@vkupon.ru&method=userinfo&sid=smtp&userip=94.100.179.26";
        str = req;
        EscapeUnprintable(str);
        UNIT_ASSERT_VALUES_EQUAL(req, str);

        TString req2 = "attributes=107&authtype=imap&dbfields=hosts.db_id.2,"
                       "subscription.login.2,subscription.login.8,subscription.login_rule.8,"
                       "subscription.suid.2,userinfo.country.uid,userinfo.lang.uid&"
                       "full_info=yes&login=cherepahaut" +
                       TString(1, 0xEE) +
                       "&method=login&password=*&sid=pop&userip=::ffff:217.118.81.214";
        str = req2;
        EscapeUnprintable(str);
        UNIT_ASSERT_VALUES_EQUAL("attributes=107&authtype=imap&dbfields=hosts.db_id.2,"
                                 "subscription.login.2,subscription.login.8,subscription.login_rule.8,"
                                 "subscription.suid.2,userinfo.country.uid,userinfo.lang.uid&full_info=yes&"
                                 "login=cherepahaut<EE>&method=login&password=*&sid=pop&userip=::ffff:217.118.81.214",
                                 str);
    }
}
