#include <passport/infra/libs/cpp/utils/ipaddr.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/generic/string.h>

#include <string>

using namespace NPassport::NUtils;

Y_UNIT_TEST_SUITE(PasspUtilsIp) {
    Y_UNIT_TEST(normalize) {
        UNIT_ASSERT_VALUES_EQUAL("1.2.3.4", TIpAddr::Normalize("1.2.3.4"));
        UNIT_ASSERT_VALUES_EQUAL("", TIpAddr::Normalize(" 1.2.03"));
        UNIT_ASSERT_VALUES_EQUAL("211.127.255.0", TIpAddr::Normalize("211.127.255.0"));
        UNIT_ASSERT_VALUES_EQUAL("", TIpAddr::Normalize("1.3.555.6"));
        UNIT_ASSERT_VALUES_EQUAL("", TIpAddr::Normalize("hkljhj"));
        UNIT_ASSERT_VALUES_EQUAL("::", TIpAddr::Normalize("::"));
        UNIT_ASSERT_VALUES_EQUAL("::1.2.3.45", TIpAddr::Normalize("::1.2.3.45"));
        UNIT_ASSERT_VALUES_EQUAL("6.5.4.1", TIpAddr::Normalize("::ffff:6.5.4.1"));
        UNIT_ASSERT_VALUES_EQUAL("", TIpAddr::Normalize("aa:bb:cc:dd:ee:ff"));
        UNIT_ASSERT_VALUES_EQUAL("aa:bb:cc::dd:ee", TIpAddr::Normalize("aa:bb:cc:00:00:00:dd:ee"));
        UNIT_ASSERT_VALUES_EQUAL("12:34::102:304", TIpAddr::Normalize("12:34::1.2.3.4"));
        UNIT_ASSERT_VALUES_EQUAL("", TIpAddr::Normalize("   "));
        UNIT_ASSERT_VALUES_EQUAL("", TIpAddr::Normalize(""));
        UNIT_ASSERT_VALUES_EQUAL("0.0.0.0", TIpAddr::Normalize("0.0.0.0"));
        UNIT_ASSERT_VALUES_EQUAL("::", TIpAddr::Normalize("::0"));
        UNIT_ASSERT_VALUES_EQUAL("", TIpAddr::Normalize("0"));
    }

    // binary powers from 1 to 2^127
    const char* const binaries[] = {"AQAAAA", "AgAAAA", "BAAAAA", "CAAAAA", "EAAAAA", "IAAAAA", "QAAAAA", "gAAAAA", "AAEAAA", "AAIAAA", "AAQAAA", "AAgAAA", "ABAAAA", "ACAAAA", "AEAAAA", "AIAAAA", "AAABAA", "AAACAA", "AAAEAA", "AAAIAA", "AAAQAA", "AAAgAA", "AABAAA", "AACAAA", "AAAAAQ", "AAAAAg", "AAAABA", "AAAACA", "AAAAEA", "AAAAIA", "AAAAQA", "AAAAgA", "AAAAAAEAAAAAAAAAAAAAAA", "AAAAAAIAAAAAAAAAAAAAAA", "AAAAAAQAAAAAAAAAAAAAAA", "AAAAAAgAAAAAAAAAAAAAAA", "AAAAABAAAAAAAAAAAAAAAA", "AAAAACAAAAAAAAAAAAAAAA", "AAAAAEAAAAAAAAAAAAAAAA", "AAAAAIAAAAAAAAAAAAAAAA", "AAAAAAABAAAAAAAAAAAAAA", "AAAAAAACAAAAAAAAAAAAAA", "AAAAAAAEAAAAAAAAAAAAAA", "AAAAAAAIAAAAAAAAAAAAAA", "AAAAAAAQAAAAAAAAAAAAAA", "AAAAAAAgAAAAAAAAAAAAAA", "AAAAAABAAAAAAAAAAAAAAA", "AAAAAACAAAAAAAAAAAAAAA", "AAAAAAAAAQAAAAAAAAAAAA", "AAAAAAAAAgAAAAAAAAAAAA", "AAAAAAAABAAAAAAAAAAAAA", "AAAAAAAACAAAAAAAAAAAAA", "AAAAAAAAEAAAAAAAAAAAAA", "AAAAAAAAIAAAAAAAAAAAAA", "AAAAAAAAQAAAAAAAAAAAAA", "AAAAAAAAgAAAAAAAAAAAAA", "AAAAAAAAAAEAAAAAAAAAAA", "AAAAAAAAAAIAAAAAAAAAAA", "AAAAAAAAAAQAAAAAAAAAAA", "AAAAAAAAAAgAAAAAAAAAAA", "AAAAAAAAABAAAAAAAAAAAA", "AAAAAAAAACAAAAAAAAAAAA", "AAAAAAAAAEAAAAAAAAAAAA", "AAAAAAAAAIAAAAAAAAAAAA", "AAAAAAAAAAABAAAAAAAAAA", "AAAAAAAAAAACAAAAAAAAAA", "AAAAAAAAAAAEAAAAAAAAAA", "AAAAAAAAAAAIAAAAAAAAAA", "AAAAAAAAAAAQAAAAAAAAAA", "AAAAAAAAAAAgAAAAAAAAAA", "AAAAAAAAAABAAAAAAAAAAA", "AAAAAAAAAACAAAAAAAAAAA", "AAAAAAAAAAAAAQAAAAAAAA", "AAAAAAAAAAAAAgAAAAAAAA", "AAAAAAAAAAAABAAAAAAAAA", "AAAAAAAAAAAACAAAAAAAAA", "AAAAAAAAAAAAEAAAAAAAAA", "AAAAAAAAAAAAIAAAAAAAAA", "AAAAAAAAAAAAQAAAAAAAAA", "AAAAAAAAAAAAgAAAAAAAAA", "AAAAAAAAAAAAAAEAAAAAAA", "AAAAAAAAAAAAAAIAAAAAAA", "AAAAAAAAAAAAAAQAAAAAAA", "AAAAAAAAAAAAAAgAAAAAAA", "AAAAAAAAAAAAABAAAAAAAA", "AAAAAAAAAAAAACAAAAAAAA", "AAAAAAAAAAAAAEAAAAAAAA", "AAAAAAAAAAAAAIAAAAAAAA", "AAAAAAAAAAAAAAABAAAAAA", "AAAAAAAAAAAAAAACAAAAAA", "AAAAAAAAAAAAAAAEAAAAAA", "AAAAAAAAAAAAAAAIAAAAAA", "AAAAAAAAAAAAAAAQAAAAAA", "AAAAAAAAAAAAAAAgAAAAAA", "AAAAAAAAAAAAAABAAAAAAA", "AAAAAAAAAAAAAACAAAAAAA", "AAAAAAAAAAAAAAAAAQAAAA", "AAAAAAAAAAAAAAAAAgAAAA", "AAAAAAAAAAAAAAAABAAAAA", "AAAAAAAAAAAAAAAACAAAAA", "AAAAAAAAAAAAAAAAEAAAAA", "AAAAAAAAAAAAAAAAIAAAAA", "AAAAAAAAAAAAAAAAQAAAAA", "AAAAAAAAAAAAAAAAgAAAAA", "AAAAAAAAAAAAAAAAAAEAAA", "AAAAAAAAAAAAAAAAAAIAAA", "AAAAAAAAAAAAAAAAAAQAAA", "AAAAAAAAAAAAAAAAAAgAAA", "AAAAAAAAAAAAAAAAABAAAA", "AAAAAAAAAAAAAAAAACAAAA", "AAAAAAAAAAAAAAAAAEAAAA", "AAAAAAAAAAAAAAAAAIAAAA", "AAAAAAAAAAAAAAAAAAABAA", "AAAAAAAAAAAAAAAAAAACAA", "AAAAAAAAAAAAAAAAAAAEAA", "AAAAAAAAAAAAAAAAAAAIAA", "AAAAAAAAAAAAAAAAAAAQAA", "AAAAAAAAAAAAAAAAAAAgAA", "AAAAAAAAAAAAAAAAAABAAA", "AAAAAAAAAAAAAAAAAACAAA", "AAAAAAAAAAAAAAAAAAAAAQ", "AAAAAAAAAAAAAAAAAAAAAg", "AAAAAAAAAAAAAAAAAAAABA", "AAAAAAAAAAAAAAAAAAAACA", "AAAAAAAAAAAAAAAAAAAAEA", "AAAAAAAAAAAAAAAAAAAAIA", "AAAAAAAAAAAAAAAAAAAAQA", "AAAAAAAAAAAAAAAAAAAAgA", nullptr};

    Y_UNIT_TEST(arithmetic) {
        TIpAddr addr;

        UNIT_ASSERT(addr.Parse("::ffff:ffff:ffff:ffff"));
        UNIT_ASSERT_VALUES_EQUAL("0:0:0:1::", (TString)addr.Next());
        addr >>= 53;
        UNIT_ASSERT_VALUES_EQUAL("::7ff", (TString)addr);
        addr <<= 2;
        UNIT_ASSERT_VALUES_EQUAL("::1ffc", (TString)addr);
        addr <<= 300;
        UNIT_ASSERT_VALUES_EQUAL("::", (TString)addr);

        UNIT_ASSERT(addr.Parse("::"));
        UNIT_ASSERT_VALUES_EQUAL("ffff:ffff:ffff:ffff:ffff:ffff:ffff:ffff", addr.Prev().ToString());

        UNIT_ASSERT(addr.Parse("::12:34:56"));
        UNIT_ASSERT_VALUES_EQUAL("::12:30:0", (TString)addr.GetRangeStart(109));
        UNIT_ASSERT_VALUES_EQUAL("::13:ffff:ffff", (TString)addr.GetRangeEnd(95));

        UNIT_ASSERT(addr.Parse("127.0.0.1"));
        UNIT_ASSERT_VALUES_EQUAL("AQAAfw", addr.ToBase64String());

        UNIT_ASSERT(addr.ParseBase64("AQAAfw"));
        UNIT_ASSERT_VALUES_EQUAL("127.0.0.1", addr.ToString());

        UNIT_ASSERT_VALUES_EQUAL("127.0.0.0", (TString)addr.GetRangeStart(24));
        UNIT_ASSERT_VALUES_EQUAL("127.0.0.255", (TString)addr.GetRangeEnd(24));

        // don't check 32 first addressess since they are treated as IpV4 and stored with 0xffff prefix
        int i = 0;
        while (i < 32) { // check base64 encode/decode for 32 bit addrs
            TIpAddr parsed;
            UNIT_ASSERT(addr.ParseBase64(binaries[i]));
            TString base64 = addr.ToBase64String();
            UNIT_ASSERT(parsed.ParseBase64(base64));
            UNIT_ASSERT_EQUAL(addr, parsed);
            ++i;
        }
        UNIT_ASSERT(addr.Parse("::1"));
        addr <<= 32;
        while (binaries[i]) {
            TIpAddr parsed;

            UNIT_ASSERT(parsed.ParseBase64(binaries[i]));
            UNIT_ASSERT_EQUAL(addr, parsed);
            TString base64 = parsed.ToBase64String();
            UNIT_ASSERT(parsed.ParseBase64(base64));
            UNIT_ASSERT_EQUAL(addr, parsed);

            addr <<= 1;
            ++i;
        }

        UNIT_ASSERT(!addr.ParseBase64("abc"));
        UNIT_ASSERT(!addr.ParseBase64("abcdef0123"));
        UNIT_ASSERT(!addr.ParseBase64("ab def"));
        UNIT_ASSERT(!addr.ParseBase64("abc+ef"));
        UNIT_ASSERT(!addr.ParseBase64("abc/ef"));
        UNIT_ASSERT(!addr.ParseBase64("abc/efg"));

        UNIT_ASSERT(addr.Parse("0.0.0.0"));
        UNIT_ASSERT_VALUES_EQUAL("0.0.0.0", addr.ToString());
        UNIT_ASSERT_VALUES_EQUAL("AAAAAA", addr.ToBase64String());

        UNIT_ASSERT(addr.Parse("0.0.0.0"));
        UNIT_ASSERT_VALUES_EQUAL("0.0.0.0", addr.ToString());
        UNIT_ASSERT_VALUES_EQUAL("AAAAAA", addr.ToBase64String());

        UNIT_ASSERT(addr.Parse("::de:adbe:af00"));
        UNIT_ASSERT_VALUES_EQUAL("::de:adbe:af00", addr.ToString());
        UNIT_ASSERT_VALUES_EQUAL("AK--rd4AAAAAAAAAAAAAAA", addr.ToBase64String());

        UNIT_ASSERT(addr.Parse("0.171.205.239"));
        UNIT_ASSERT_VALUES_EQUAL("0.171.205.239", addr.ToString());
        UNIT_ASSERT_VALUES_EQUAL("782rAA", addr.ToBase64String());

        UNIT_ASSERT(addr.Parse("::1:2345:abcd:ef98:7600"));
        UNIT_ASSERT_VALUES_EQUAL("::1:2345:abcd:ef98:7600", addr.ToString());
        UNIT_ASSERT_VALUES_EQUAL("AHaY782rRSMBAAAAAAAAAA", addr.ToBase64String());

        const TString s = "::12:34:55:0:0:0";
        UNIT_ASSERT(addr.Parse(s, 0, s.size()));
        UNIT_ASSERT(addr.Parse(s));
    }

    Y_UNIT_TEST(projectId) {
        TIpAddr addr;
        UNIT_ASSERT(addr.Parse("8.8.8.8"));
        UNIT_ASSERT_VALUES_EQUAL(0, addr.ProjectId());
        UNIT_ASSERT_VALUES_EQUAL("6.5.4.1", TIpAddr::Normalize("::ffff:6.5.4.1"));
        UNIT_ASSERT_VALUES_EQUAL(0, addr.ProjectId());

        UNIT_ASSERT(addr.Parse("::ffff:ffff:ffff:ffff"));
        UNIT_ASSERT_VALUES_EQUAL(0xffffffff, addr.ProjectId());

        UNIT_ASSERT(addr.Parse("2a02:6b8:b010:1d::7f"));
        UNIT_ASSERT_VALUES_EQUAL(0, addr.ProjectId());

        UNIT_ASSERT(addr.Parse("2a02:6b8:c00:103b:0:604:9:2224"));
        UNIT_ASSERT_VALUES_EQUAL(0x604, addr.ProjectId());
    }

    Y_UNIT_TEST(operators) {
        TIpAddr addr;
        TIpAddr addr2;
        UNIT_ASSERT(addr.Parse("::de:adbe:af00"));
        addr2 = addr;

        TStringStream s;
        s << addr;
        UNIT_ASSERT_VALUES_EQUAL("::de:adbe:af00", s.Str());

        UNIT_ASSERT(addr == addr2);
        UNIT_ASSERT(!(addr < addr2));
        UNIT_ASSERT(addr <= addr2);
        UNIT_ASSERT(!(addr > addr2));
        UNIT_ASSERT(addr >= addr2);

        addr2.Parse("::de:adbe:af01");
        UNIT_ASSERT(addr != addr2);
        UNIT_ASSERT(addr < addr2);
        UNIT_ASSERT(addr <= addr2);
        UNIT_ASSERT(!(addr > addr2));
        UNIT_ASSERT(!(addr >= addr2));

        addr2.Parse("::af00");
        UNIT_ASSERT(addr != addr2);
        UNIT_ASSERT(!(addr < addr2));
        UNIT_ASSERT(!(addr <= addr2));
        UNIT_ASSERT(addr > addr2);
        UNIT_ASSERT(addr >= addr2);

        addr2.Parse("::af00");

        addr2 -= 1;
        UNIT_ASSERT_VALUES_EQUAL("::aeff", addr2.ToString());

        addr2 <<= 96;
        UNIT_ASSERT_VALUES_EQUAL("0:aeff::", addr2.ToString());

        addr2 >>= 96;
        UNIT_ASSERT_VALUES_EQUAL("::aeff", addr2.ToString());

        addr2 >>= 300;
        UNIT_ASSERT_VALUES_EQUAL("::", addr2.ToString());

        addr2.Parse("0:0:0:1::");
        UNIT_ASSERT_VALUES_EQUAL("0:0:0:1::", addr2.ToString());

        addr2 -= 1;
        UNIT_ASSERT_VALUES_EQUAL("::ffff:ffff:ffff:ffff", addr2.ToString());
    }

    Y_UNIT_TEST(packed) {
        TString ip = "2a02:6b8:b080:7106::1:14";
        std::string result({0x2a, 0x02, 0x06, (char)0xb8, (char)0xb0, (char)0x80, 0x71, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x14});

        UNIT_ASSERT_VALUES_EQUAL(TIpAddr(ip).ToPackedStringV6(), TString(result));
        UNIT_ASSERT_VALUES_EQUAL(TIpAddr(ip).ToPackedStringShortest(), TString(result));

        ip = "127.0.0.1";
        result = std::string({0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, (char)0xff, (char)0xff, 0x7f, 0x00, 0x00, 0x01});
        UNIT_ASSERT_VALUES_EQUAL(TIpAddr(ip).ToPackedStringV6(), TString(result));
        result = std::string({0x7f, 0x00, 0x00, 0x01});
        UNIT_ASSERT_VALUES_EQUAL(TIpAddr(ip).ToPackedStringShortest(), TString(result));
    }

    Y_UNIT_TEST(isLoopback) {
        UNIT_ASSERT(TIpAddr("127.0.0.0").IsLoopback());
        UNIT_ASSERT(TIpAddr("127.0.0.1").IsLoopback());
        UNIT_ASSERT(TIpAddr("127.1.0.1").IsLoopback());
        UNIT_ASSERT(!TIpAddr("126.0.0.1").IsLoopback());
        UNIT_ASSERT(!TIpAddr("128.0.0.1").IsLoopback());

        UNIT_ASSERT(TIpAddr("::1").IsLoopback());
        UNIT_ASSERT(!TIpAddr("::2").IsLoopback());
    }

    Y_UNIT_TEST(isPrivate) {
        UNIT_ASSERT(TIpAddr("192.168.0.1").IsPrivate());
        UNIT_ASSERT(TIpAddr("192.168.255.255").IsPrivate());
        UNIT_ASSERT(TIpAddr("172.16.0.1").IsPrivate());
        UNIT_ASSERT(TIpAddr("172.16.255.1").IsPrivate());
        UNIT_ASSERT(TIpAddr("10.0.0.1").IsPrivate());
        UNIT_ASSERT(TIpAddr("10.16.255.1").IsPrivate());
        UNIT_ASSERT(TIpAddr("100.64.0.1").IsPrivate());
        UNIT_ASSERT(TIpAddr("100.127.255.254").IsPrivate());
        UNIT_ASSERT(TIpAddr("198.18.0.1").IsPrivate());
        UNIT_ASSERT(TIpAddr("198.19.255.1").IsPrivate());
        UNIT_ASSERT(TIpAddr("169.254.0.1").IsPrivate());
        UNIT_ASSERT(!TIpAddr("192.169.0.1").IsPrivate());
        UNIT_ASSERT(!TIpAddr("172.32.0.0").IsPrivate());
        UNIT_ASSERT(!TIpAddr("11.0.0.0").IsPrivate());
        UNIT_ASSERT(!TIpAddr("100.63.255.1").IsPrivate());
        UNIT_ASSERT(!TIpAddr("100.128.0.0").IsPrivate());
        UNIT_ASSERT(!TIpAddr("198.17.255.255").IsPrivate());
        UNIT_ASSERT(!TIpAddr("198.20.0.0").IsPrivate());
        UNIT_ASSERT(!TIpAddr("169.255.0.0").IsPrivate());

        UNIT_ASSERT(TIpAddr("fd0c:b055:482f:43b6:ff::").IsPrivate());
        UNIT_ASSERT(TIpAddr("fd0c:b055:482f:43b6::1").IsPrivate());
        UNIT_ASSERT(TIpAddr("fc00::1").IsPrivate());
        UNIT_ASSERT(TIpAddr("fec0::1").IsPrivate());
        UNIT_ASSERT(TIpAddr("fe80::1").IsPrivate());
        UNIT_ASSERT(!TIpAddr("fb00::").IsPrivate());
        UNIT_ASSERT(!TIpAddr("ff00::1").IsPrivate());
    }

    Y_UNIT_TEST(isMulticast) {
        UNIT_ASSERT(TIpAddr("224.0.0.0").IsMulticast());
        UNIT_ASSERT(TIpAddr("234.0.0.1").IsMulticast());
        UNIT_ASSERT(!TIpAddr("223.0.0.0").IsMulticast());
        UNIT_ASSERT(!TIpAddr("240.0.0.0").IsMulticast());

        UNIT_ASSERT(TIpAddr("ff01::").IsMulticast());
        UNIT_ASSERT(TIpAddr("ffff::").IsMulticast());
        UNIT_ASSERT(!TIpAddr("fe00::").IsMulticast());
        UNIT_ASSERT(!TIpAddr("fa00::").IsMulticast());
    }

    Y_UNIT_TEST(toStringHalfV6) {

        UNIT_ASSERT_VALUES_EQUAL("1.1.1.1", TIpAddr("1.1.1.1").ToStringHalfV6());
        UNIT_ASSERT_VALUES_EQUAL("127.0.0.1", TIpAddr("127.0.0.1").ToStringHalfV6());

        UNIT_ASSERT_VALUES_EQUAL("ffff:ffff:ffff:ffff::", TIpAddr("ffff:ffff:ffff:ffff:ffff:ffff:ffff:ffff").ToStringHalfV6());
        UNIT_ASSERT_VALUES_EQUAL("2a02:6b8:b080:7106::", TIpAddr("2a02:6b8:b080:7106::1:2:1415").ToStringHalfV6());
        UNIT_ASSERT_VALUES_EQUAL("::", TIpAddr("::").ToStringHalfV6());
    }
}
