#include "common.h"

#include <passport/infra/libs/cpp/utils/file.h>
#include <passport/infra/libs/cpp/utils/thread_local_id.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/log/logger.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/datetime/base.h>
#include <util/generic/string.h>
#include <util/stream/file.h>
#include <util/stream/format.h>
#include <util/system/fs.h>

using namespace NPassport::NUtils;
using namespace NPassport;

const TString tmpFileName("tmp_log_file");

Y_UNIT_TEST_SUITE(PasspUtilsLogs) {
    Y_UNIT_TEST(file) {
        NFs::Remove(tmpFileName);
        UNIT_ASSERT(!NFs::Exists(tmpFileName));
        UNIT_ASSERT_EXCEPTION_CONTAINS(TFileLoader(
                                           tmpFileName, [](auto, auto) {}, TDuration::Seconds(0)),
                                       yexception,
                                       R"(can't open "tmp_log_file")");

        std::unique_ptr<TLogger> log;
        UNIT_ASSERT_NO_EXCEPTION(log = std::make_unique<TLogger>(tmpFileName, "DEBUG", true));
        UNIT_ASSERT_VALUES_EQUAL((int)TLogger::ELevel::DEBUG, (int)log->getLevel());

        UNIT_ASSERT(NFs::Exists(tmpFileName));

        UNIT_ASSERT_EXCEPTION_CONTAINS(TFileLoader(
                                           tmpFileName, [](auto, auto) {}, TDuration::Seconds(0)),
                                       yexception,
                                       "FileLoader: file is empty: tmp_log_file");
        Sleep(TDuration::MilliSeconds(1100));
        UNIT_ASSERT_EXCEPTION_CONTAINS(readFile(tmpFileName), yexception, "FileLoader: file is empty: " + tmpFileName);

        log->info("Hello");
        Sleep(TDuration::Seconds(1));
        log->rotate();
        log.reset();
        UNIT_ASSERT_NO_EXCEPTION(log = std::make_unique<TLogger>(tmpFileName, "DEBUG", true));
        log->error("World");
        log.reset();
        Sleep(TDuration::MilliSeconds(1100));

        UNIT_ASSERT_NO_EXCEPTION(TFileLoader(
            tmpFileName, [](auto, auto) {}, TDuration::MilliSeconds(50)));

        UNIT_ASSERT_VALUES_EQUAL("INFO: Hello\nERROR: World\n", readFile(tmpFileName));
        NFs::Remove(tmpFileName);
    }

    Y_UNIT_TEST(loggerStatic) {
        TString request_id("id");
        UNIT_ASSERT_VALUES_EQUAL("", getThreadLocalRequestId());
        setThreadLocalRequestId(&request_id);
        UNIT_ASSERT_VALUES_EQUAL(request_id, getThreadLocalRequestId());

        UNIT_ASSERT_VALUES_EQUAL("DEBUG", TLogger::levelToString(TLogger::ELevel::DEBUG));
        UNIT_ASSERT_VALUES_EQUAL("INFO", TLogger::levelToString(TLogger::ELevel::INFO));
        UNIT_ASSERT_VALUES_EQUAL("WARNING", TLogger::levelToString(TLogger::ELevel::WARNING));
        UNIT_ASSERT_VALUES_EQUAL("ERROR", TLogger::levelToString(TLogger::ELevel::ERROR));
    }

    Y_UNIT_TEST(logger) {
        UNIT_ASSERT_EXCEPTION_CONTAINS(std::make_unique<TLogger>(tmpFileName, "TEST", true), yexception, "bad log string to level cast");
        UNIT_ASSERT_EXCEPTION_CONTAINS(std::make_unique<TLogger>("/root/tmp/log.log", "INFO", true),
                                       yexception,
                                       "Failed to open file: /root/tmp/log.log");

        NFs::Remove(tmpFileName);
        std::unique_ptr<TLogger> log;

        UNIT_ASSERT_NO_EXCEPTION(log = std::make_unique<TLogger>(tmpFileName, "DEBUG", true));
        UNIT_ASSERT_VALUES_EQUAL((int)TLogger::ELevel::DEBUG, (int)log->getLevel());
        log->debug("debug: %s", "ON");
        log->info("info: %d %s", 1, "test");
        log->warning("warning!");
        log->error("error: %d %d %d %d", 1, 2, 3, 4);
        log.reset();

        {
            TFileInput log_file(tmpFileName);
            UNIT_ASSERT_VALUES_EQUAL("DEBUG: debug: ON\n"
                                     "INFO: info: 1 test\n"
                                     "WARNING: warning!\n"
                                     "ERROR: error: 1 2 3 4\n",
                                     log_file.ReadAll());
        }

        UNIT_ASSERT_NO_EXCEPTION(log = std::make_unique<TLogger>(tmpFileName, "INFO", false));
        UNIT_ASSERT_VALUES_EQUAL((int)TLogger::ELevel::INFO, (int)log->getLevel());
        log->debug("debug: %s", "ON");
        log->info("info: %d %s", 2, "test");
        Sleep(TDuration::Seconds(1));
        log->rotate();
        log->warning("warning!");
        log->error("error: %o %x %X %#x", 11, 12, 13, 14);
        log.reset();

        {
            TFileInput log_file(tmpFileName);
            UNIT_ASSERT_VALUES_EQUAL("DEBUG: debug: ON\n"
                                     "INFO: info: 1 test\n"
                                     "WARNING: warning!\n"
                                     "ERROR: error: 1 2 3 4\n"
                                     "info: 2 test\n"
                                     "File reopened\n"
                                     "warning!\n"
                                     "error: 13 c D 0xe\n",
                                     log_file.ReadAll());
        }

        NFs::Remove(tmpFileName);
        TString request_id("test_req_id");
        setThreadLocalRequestId(&request_id);

        UNIT_ASSERT_NO_EXCEPTION(log = std::make_unique<TLogger>(tmpFileName, "WARNING", true, "_DEFAULT_"));
        UNIT_ASSERT_VALUES_EQUAL((int)TLogger::ELevel::WARNING, (int)log->getLevel());
        log->debug("debug: %s", "ON");
        log->info("info: %d %s", 2, "test");
        log->warning("warning!");
        setThreadLocalRequestId(nullptr);
        log->error("not warning!");
        log.reset();

        {
            TFileInput log_file(tmpFileName);
            UNIT_ASSERT_STRING_CONTAINS(log_file.ReadLine(), " test_req_id WARNING: warning!");
            UNIT_ASSERT_STRING_CONTAINS(log_file.ReadLine(), " ERROR: not warning!");
        }

        TString cur_time = TInstant::Now().FormatLocalTime("%A %B %d-%m-%Y %z %Z");
        UNIT_ASSERT_NO_EXCEPTION(log = std::make_unique<TLogger>(tmpFileName, "ERROR", true, "%A %B %d-%m-%Y %z %Z"));
        UNIT_ASSERT_VALUES_EQUAL((int)TLogger::ELevel::ERROR, (int)log->getLevel());
        log->debug("debug: %s", "ON");
        log->info("info: %d %s", 2, "test");
        log->warning("warning!");
        log->error("err\nor!");
        log.reset();

        {
            TFileInput log_file(tmpFileName);
            UNIT_ASSERT_STRING_CONTAINS(log_file.ReadLine(), " test_req_id WARNING: warning!");
            UNIT_ASSERT_STRING_CONTAINS(log_file.ReadLine(), "  ERROR: not warning!");
            UNIT_ASSERT_STRING_CONTAINS(log_file.ReadLine(), cur_time + " ERROR: err");
            UNIT_ASSERT_VALUES_EQUAL(log_file.ReadLine(), "or!");
        }

        UNIT_ASSERT_NO_EXCEPTION(log = std::make_unique<TLogger>(tmpFileName, "DEBUG", true, "%A %B %d-%m-%Y %z %Z"));
        UNIT_ASSERT_VALUES_EQUAL((int)TLogger::ELevel::DEBUG, (int)log->getLevel());
        log->debug() << "debug: "
                     << "ON";
        log->info() << "info: " << 2 << " "
                    << "test";
        log->warning() << "warning!";
        log->error() << "err\nor!";
        log.reset();

        {
            TFileInput log_file(tmpFileName);
            UNIT_ASSERT_STRING_CONTAINS(log_file.ReadLine(), " test_req_id WARNING: warning!");
            UNIT_ASSERT_STRING_CONTAINS(log_file.ReadLine(), "  ERROR: not warning!");
            UNIT_ASSERT_STRING_CONTAINS(log_file.ReadLine(), cur_time + " ERROR: err");
            UNIT_ASSERT_VALUES_EQUAL(log_file.ReadLine(), "or!");
            UNIT_ASSERT_STRING_CONTAINS(log_file.ReadLine(), " DEBUG: debug: ON");
            UNIT_ASSERT_STRING_CONTAINS(log_file.ReadLine(), " INFO: info: 2 test");
            UNIT_ASSERT_STRING_CONTAINS(log_file.ReadLine(), " WARNING: warning!");
            UNIT_ASSERT_STRING_CONTAINS(log_file.ReadLine(), " ERROR: err");
            UNIT_ASSERT_VALUES_EQUAL(log_file.ReadLine(), "or!");
        }
    }

    Y_UNIT_TEST(commonLog) {
        NFs::Remove(tmpFileName);
        std::unique_ptr<TLogger> log;

        UNIT_ASSERT_NO_EXCEPTION(log = std::make_unique<TLogger>(tmpFileName, "INFO", true));
        TLog::debug("Common log writes to std error: %s", "Hello, there!");
        TLog::info("Common log writes to std error: %s", "Hello, there!");
        TLog::warning("Common log writes to std error: %s", "Hello, there!");
        TLog::error("Common log writes to std error: %s", "Hello, there!");

        TLog::init(std::move(log));
        TLog::debug("you won't see this :)");
        TLog::info("common log info: %f", 3.14);
        TLog::logGotSignal(TLog::ESignal::Sighup);
        TLog::warning("warning: %s", "achtung!");
        Sleep(TDuration::Seconds(1));
        TLog::rotate();
        TLog::error("ops...");
        TLog::reset();

        {
            TFileInput log_file(tmpFileName);
            UNIT_ASSERT_VALUES_EQUAL("INFO: common log info: 3.140000\n"
                                     "DEBUG: Got SIGHUP\n"
                                     "WARNING: warning: achtung!\n"
                                     "INFO: File reopened\n"
                                     "ERROR: ops...\n",
                                     log_file.ReadAll());
        }
    }

    Y_UNIT_TEST(commonLogHandler) {
        NFs::Remove(tmpFileName);
        std::unique_ptr<TLogger> log;

        UNIT_ASSERT_NO_EXCEPTION(log = std::make_unique<TLogger>(tmpFileName, "INFO", true));
        TLog::debug() << "Common log writes to std error: Hello, there!";
        TLog::info() << "Common log writes to std error: Hello, there!";
        TLog::warning() << "Common log writes to std error: Hello, there!";
        TLog::error() << "Common log writes to std error: Hello, there!";

        TLog::init(std::move(log));
        TLog::debug(100) << "you won't see this :)";
        TLog::info(100) << "common log info: " << Prec(3.14, PREC_POINT_DIGITS, 6);
        TLog::warning(100) << "warning: achtung!";
        Sleep(TDuration::Seconds(1));
        TLog::rotate();
        TLog::error(100) << "ops...";
        TLog::reset();

        {
            TFileInput log_file(tmpFileName);
            UNIT_ASSERT_STRINGS_EQUAL("INFO: common log info: 3.140000\n"
                                      "WARNING: warning: achtung!\n"
                                      "INFO: File reopened\n"
                                      "ERROR: ops...\n",
                                      log_file.ReadAll());
        }
    }

    Y_UNIT_TEST(dbPoolLog) {
        NFs::Remove(tmpFileName);
        setThreadLocalRequestId(nullptr);

        TDbPoolLog logStderr;
        logStderr.debug("DbPoolLog log also writes to std error: %s", "Hello, there!");
        logStderr.info("DbPoolLog log also writes to std error: %s", "Hello, there!");
        logStderr.warning("DbPoolLog log also writes to std error: %s", "Hello, there!");
        logStderr.error("DbPoolLog log also writes to std error: %s", "Hello, there!");

        {
            TDbPoolLog logFile(tmpFileName);
            logFile.debug("you will see this :)");
            logFile.info("and this: %f", 3.14);
            logFile.warning("warning: %s", "achtung!");
            logFile.error("ops...");
            logFile.debug() << "2: you will see this :)";
            logFile.info() << "2: and this: " << 3.14;
            logFile.warning() << "2: warning: "
                              << "achtung!";
            logFile.error() << "2: ops...";
        }

        logStderr.debug("DbPoolLog log also writes to std error 2: %s", "Hello, there!");
        logStderr.info("DbPoolLog log also writes to std error 2: %s", "Hello, there!");
        logStderr.warning("DbPoolLog log also writes to std error 2: %s", "Hello, there!");
        logStderr.error("DbPoolLog log also writes to std error 2: %s", "Hello, there!");

        {
            TFileInput log_file(tmpFileName);
            UNIT_ASSERT_STRING_CONTAINS(log_file.ReadLine(), "  you will see this :)");
            UNIT_ASSERT_STRING_CONTAINS(log_file.ReadLine(), "  and this: 3.140000");
            UNIT_ASSERT_STRING_CONTAINS(log_file.ReadLine(), "  warning: achtung!");
            UNIT_ASSERT_STRING_CONTAINS(log_file.ReadLine(), "  ops...");
            UNIT_ASSERT_STRING_CONTAINS(log_file.ReadLine(), "  2: you will see this :)");
            UNIT_ASSERT_STRING_CONTAINS(log_file.ReadLine(), "  2: and this: 3.14");
            UNIT_ASSERT_STRING_CONTAINS(log_file.ReadLine(), "  2: warning: achtung!");
            UNIT_ASSERT_STRING_CONTAINS(log_file.ReadLine(), "  2: ops...");
        }
    }
}
