#include "enum.h"

#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport::NUtils;

Y_UNIT_TEST_SUITE(PasspUtilsStrings) {
    Y_UNIT_TEST(notDigit) {
        for (int i = 0; i < 256; ++i) {
            bool res = i < '0' || i > '9';
            UNIT_ASSERT_VALUES_EQUAL(res, NotDigit(char(i)));
        }
    }

    Y_UNIT_TEST(digitsOnly) {
        UNIT_ASSERT_VALUES_EQUAL(true, DigitsOnly("100050010"));
        UNIT_ASSERT_VALUES_EQUAL(true, DigitsOnly("50010"));
        UNIT_ASSERT_VALUES_EQUAL(true, DigitsOnly(""));

        UNIT_ASSERT_VALUES_EQUAL(false, DigitsOnly("10005001a"));
        UNIT_ASSERT_VALUES_EQUAL(false, DigitsOnly("my uid is best"));
    }

    Y_UNIT_TEST(hexDigitsOnly) {
        UNIT_ASSERT_VALUES_EQUAL(true, HexDigitsOnly("100050010"));
        UNIT_ASSERT_VALUES_EQUAL(true, HexDigitsOnly("50010"));
        UNIT_ASSERT_VALUES_EQUAL(true, HexDigitsOnly(""));

        UNIT_ASSERT_VALUES_EQUAL(true, HexDigitsOnly("10005001a"));
        UNIT_ASSERT_VALUES_EQUAL(false, HexDigitsOnly("my uid is best"));
    }

    Y_UNIT_TEST(lowOnly) {
        UNIT_ASSERT_VALUES_EQUAL(false, LowercaseLettersOnly("100050010"));
        UNIT_ASSERT_VALUES_EQUAL(false, LowercaseLettersOnly("50010"));
        UNIT_ASSERT_VALUES_EQUAL(false, LowercaseLettersOnly("10005001a"));
        UNIT_ASSERT_VALUES_EQUAL(false, LowercaseLettersOnly("my uid is best"));

        UNIT_ASSERT_VALUES_EQUAL(true, LowercaseLettersOnly(""));
        UNIT_ASSERT_VALUES_EQUAL(true, LowercaseLettersOnly("myuidisbest"));
        UNIT_ASSERT_VALUES_EQUAL(false, LowercaseLettersOnly("myUidUsBest"));
    }

    Y_UNIT_TEST(secureCompare) {
        UNIT_ASSERT_VALUES_EQUAL(false, SecureCompare(nullptr, 100500, "val", 3));
        UNIT_ASSERT_VALUES_EQUAL(false, SecureCompare(nullptr, 100500, "", 1));

        UNIT_ASSERT_VALUES_EQUAL(true, SecureCompare("val", 3, "val", 3));
        UNIT_ASSERT_VALUES_EQUAL(false, SecureCompare("val", 3, "VAL", 3));

        UNIT_ASSERT_VALUES_EQUAL(false, SecureCompare("val0", 4, "val", 3));
        UNIT_ASSERT_VALUES_EQUAL(true, SecureCompare("val0", 3, "val", 3));

        UNIT_ASSERT_VALUES_EQUAL(false, SecureCompare("val", 3, "val0", 4));
        UNIT_ASSERT_VALUES_EQUAL(true, SecureCompare("val", 3, "val0", 3));

        UNIT_ASSERT_VALUES_EQUAL(true, SecureCompare(TStringBuf("val"), TStringBuf("val")));
        UNIT_ASSERT_VALUES_EQUAL(false, SecureCompare(TStringBuf("val"), TStringBuf("va1")));
    }

    Y_UNIT_TEST(toBoolean) {
        UNIT_ASSERT_VALUES_EQUAL(true, ToBoolean("yes"));
        UNIT_ASSERT_VALUES_EQUAL(true, ToBoolean("1"));
        UNIT_ASSERT_VALUES_EQUAL(true, ToBoolean("true"));

        UNIT_ASSERT_VALUES_EQUAL(false, ToBoolean(""));
        UNIT_ASSERT_VALUES_EQUAL(false, ToBoolean("0"));
        UNIT_ASSERT_VALUES_EQUAL(false, ToBoolean("yes123"));
        UNIT_ASSERT_VALUES_EQUAL(false, ToBoolean("123"));
        UNIT_ASSERT_VALUES_EQUAL(false, ToBoolean("false"));
        UNIT_ASSERT_VALUES_EQUAL(false, ToBoolean("ololo"));
    }

    Y_UNIT_TEST(toInt) {
        UNIT_ASSERT_VALUES_EQUAL(1LL, ToInt("1", ""));
        UNIT_ASSERT_VALUES_EQUAL(123LL, ToInt("123", ""));
        UNIT_ASSERT_VALUES_EQUAL(92233LL, ToInt("92233", ""));
        UNIT_ASSERT_VALUES_EQUAL(-92233LL, ToInt("-92233", ""));
        UNIT_ASSERT_VALUES_EQUAL(0x7ffffffffffffffeLL, ToInt("9223372036854775806", ""));
        UNIT_ASSERT_VALUES_EQUAL(-9223372036854775805LL, ToInt("-9223372036854775805", ""));
        UNIT_ASSERT_EXCEPTION_CONTAINS(ToInt("18446744073709551614", "number"), std::invalid_argument, "invalid number value: 18446744073709551614"); // 0xfffffffffffffffeULL
    }

    Y_UNIT_TEST(toUInt) {
        UNIT_ASSERT_VALUES_EQUAL(1ULL, ToUInt("1", ""));
        UNIT_ASSERT_VALUES_EQUAL(123ULL, ToUInt("123", ""));
        UNIT_ASSERT_VALUES_EQUAL(92233ULL, ToUInt("92233", ""));
        UNIT_ASSERT_EXCEPTION_CONTAINS(ToUInt("-92233", "negative"), std::invalid_argument, "invalid negative value: -92233");
        UNIT_ASSERT_VALUES_EQUAL(0x7ffffffffffffffeULL, ToUInt("9223372036854775806", ""));
        UNIT_ASSERT_EXCEPTION_CONTAINS(ToUInt("18446744073709551616", "uint"), std::invalid_argument, "invalid uint value: 18446744073709551616"); // 0x10000000000000000ULL,
        UNIT_ASSERT_VALUES_EQUAL(0xfffffffffffffffeULL, ToUInt("18446744073709551614", ""));
        UNIT_ASSERT_EXCEPTION_CONTAINS(ToUInt("-9223372036854775805", "big negative"), std::invalid_argument, "invalid big negative value: -9223372036854775805");
    }

    Y_UNIT_TEST(utf8CharLength) {
        std::vector<TString> valid = {
            "\xf4\x82\x83\xbb",
            "\xf0\x90\xbc\xbf",
            "\xf3\x85\xab\xba",
            "\xe0\xaa\xbb",
            "\xe3\xa0\xb0",
            "\xc2\xa2",
            "\xd2\xb2",
            "\xdf\x80",
            "\x01",
            "\x20",
            "\x40",
            "\x7f",
        };

        for (const TString& s : valid) {
            TStringBuf buf = s;
            UNIT_ASSERT_VALUES_EQUAL_C(s.size(), Utf8CharLength(buf.begin(), buf.end()), Bin2hex(buf));
        }

        std::vector<TString> invalid = {
            "\xf0\x80\xbc\xbf",
            "\xff",
            "\xf5",
            "\xf4",
            "\xf0",
            "\xe0",
            "\xc0",
            "\xc2",
            "\xd2",
            "\xf5\xaa",
            "\xf4\xaa\xaa\xbb",
            "\xf4\x85\xab\x20",
            "\xf3\x88",
            "\xf3\xf0\xa0\x80",
            "\xf0\x20\x40\x80",
            "\xe0\x80",
            "\xe0\x80\xff",
            "\xe0\x88\x20",
            "\xc2\xc2",
            "\xc2\x08",
            "\xd0\xd0",
            "\xf7",
            "\xed\xa0\xbd",
        };
        for (const TString& s : invalid) {
            TStringBuf buf = s;
            UNIT_ASSERT_VALUES_EQUAL_C(0, Utf8CharLength(buf.begin(), buf.end()), Bin2hex(s));
        }
    }

    Y_UNIT_TEST(replaceAny) {
        UNIT_ASSERT_VALUES_EQUAL("abalabalabalaba", ReplaceAny("olololo", "o", "aba"));
        UNIT_ASSERT_VALUES_EQUAL("magic cr123e123 f123e123", ReplaceAny("magic cruel fuel", "ul", "123"));
        UNIT_ASSERT_VALUES_EQUAL("magic_cruel_fuel", ReplaceAny("magic cruel fuel", " ", "_"));

        UNIT_ASSERT_VALUES_EQUAL("lll", ReplaceAny("olololo", "o", ""));
        UNIT_ASSERT_VALUES_EQUAL("olololo", ReplaceAny("olololo", "", "aba"));
        UNIT_ASSERT_VALUES_EQUAL("", ReplaceAny("", "o", "aba"));
    }

    Y_UNIT_TEST(hostFromUri) {
        UNIT_ASSERT_VALUES_EQUAL("", HostFromUri(""));
        UNIT_ASSERT_VALUES_EQUAL("blackbox.yandex.net",
                                 HostFromUri("http://blackbox.yandex.net/"));
        UNIT_ASSERT_VALUES_EQUAL("blackbox.yandex.net",
                                 HostFromUri("https://blackbox.yandex.net/"));
        UNIT_ASSERT_VALUES_EQUAL("",
                                 HostFromUri("https://blackbox.yandex.net/blackbox"));
        UNIT_ASSERT_VALUES_EQUAL("blackbox.yandex.net",
                                 HostFromUri("https://blackbox.yandex.net:443/"));

        const std::vector<TString> ok = {
            "yandex.ru",
            "yandex.ru:80",
            "http://yandex.ru",
            "http://yandex.ru:80",
            "https://yandex.ru",
            "https://yandex.ru:80",
        };
        const std::vector<TString> ok2 = {
            "pass-dd-s11.sezam.yandex.net",
            "pass-dd-s11.sezam.yandex.net:80",
            "http://pass-dd-s11.sezam.yandex.net",
            "http://pass-dd-s11.sezam.yandex.net:80",
            "https://pass-dd-s11.sezam.yandex.net",
            "https://pass-dd-s11.sezam.yandex.net:80",
        };

        for (const TString& s : ok) {
            UNIT_ASSERT_VALUES_EQUAL("yandex.ru", HostFromUri(s));
        }
        for (const TString& s : ok2) {
            UNIT_ASSERT_VALUES_EQUAL("pass-dd-s11.sezam.yandex.net", HostFromUri(s));
        }

        const std::vector<TString> bad = {
            "yandex.ru?lol=true",
            "yandex.ru:80?lol=true",
            "http://yandex.ru?lol=true",
            "http://yandex.ru:80?lol=true",
            "https://yandex.ru?lol=true",
            "https://yandex.ru:80?lol=true",
            "yandex.ru/shutdown",
            "yandex.ru:80/shutdown",
            "http://yandex.ru/shutdown",
            "http://yandex.ru:80/shutdown",
            "https://yandex.ru/shutdown",
            "https://yandex.ru:80/shutdown",
            "pass-dd-s11.sezam.yandex.net?lol=true",
            "pass-dd-s11.sezam.yandex.net:80?lol=true",
            "http://pass-dd-s11.sezam.yandex.net?lol=true",
            "http://pass-dd-s11.sezam.yandex.net:80?lol=true",
            "https://pass-dd-s11.sezam.yandex.net?lol=true",
            "https://pass-dd-s11.sezam.yandex.net:80?lol=true",
            "pass-dd-s11.sezam.yandex.net/shutdown",
            "pass-dd-s11.sezam.yandex.net:80/shutdown",
            "http://pass-dd-s11.sezam.yandex.net/shutdown",
            "http://pass-dd-s11.sezam.yandex.net:80/shutdown",
            "https://pass-dd-s11.sezam.yandex.net/shutdown",
            "https://pass-dd-s11.sezam.yandex.net:80/shutdown",
            TString("yandex.ru").append(1, 0),
            TString("yandex.ru").append(1, 0),
            TString("yandex.ru:80").append(1, 0),
            TString("http://yandex.ru").append(1, 0),
            TString("http://yandex.ru:80").append(1, 0),
            TString("https://yandex.ru").append(1, 0),
            TString("https://yandex.ru:80").append(1, 0),
            TString(1, 0).append("yandex.ru"),
            TString(1, 0).append("yandex.ru:80"),
            TString(1, 0).append("http://yandex.ru"),
            TString(1, 0).append("http://yandex.ru:80"),
            TString(1, 0).append("https://yandex.ru"),
            TString(1, 0).append("https://yandex.ru:80"),
            "yande*x.ru",
            "yandeывx.ru:80",
            "http\v://yandex.ru",
            "http://ya\tndex.ru:80",
            "https://ya\rndex.ru",
            "http\ns://yandex.ru:80",
            "yandex.ru:80:80",
            "http://http://yandex.ru",
            "http:/yandex.ru:80",
            "httpss://yandex.ru",
            "https://yandex.ru::80",
        };

        for (const TString& s : bad) {
            UNIT_ASSERT_VALUES_EQUAL("", HostFromUri(s));
        }
    }

    Y_UNIT_TEST(append) {
        TString s;
        Append(s, ".", ".");
        UNIT_ASSERT_VALUES_EQUAL(s, "..");
        AppendExt(s, 700, "/", "/");
        UNIT_ASSERT_VALUES_EQUAL(s, "..//");
        s.clear();

        Append(s, ".", '.', ".");
        UNIT_ASSERT_VALUES_EQUAL(s, "...");
        s.clear();

        Append(s, ".", ".", ".", ".");
        UNIT_ASSERT_VALUES_EQUAL(s, "....");
        s.clear();

        Append(s, ".", ".", ".", ".", ".");
        UNIT_ASSERT_VALUES_EQUAL(s, ".....");
        s.clear();

        Append(s, ".", ".", ".", ".", ".", ".");
        UNIT_ASSERT_VALUES_EQUAL(s, "......");
        s.clear();

        Append(s, ".", ".", ".", ".", ".", ".", ".");
        UNIT_ASSERT_VALUES_EQUAL(s, ".......");
        s.clear();

        Append(s, ".", ".", ".", ".", ".", ".", ".", ".");
        UNIT_ASSERT_VALUES_EQUAL(s, "........");
        s.clear();

        Append(s, ".", ".", ".", ".", ".", ".", ".", ".", ".");
        UNIT_ASSERT_VALUES_EQUAL(s, ".........");
        s.clear();

        Append(s, ".", 17, '.');
        UNIT_ASSERT_VALUES_EQUAL(s, ".17.");
        s.clear();

        Append(s, 1, ".", 17, ".", 100500);
        UNIT_ASSERT_VALUES_EQUAL(s, "1.17.100500");
        s.clear();

        Append(s, std::numeric_limits<ui64>::min(), "_____", std::numeric_limits<ui64>::max());
        UNIT_ASSERT_VALUES_EQUAL(s, "0_____18446744073709551615");
        s.clear();

        Append(s, std::numeric_limits<i64>::min(), "_____", std::numeric_limits<i64>::max());
        UNIT_ASSERT_VALUES_EQUAL(s, "-9223372036854775808_____9223372036854775807");
        s.clear();
    }

    Y_UNIT_TEST(createStr) {
        UNIT_ASSERT_VALUES_EQUAL(CreateStr(".", "."), "..");
        UNIT_ASSERT_VALUES_EQUAL(CreateStr(".", ".", "."), "...");
        UNIT_ASSERT_VALUES_EQUAL(CreateStr(".", ".", ".", "."), "....");
        UNIT_ASSERT_VALUES_EQUAL(CreateStr(".", ".", ".", ".", "."), ".....");
        UNIT_ASSERT_VALUES_EQUAL(CreateStr(".", ".", ".", ".", ".", "."), "......");
        UNIT_ASSERT_VALUES_EQUAL(CreateStr(".", ".", ".", ".", ".", ".", "."), ".......");

        UNIT_ASSERT_VALUES_EQUAL(CreateStr(".", 17, "."), ".17.");
        UNIT_ASSERT_VALUES_EQUAL(CreateStr(1, ".", 17, ".", 100500), "1.17.100500");
        UNIT_ASSERT_VALUES_EQUAL(CreateStr(std::numeric_limits<ui64>::min(), "_____", std::numeric_limits<ui64>::max()),
                                 "0_____18446744073709551615");
        UNIT_ASSERT_VALUES_EQUAL(CreateStr(std::numeric_limits<i64>::min(), "_____", std::numeric_limits<i64>::max()),
                                 "-9223372036854775808_____9223372036854775807");

        UNIT_ASSERT_VALUES_EQUAL(CreateStr(".", ETestEnum::Value1, "."), ".Value1.");
    }

    Y_UNIT_TEST(appendSeparated) {
        TString s;
        UNIT_ASSERT_VALUES_EQUAL(AppendSeparated(s, ' ', ""), "");
        UNIT_ASSERT_VALUES_EQUAL(AppendSeparated(s, "separator", ""), "");

        UNIT_ASSERT_VALUES_EQUAL(AppendSeparated(s, " ; ", "one"), "one");
        UNIT_ASSERT_VALUES_EQUAL(AppendSeparated(s, ',', "two"), "one,two");
        UNIT_ASSERT_VALUES_EQUAL(AppendSeparated(s, ';', ""), "one,two");
        UNIT_ASSERT_VALUES_EQUAL(AppendSeparated(s, "; ", ""), "one,two");
        UNIT_ASSERT_VALUES_EQUAL(AppendSeparated(s, ",", "0"), "one,two,0");
        UNIT_ASSERT_VALUES_EQUAL(AppendSeparated(s, ", ", "three "), "one,two,0, three ");

        UNIT_ASSERT_VALUES_EQUAL(AppendSeparated(s, ';', "раз"), "one,two,0, three ;раз");
        UNIT_ASSERT_VALUES_EQUAL(AppendSeparated(s, '+', "два"), "one,two,0, three ;раз+два");
        UNIT_ASSERT_VALUES_EQUAL(AppendSeparated(s, '=', "три"), "one,two,0, three ;раз+два=три");
        UNIT_ASSERT_VALUES_EQUAL(AppendSeparated(s, '?', ""), "one,two,0, three ;раз+два=три");
        UNIT_ASSERT_VALUES_EQUAL(AppendSeparated(s, "???", ""), "one,two,0, three ;раз+два=три");

        UNIT_ASSERT_VALUES_EQUAL(AppendSeparated(s, ';', "1"), "one,two,0, three ;раз+два=три;1");
        UNIT_ASSERT_VALUES_EQUAL(AppendSeparated(s, TString(","), TString("2")), "one,two,0, three ;раз+два=три;1,2");
    }

    Y_UNIT_TEST(addMessage) {
        TString s;
        AddMessage(s, "abc");
        UNIT_ASSERT_VALUES_EQUAL(s, "abc");
        AddMessage(s, "asd");
        UNIT_ASSERT_VALUES_EQUAL(s, "abc, asd");
        AddMessage(s, "789");
        UNIT_ASSERT_VALUES_EQUAL(s, "abc, asd, 789");
        AddMessage(s, "");
        UNIT_ASSERT_VALUES_EQUAL(s, "abc, asd, 789");
    }
}
