#include "config.h"

#include "error_info.h"
#include "schema.h"
#include "xml_utils.h"

#include <passport/infra/libs/cpp/utils/log/file_logger.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/split.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <contrib/libs/libxml/include/libxml/parser.h>
#include <contrib/libs/libxml/include/libxml/tree.h>
#include <contrib/libs/libxml/include/libxml/xinclude.h>
#include <contrib/libs/libxml/include/libxml/xpath.h>

#include <library/cpp/xml/init/ptr.h>

#include <util/stream/file.h>

namespace NPassport::NXml {
    struct TConfig::TImpl {
        ::NXml::TxmlDocHolder Doc;
    };

    TConfig::TConfig(std::unique_ptr<TImpl> impl)
        : Impl_(std::move(impl))
    {
        Y_ENSURE(xmlDocGetRootElement(Impl_->Doc.Get()), "got empty config");
        Y_ENSURE(xmlXIncludeProcess(Impl_->Doc.Get()) >= 0, TUtils::GetLastXmlError());
    }

    TConfig::TConfig(TConfig&&) noexcept = default;
    TConfig::~TConfig() = default;

    TConfig TConfig::ReadFromFile(const TString& file) {
        char dummy;
        Y_ENSURE(TFileInput(file).ReadChar(dummy), "Failed to read file: " << file);

        TErrorInfo err(TErrorInfo::Structured);

        // xmlParseFile keeps lines in document - useful info in errors
        TImpl impl{::NXml::TxmlDocHolder(xmlParseFile(file.c_str()))};
        Y_ENSURE(impl.Doc, "Failed to parse :" << err.Get());

        return TConfig(std::make_unique<TImpl>(std::move(impl)));
    }

    TConfig TConfig::ReadFromMemory(const TStringBuf body) {
        TErrorInfo err(TErrorInfo::Structured);

        TImpl impl{::NXml::TxmlDocHolder(xmlParseMemory(body.data(), body.size()))};
        Y_ENSURE(impl.Doc, "Failed to parse :" << err.Get());

        return TConfig(std::make_unique<TImpl>(std::move(impl)));
    }

    static TString GetNodeContent(xmlXPathObjectPtr object, TConfig::EFetchContent content, const TString& xpath) {
        if (nullptr == object->nodesetval || 0 == object->nodesetval->nodeNr) {
            ythrow TConfig::TMissingException() << "nonexistent config param: " << xpath;
        }

        xmlNodeSetPtr ns = object->nodesetval;
        Y_ENSURE(ns->nodeTab[0], TUtils::GetLastXmlError());

        switch (content) {
            case TConfig::EFetchContent::Default:
                return TUtils::Value(ns->nodeTab[0]);
            case TConfig::EFetchContent::All:
                return TUtils::ValueAllContent(ns->nodeTab[0]);
        }
    }

    static TString GetObjectContent(xmlXPathObjectPtr object, TConfig::EFetchContent content, const TString& xpath) {
        TString res;

        switch (object->type) {
            case xmlXPathObjectType::XPATH_NODESET:
                res = GetNodeContent(object, content, xpath);
                break;
            case xmlXPathObjectType::XPATH_BOOLEAN:
                res = object->boolval ? "1" : "0";
                break;
            case xmlXPathObjectType::XPATH_NUMBER:
                res = IntToString<10>(i64(object->floatval));
                break;
            case xmlXPathObjectType::XPATH_STRING:
                res = (const char*)object->stringval;
                break;
            default:
                ythrow yexception() << "xpath type==" << (int)object->type << " is not supported: " << xpath;
        }

        return res;
    }

    TString TConfig::AsStringImpl(const TString& xpath, EFetchContent content) const {
        ::NXml::TxmlXPathContextHolder xctx(xmlXPathNewContext(Impl_->Doc.Get()));
        Y_ENSURE(xctx, TUtils::GetLastXmlError());

        TErrorInfo err(TErrorInfo::Generic);
        ::NXml::TxmlXPathObjectHolder object(xmlXPathEvalExpression((const xmlChar*)xpath.c_str(), xctx.Get()));
        Y_ENSURE(object, err.Get());

        TString res = GetObjectContent(object.Get(), content, xpath);

        Y_ENSURE_EX(!res.empty(), TEmptyException() << "config param is empty: " << xpath);

        return res;
    }

    int TConfig::AsInt(const TString& xpath) const {
        return As<int>(xpath);
    }

    int TConfig::AsInt(const TString& xpath, int defval) const {
        return As<int>(xpath, defval);
    }

    TString
    TConfig::AsString(const TString& xpath, EFetchContent content) const {
        return AsStringImpl(xpath, content);
    }

    TString
    TConfig::AsString(const TString& xpath, const TString& defval, EFetchContent content) const {
        try {
            return AsString(xpath, content);
        } catch (const std::exception& e) {
            return defval;
        }
    }

    bool TConfig::AsBool(const TString& xpath) const {
        return As<bool>(xpath);
    }

    bool TConfig::AsBool(const TString& xpath, bool defval) const {
        return As<bool>(xpath, defval);
    }

    double TConfig::AsDouble(const TString& xpath) const {
        return As<double>(xpath);
    }

    double TConfig::AsDouble(const TString& xpath, double defval) const {
        return As<double>(xpath, defval);
    }

    template <>
    bool TConfig::As<bool>(const TString& xpath) const {
        return NUtils::ToBoolean(As<TString>(xpath));
    }

    template <>
    double TConfig::As<double>(const TString& xpath) const {
        double res;
        const TString str = As<TString>(xpath);
        Y_ENSURE_EX(TryFromString(str, res),
                    TBadValueException() << "config param '" << xpath
                                         << "' cannot be casted to double: '" << str << "'");

        return res;
    }

    template <>
    TString TConfig::As<TString>(const TString& xpath) const {
        return AsStringImpl(xpath, EFetchContent::Default);
    }

    bool TConfig::Contains(const TString& xpath) const {
        ::NXml::TxmlXPathContextHolder xctx(xmlXPathNewContext(Impl_->Doc.Get()));
        Y_ENSURE(xctx, TUtils::GetLastXmlError());

        TErrorInfo err(TErrorInfo::Generic);
        ::NXml::TxmlXPathObjectHolder object(xmlXPathEvalExpression((const xmlChar*)xpath.c_str(), xctx.Get()));
        Y_ENSURE(object, err.Get());

        return (nullptr != object->nodesetval && 0 != object->nodesetval->nodeNr);
    }

    std::unique_ptr<NUtils::TFileLogger> TConfig::CreateLogger(const TString& xpath) const {
        TString filename = As<TString>(xpath + "/file");
        TString level = As<TString>(xpath + "/level");
        bool printLevel = As<TString>(xpath + "/print-level", "no") == "yes";
        TString timeFormat = As<TString>(xpath + "/time-format", "");
        int linesPerShot = As<int>(xpath + "/lines_per_shot", 1024);

        return std::make_unique<NUtils::TFileLogger>(filename, level, printLevel, timeFormat, linesPerShot);
    }

    void TConfig::InitCommonLog(const TString& xpath) const {
        const TString fileName = As<TString>(xpath + "/file");
        if (fileName != "_NOLOG_") {
            TLog::Init(CreateLogger(xpath));
        }
    }

    TConfig::TError TConfig::CheckSchema(const TDtdSchema& schema) const {
        return schema.Check(Impl_->Doc.Get());
    }

    TConfig::TError TConfig::CheckSchema(const TXsdSchema& schema) const {
        return schema.Check(Impl_->Doc.Get());
    }

    std::vector<TString> TConfig::SubKeys(const TString& xpath) const {
        ::NXml::TxmlXPathContextHolder xctx(xmlXPathNewContext(Impl_->Doc.Get()));
        Y_ENSURE(xctx, TUtils::GetLastXmlError());

        TErrorInfo err(TErrorInfo::Generic);
        ::NXml::TxmlXPathObjectHolder object(xmlXPathEvalExpression((const xmlChar*)xpath.c_str(), xctx.Get()));
        Y_ENSURE(object, err.Get());

        if (nullptr != object->nodesetval && 0 != object->nodesetval->nodeNr) {
            xmlNodeSetPtr ns = object->nodesetval;
            std::vector<TString> v;
            v.reserve(ns->nodeNr);

            for (int i = 0; i < ns->nodeNr; ++i) {
                Y_ENSURE(ns->nodeTab[i], TUtils::GetLastXmlError());
                v.push_back(NUtils::CreateStr(xpath, "[", i + 1, "]"));
            }

            return v;
        }

        return {};
    }
}
