#pragma once

#include <util/generic/string.h>
#include <util/generic/yexception.h>
#include <util/string/cast.h>

#include <memory>
#include <vector>

namespace NPassport::NUtils {
    class TFileLogger;
}

namespace NPassport::NXml {
    class TDtdSchema;
    class TXsdSchema;

    class TConfig: TMoveOnly {
    public:
        class TException: public yexception {
        };
        class TMissingException: public TException {
        };
        class TEmptyException: public TException {
        };
        class TBadValueException: public TException {
        };

    public:
        TConfig(TConfig&&) noexcept;
        ~TConfig();

        static TConfig ReadFromFile(const TString& file);
        static TConfig ReadFromMemory(const TStringBuf body);

        int AsInt(const TString& xpath) const;
        int AsInt(const TString& xpath, int defval) const;

        template <typename T>
        T AsNum(const TString& xpath) const {
            T res;
            const TString str = As<TString>(xpath);
            Y_ENSURE_EX(TryIntFromString<10>(str, res),
                        TBadValueException() << "config param '" << xpath
                                             << "' cannot be casted to int: '" << str << "'");
            return res;
        }

        template <typename T>
        T AsNum(const TString& xpath, T defval) const {
            try {
                return AsNum<T>(xpath);
            } catch (const TEmptyException&) {
                return defval;
            } catch (const TMissingException&) {
                return defval;
            }
        }

        enum class EFetchContent {
            Default,
            All,
        };

        TString AsString(const TString& xpath, EFetchContent content = EFetchContent::Default) const;
        TString AsString(const TString& xpath, const TString& defval, EFetchContent content = EFetchContent::Default) const;

        bool AsBool(const TString& xpath) const;
        bool AsBool(const TString& xpath, bool defval) const;

        double AsDouble(const TString& xpath) const;
        double AsDouble(const TString& xpath, double defval) const;

        template <typename T>
        T As(const TString& xpath) const {
            static_assert(std::is_integral_v<T>, "Given type is not supported");

            T res;
            const TString str = As<TString>(xpath);
            Y_ENSURE_EX(TryIntFromString<10>(str, res),
                        TBadValueException() << "config param '" << xpath
                                             << "' cannot be casted to int: '" << str << "'");
            return res;
        }

        template <>
        bool As<bool>(const TString& xpath) const;
        template <>
        double As<double>(const TString& xpath) const;
        template <>
        TString As<TString>(const TString& xpath) const;

        template <typename T>
        T As(const TString& xpath, T defval) const {
            try {
                return As<T>(xpath);
            } catch (const TEmptyException&) {
                return defval;
            } catch (const TMissingException&) {
                return defval;
            }
        }

        std::vector<TString> SubKeys(const TString& xpath) const;

        bool Contains(const TString& xpath) const;

        std::unique_ptr<NUtils::TFileLogger> CreateLogger(const TString& xpath) const;
        void InitCommonLog(const TString& xpath) const;

        using TError = TString;
        TError CheckSchema(const TDtdSchema& schema) const;
        TError CheckSchema(const TXsdSchema& schema) const;

    private:
        struct TImpl;
        TConfig(std::unique_ptr<TImpl> impl);

        TString AsStringImpl(const TString& xpath, EFetchContent content) const;

        std::unique_ptr<TImpl> Impl_;
    };
}
