#include "schema.h"

#include "error_info.h"
#include "xml_utils.h"

#include <util/generic/strbuf.h>
#include <util/generic/yexception.h>

namespace NPassport::NXml {
    TDtdSchema::TDtdSchema(const TStringBuf schema) {
        TErrorInfo err(TErrorInfo::Structured);

        xmlParserInputBufferPtr buf = xmlParserInputBufferCreateMem(schema.data(), schema.size(), XML_CHAR_ENCODING_UTF8);
        Y_ENSURE(buf, "Failed to create buf: " << TUtils::GetLastXmlError());

        // http://xmlsoft.org/html/libxml-parser.html#xmlIOParseDTD
        // xmlIOParseDTD: @input will be freed by the function in any case
        Schema_.Reset(xmlIOParseDTD(nullptr, buf, XML_CHAR_ENCODING_UTF8));
        Y_ENSURE(Schema_, "Failed to parse schema. " << err.Get());
    }

    TString TDtdSchema::Check(xmlDoc* doc) const {
        TErrorInfo err(TErrorInfo::Structured);

        ::NXml::TxmlValidCtxtPtr ctx(xmlNewValidCtxt());
        Y_ENSURE(ctx, "Failed to create valid ctx: " << TUtils::GetLastXmlError());

        const auto code = xmlValidateDtd(ctx.Get(), doc, Schema_.Get());
        Y_ENSURE(code >= 0, "Internal error occurred while validation XML: " << TUtils::GetLastXmlError());

        return code == 0 ? err.Get() : TString();
    }

    TXsdSchema::TXsdSchema(const TStringBuf schema) {
        TErrorInfo err(TErrorInfo::Structured);

        ::NXml::TxmlSchemaParserCtxtPtr parserCtx(xmlSchemaNewMemParserCtxt(schema.data(), schema.size()));
        Y_ENSURE(parserCtx, "Failed to create Schema parser context: " << TUtils::GetLastXmlError());

        Schema_.Reset(xmlSchemaParse(parserCtx.Get()));
        Y_ENSURE(Schema_, "Failed to parse schema. " << err.Get());
    }

    TString TXsdSchema::Check(xmlDoc* doc) const {
        TErrorInfo err(TErrorInfo::Structured);

        ::NXml::TxmlSchemaValidCtxtPtr ctxt(xmlSchemaNewValidCtxt(Schema_.Get()));
        Y_ENSURE(ctxt, "Failed to create validation context: " << TUtils::GetLastXmlError());

        const auto errorCode = xmlSchemaValidateDoc(ctxt.Get(), doc);

        Y_ENSURE(errorCode >= 0, "Internal error occurred while validation XML: " << TUtils::GetLastXmlError());
        return errorCode ? err.Get() : TString();
    }
}
