#include <passport/infra/libs/cpp/xml/config.h>
#include <passport/infra/libs/cpp/xml/schema.h>

#include <passport/infra/libs/cpp/dbpool/db_pool.h>
#include <passport/infra/libs/cpp/utils/log/file_logger.h>
#include <passport/infra/libs/cpp/utils/log/global.h>

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>

#include <util/stream/file.h>
#include <util/system/fs.h>

using namespace NPassport::NXml;
using namespace NPassport::NUtils;
using namespace NPassport;

static TString GetFilePath(const char* localname) {
    return (ArcadiaSourceRoot() + "/passport/infra/libs/cpp/xml/ut/" + localname).data();
}

static const TString missingFile(GetFilePath("no_such_file"));
static const TString wrongFile(GetFilePath("config_ut.cpp"));
static const TString configFile(GetFilePath("config.xml"));

Y_UNIT_TEST_SUITE(PasspConfig) {
    Y_UNIT_TEST(badconfigs) {
        UNIT_ASSERT_EXCEPTION_CONTAINS(TConfig::ReadFromFile(missingFile), TFileError, R"(can't open )");
        UNIT_ASSERT_EXCEPTION_CONTAINS(TConfig::ReadFromFile(wrongFile), yexception, "Failed to parse :line=1. errcode=4. message: Start tag expected, '<' not found");
    }

    Y_UNIT_TEST(xpathAccessContains) {
        TConfig conf = TConfig::ReadFromFile(configFile);

        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.Contains("/root]]"), yexception, "Invalid expression");
        UNIT_ASSERT(!conf.Contains("/root/missing_value"));
        UNIT_ASSERT(conf.Contains("/root/null_value"));
        UNIT_ASSERT(conf.Contains("/root/int_value"));
        UNIT_ASSERT(conf.Contains("/root/string_value"));
    }

    Y_UNIT_TEST(xpathAccessAsString) {
        TConfig conf = TConfig::ReadFromFile(configFile);

        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.AsString("/root]]"), yexception, "Invalid expression");
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.AsString("/root/missing_value"),
                                       TConfig::TMissingException,
                                       "nonexistent config param: /root/missing_value");
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.AsString("/root/null_value"),
                                       TConfig::TEmptyException,
                                       "config param is empty: /root/null_value");
        UNIT_ASSERT_VALUES_EQUAL("1234", conf.AsString("/root/int_value"));
        UNIT_ASSERT_VALUES_EQUAL("foo", conf.AsString("/root/string_value"));

        UNIT_ASSERT_VALUES_EQUAL("def_val", conf.AsString("/root]]", "def_val"));
        UNIT_ASSERT_VALUES_EQUAL("def_val", conf.AsString("/root/missing_value", "def_val"));
        UNIT_ASSERT_VALUES_EQUAL("def_val", conf.AsString("/root/null_value", "def_val"));
        UNIT_ASSERT_VALUES_EQUAL("1234", conf.AsString("/root/int_value", "def_val"));
        UNIT_ASSERT_VALUES_EQUAL("foo", conf.AsString("/root/string_value", "def_val"));

        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.As<TString>("/root]]"), yexception, "Invalid expression");
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.As<TString>("/root/missing_value"),
                                       TConfig::TMissingException,
                                       "nonexistent config param: /root/missing_value");
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.As<TString>("/root/null_value"),
                                       TConfig::TEmptyException,
                                       "config param is empty: /root/null_value");
        UNIT_ASSERT_VALUES_EQUAL("1234", conf.As<TString>("/root/int_value"));
        UNIT_ASSERT_VALUES_EQUAL("foo", conf.As<TString>("/root/string_value"));

        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.As<TString>("/root]]", "def_val"), yexception, "Invalid expression");
        UNIT_ASSERT_VALUES_EQUAL("def_val", conf.As<TString>("/root/missing_value", "def_val"));
        UNIT_ASSERT_VALUES_EQUAL("def_val", conf.As<TString>("/root/null_value", "def_val"));
        UNIT_ASSERT_VALUES_EQUAL("1234", conf.As<TString>("/root/int_value", "def_val"));
        UNIT_ASSERT_VALUES_EQUAL("foo", conf.As<TString>("/root/string_value", "def_val"));
    }

    Y_UNIT_TEST(xpathAccessAsNum) {
        TConfig conf = TConfig::ReadFromFile(configFile);

        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.AsInt("/root]]"), yexception, "Invalid expression");
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.AsInt("/root/missing_value"),
                                       TConfig::TMissingException,
                                       "nonexistent config param: /root/missing_value");
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.AsInt("/root/null_value"),
                                       TConfig::TEmptyException,
                                       "config param is empty: /root/null_value");
        UNIT_ASSERT_VALUES_EQUAL(1234, conf.AsInt("/root/int_value"));
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.AsInt("/root/string_value"),
                                       TConfig::TBadValueException,
                                       "config param '/root/string_value' cannot be casted to int: 'foo'");

        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.AsInt("/root]]", 666), yexception, "Invalid expression");
        UNIT_ASSERT_VALUES_EQUAL(666, conf.AsInt("/root/missing_value", 666));
        UNIT_ASSERT_VALUES_EQUAL(666, conf.AsInt("/root/null_value", 666));
        UNIT_ASSERT_VALUES_EQUAL(1234, conf.AsInt("/root/int_value", 666));
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.AsInt("/root/string_value", 666),
                                       TConfig::TBadValueException,
                                       "config param '/root/string_value' cannot be casted to int: 'foo'");

        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.As<int>("/root]]"), yexception, "Invalid expression");
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.As<int>("/root/missing_value"),
                                       TConfig::TMissingException,
                                       "nonexistent config param: /root/missing_value");
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.As<int>("/root/null_value"),
                                       TConfig::TEmptyException,
                                       "config param is empty: /root/null_value");
        UNIT_ASSERT_VALUES_EQUAL(1234, conf.As<int>("/root/int_value"));
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.As<int>("/root/string_value"),
                                       TConfig::TBadValueException,
                                       "config param '/root/string_value' cannot be casted to int: 'foo'");

        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.As<int>("/root]]", 666), yexception, "Invalid expression");
        UNIT_ASSERT_VALUES_EQUAL(666, conf.As<int>("/root/missing_value", 666));
        UNIT_ASSERT_VALUES_EQUAL(666, conf.As<int>("/root/null_value", 666));
        UNIT_ASSERT_VALUES_EQUAL(1234, conf.As<int>("/root/int_value", 666));
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.As<int>("/root/string_value", 666),
                                       TConfig::TBadValueException,
                                       "config param '/root/string_value' cannot be casted to int: 'foo'");
    }

    Y_UNIT_TEST(xpathAccessAsBool) {
        TConfig conf = TConfig::ReadFromFile(configFile);

        UNIT_ASSERT(conf.AsBool("/root/log/print-level"));
        UNIT_ASSERT(conf.AsBool("/root/log/print-level", false));
        UNIT_ASSERT(!conf.AsBool("/root/log/file"));
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.AsBool("/root/missing_value"),
                                       TConfig::TMissingException,
                                       "nonexistent config param: /root/missing_value");
        UNIT_ASSERT(conf.AsBool("/root/missing_value", true));
        UNIT_ASSERT(!conf.AsBool("/root/missing_value", false));

        UNIT_ASSERT(conf.As<bool>("/root/log/print-level"));
        UNIT_ASSERT(conf.As<bool>("/root/log/print-level", false));
        UNIT_ASSERT(!conf.As<bool>("/root/log/file"));
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.As<bool>("/root/missing_value"),
                                       TConfig::TMissingException,
                                       "nonexistent config param: /root/missing_value");
        UNIT_ASSERT(conf.As<bool>("/root/missing_value", true));
        UNIT_ASSERT(!conf.As<bool>("/root/missing_value", false));
    }

    Y_UNIT_TEST(xpathAccessSubKeys) {
        TConfig conf = TConfig::ReadFromFile(configFile);

        std::vector<TString> vec;
        UNIT_ASSERT_EXCEPTION_CONTAINS(conf.SubKeys("/root]]"), yexception, "Invalid expression");
        UNIT_ASSERT(vec.empty());

        vec = conf.SubKeys("/root/missing_value");
        UNIT_ASSERT(vec.empty());

        vec = conf.SubKeys("/root/null_value");
        UNIT_ASSERT_VALUES_EQUAL(1, vec.size());
        UNIT_ASSERT_VALUES_EQUAL("/root/null_value[1]", vec[0]);

        vec = conf.SubKeys("/root/int_value");
        UNIT_ASSERT_VALUES_EQUAL(1, vec.size());
        UNIT_ASSERT_VALUES_EQUAL("/root/int_value[1]", vec[0]);

        vec = conf.SubKeys("/root/string_value");
        UNIT_ASSERT_VALUES_EQUAL(3, vec.size());
        UNIT_ASSERT_VALUES_EQUAL("/root/string_value[1]", vec[0]);
        UNIT_ASSERT_VALUES_EQUAL("/root/string_value[2]", vec[1]);
        UNIT_ASSERT_VALUES_EQUAL("/root/string_value[3]", vec[2]);
    }

    Y_UNIT_TEST(createLog) {
        TConfig conf = TConfig::ReadFromFile(configFile);

        std::unique_ptr<TFileLogger> log = conf.CreateLogger("/root/log");
        UNIT_ASSERT(log.get());
        UNIT_ASSERT_VALUES_EQUAL((int)ILogger::ELevel::INFO, (int)log->GetLevel());
        log->Debug("debug: %s", "ON");
        log->Info("info: %d %s", 2, "test");
        log->Warning("warning!");
        log->Error("error: %o %x %X %#x", 11, 12, 13, 14);
        log.reset();

        {
            TFileInput log_file((TString("config_temp_log_file")));
            UNIT_ASSERT_VALUES_EQUAL("INFO: info: 2 test\n"
                                     "WARNING: warning!\n"
                                     "ERROR: error: 13 c D 0xe\n",
                                     log_file.ReadAll());
        }

        conf.InitCommonLog("/root/default_log");
        conf.InitCommonLog("/root/log");
        TLog::Warning("common warning");
        TDbPoolLog().Error("misterious dbpool error!");
        TLog::Reset();

        {
            TFileInput log_file((TString("config_temp_log_file")));
            UNIT_ASSERT_VALUES_EQUAL("INFO: info: 2 test\n"
                                     "WARNING: warning!\n"
                                     "ERROR: error: 13 c D 0xe\n"
                                     "WARNING: common warning\n",
                                     log_file.ReadAll());
        }
    }

    Y_UNIT_TEST(fromMemory) {
        TConfig config = TConfig::ReadFromMemory(R"(<doc><value>100500</value></doc>)");
        UNIT_ASSERT_VALUES_EQUAL(100500, config.AsInt("/doc/value"));
    }

    Y_UNIT_TEST(dtdSchema) {
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            TDtdSchema(R"(<?xml version="1.0" encoding="UTF-8"?>
                       <!ELEMENT config (pools?, http_daemon, unistat?, daemon?, component, daemon?)"),
            yexception,
            "Failed to parse schema. line=2. errcode=55. message: ContentDecl : ',' '|' or ')' expected\n"
            "line=2. errcode=73. message: expected '>'");

        const TConfig conf = TConfig::ReadFromFile(configFile);
        UNIT_ASSERT_VALUES_EQUAL(
            "line=2. node=root. errcode=504. message: Element root content does not follow the DTD, expecting (string_value | int_value | null_value | log | default_log)+, got (string_value int_value null_value string_value string_value log default_log default_db )\n"
            "line=17. node=default_db. errcode=534. message: No declaration for element default_db\n"
            "line=18. node=poolsize. errcode=534. message: No declaration for element poolsize\n"
            "line=19. node=get_timeout. errcode=534. message: No declaration for element get_timeout\n"
            "line=20. node=connect_timeout. errcode=534. message: No declaration for element connect_timeout\n"
            "line=21. node=query_timeout. errcode=534. message: No declaration for element query_timeout\n"
            "line=22. node=fail_threshold. errcode=534. message: No declaration for element fail_threshold\n"
            "line=23. node=db_driver. errcode=534. message: No declaration for element db_driver\n"
            "line=24. node=db_host. errcode=534. message: No declaration for element db_host\n"
            "line=25. node=db_port. errcode=534. message: No declaration for element db_port\n"
            "line=26. node=db_name. errcode=534. message: No declaration for element db_name\n"
            "line=27. node=db_user. errcode=534. message: No declaration for element db_user\n"
            "line=28. node=db_pass. errcode=534. message: No declaration for element db_pass\n"
            "line=29. node=extended. errcode=534. message: No declaration for element extended\n",
            conf.CheckSchema(TDtdSchema(
                R"(<?xml version="1.0" encoding="UTF-8"?>
                <!ELEMENT root (string_value|int_value|null_value|log|default_log)+>

                <!ELEMENT string_value (#PCDATA)>
                <!ELEMENT int_value (#PCDATA)>
                <!ELEMENT null_value EMPTY>

                <!ELEMENT log (level, print-level, file)>
                <!ELEMENT level (#PCDATA)>
                <!ELEMENT print-level (#PCDATA)>
                <!ELEMENT file (#PCDATA)>

                <!ELEMENT default_log (file)>
                <!ELEMENT file (#PCDATA)>
                )")));
        UNIT_ASSERT_VALUES_EQUAL(
            "",
            conf.CheckSchema(TDtdSchema(
                R"(<?xml version="1.0" encoding="UTF-8"?>
                <!ELEMENT root (string_value|int_value|null_value|log|default_log|default_db)+>

                <!ELEMENT string_value (#PCDATA)>
                <!ELEMENT int_value (#PCDATA)>
                <!ELEMENT null_value EMPTY>

                <!ELEMENT log (level, print-level, file)>
                <!ELEMENT level (#PCDATA)>
                <!ELEMENT print-level (#PCDATA)>
                <!ELEMENT file (#PCDATA)>

                <!ELEMENT default_log (file)>
                <!ELEMENT file (#PCDATA)>

                <!ELEMENT default_db (poolsize, get_timeout, connect_timeout, query_timeout, fail_threshold, db_driver, db_host, db_port, db_name, db_user, db_pass, extended)>
                <!ELEMENT poolsize (#PCDATA)>
                <!ELEMENT get_timeout (#PCDATA)>
                <!ELEMENT connect_timeout (#PCDATA)>
                <!ELEMENT query_timeout (#PCDATA)>
                <!ELEMENT fail_threshold (#PCDATA)>
                <!ELEMENT db_driver (#PCDATA)>
                <!ELEMENT db_host (#PCDATA)>
                <!ELEMENT db_port (#PCDATA)>
                <!ELEMENT db_name (#PCDATA)>
                <!ELEMENT db_user (#PCDATA)>
                <!ELEMENT db_pass (#PCDATA)>
                <!ELEMENT extended (#PCDATA)>
                )")));
    }

    Y_UNIT_TEST(xsdSchema) {
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            TXsdSchema(R"(<xs:schema xmlns:xs="http://www.w3.org/2001/XMLSchema">
    <xs:element name="string_value" type="Str"/>
    <xs:complexType name="Str">
        <xs:all>
            <xs:element name="name" type="xs:string"/>
            <xs:element name="population" type="xs:decimal"/>
        </xs:all>
    </xs:complexType>)"),
            yexception,
            "Failed to parse schema. line=8. errcode=77. message: Premature end of data in tag schema line 1\n"
            "line=0. errcode=3067. message: Failed to parse the XML resource 'in_memory_buffer'");

        const TConfig conf = TConfig::ReadFromFile(configFile);
        UNIT_ASSERT_VALUES_EQUAL(
            "line=3. node=string_value. errcode=1871. message: Element 'string_value': This element is not expected. Expected is one of ( name, population ).\n",
            conf.CheckSchema(TXsdSchema(
                R"(<xs:schema xmlns:xs="http://www.w3.org/2001/XMLSchema">
                     <xs:element name="root" type="Root"/>
                     <xs:complexType name="Root">
                         <xs:all>
                             <xs:element name="name" type="xs:string"/>
                             <xs:element name="population" type="xs:decimal"/>
                         </xs:all>
                     </xs:complexType>
                 </xs:schema>)")));

        UNIT_ASSERT_VALUES_EQUAL(
            "line=2. node=root. errcode=1841. message: Character content is not allowed, because the content type is emptyline=2. node=root. errcode=1841. message: Element content is not allowed, because the content type is empty",
            conf.CheckSchema(TXsdSchema(
                R"(<xs:schema xmlns:xs="http://www.w3.org/2001/XMLSchema">
                     <xs:element name="root" type="Root"/>
                     <xs:complexType name="Root">
                         <xs:all>
                         </xs:all>
                     </xs:complexType>
                 </xs:schema>)")));

        UNIT_ASSERT_VALUES_EQUAL(
            "line=4. node=int_value. errcode=1871. message: Element 'int_value': This element is not expected.\n",
            conf.CheckSchema(TXsdSchema(
                R"(<xs:schema xmlns:xs="http://www.w3.org/2001/XMLSchema">
                     <xs:element name="root" type="Root"/>
                     <xs:complexType name="Root">
                         <xs:all>
                             <xs:element name="string_value" type="xs:string" minOccurs="0"/>
                         </xs:all>
                     </xs:complexType>
                 </xs:schema>)")));

        UNIT_ASSERT_VALUES_EQUAL(
            "",
            conf.CheckSchema(TXsdSchema(
                R"(<xs:schema attributeFormDefault="unqualified" elementFormDefault="qualified" xmlns:xs="http://www.w3.org/2001/XMLSchema">
                <xs:element name="root">
                  <xs:complexType>
                    <xs:choice maxOccurs="unbounded" minOccurs="0">
                      <xs:element type="xs:string" name="string_value"/>
                      <xs:element type="xs:string" name="null_value"/>
                      <xs:element type="xs:short" name="int_value"/>
                      <xs:element name="log">
                        <xs:complexType>
                          <xs:all>
                            <xs:element type="xs:string" name="print-level"/>
                            <xs:element type="xs:string" name="level"/>
                            <xs:element type="xs:string" name="file">
                              <xs:annotation>
                                <xs:documentation>default</xs:documentation>
                              </xs:annotation>
                            </xs:element>
                          </xs:all>
                        </xs:complexType>
                      </xs:element>
                      <xs:element name="default_log">
                        <xs:complexType>
                          <xs:all>
                            <xs:element type="xs:string" name="file"/>
                          </xs:all>
                        </xs:complexType>
                      </xs:element>
                      <xs:element name="default_db">
                        <xs:complexType>
                          <xs:all>
                            <xs:element type="xs:byte" name="poolsize"/>
                            <xs:element type="xs:short" name="get_timeout"/>
                            <xs:element type="xs:short" name="connect_timeout"/>
                            <xs:element type="xs:short" name="query_timeout"/>
                            <xs:element type="xs:string" name="db_driver"/>
                            <xs:element type="xs:short" name="fail_threshold"/>
                            <xs:element type="xs:string" name="db_host"/>
                            <xs:element type="xs:byte" name="db_port"/>
                            <xs:element type="xs:string" name="db_name"/>
                            <xs:element type="xs:string" name="db_user"/>
                            <xs:element type="xs:string" name="db_pass"/>
                            <xs:element type="xs:string" name="extended"/>
                          </xs:all>
                        </xs:complexType>
                      </xs:element>
                    </xs:choice>
                  </xs:complexType>
                </xs:element>
              </xs:schema>)")));
    }
}
