#include "xml_utils.h"

#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <contrib/libs/libxml/include/libxml/xmlerror.h>
#include <contrib/libs/libxml/include/libxml/xpath.h>

namespace NPassport::NXml {
    TUtils::TUtils() {
        xmlInitParser();

        xmlLoadExtDtdDefaultValue = 1;
        xmlSubstituteEntitiesDefault(0);
    }

    TUtils::~TUtils() {
        xmlCleanupParser();
    }

    TString TUtils::GetLastXmlError() {
        const char* message = "unknown xml error";
        xmlErrorPtr err = xmlGetLastError();
        if (err && err->message) {
            message = err->message;
        }

        TString res(message);
        xmlResetLastError();
        return res;
    }

    TString
    TUtils::Value(xmlNodePtr node) {
        assert(node);
        xmlNodePtr child = node->children;
        if (child && xmlNodeIsText(child) && child->content) {
            return TString((const char*)child->content);
        }
        return TString();
    }

    const char*
    TUtils::Value(xmlAttrPtr attr) {
        assert(attr);
        xmlNodePtr child = attr->children;
        if (child && xmlNodeIsText(child) && child->content) {
            return (const char*)child->content;
        }
        return nullptr;
    }

    TString
    TUtils::ValueAllContent(xmlNodePtr node) {
        assert(node);
        xmlChar* ctp = xmlNodeGetContent(node);
        if (!ctp) {
            return TString();
        }
        TString aux((const char*)ctp);
        xmlFree(ctp);
        return aux;
    }

    const char*
    TUtils::AttrValue(xmlNodePtr node, const char* name) {
        assert(node);
        xmlAttrPtr attr = xmlHasProp(node, (const xmlChar*)name);
        return attr ? Value(attr) : nullptr;
    }

    static const TString QUO("&quot;");
    static const TString APO("&apos;");
    static const TString LANGLE("&lt;");
    static const TString RANGLE("&gt;");
    static const TString AMP("&amp;");

    TString&
    TUtils::Escape(char c, TString& buf) {
        switch (c) {
            case '<':
                buf.append(LANGLE);
                break;
            case '>':
                buf.append(RANGLE);
                break;
            case '&':
                buf.append(AMP);
                break;
            case '"':
                buf.append(QUO);
                break;
            case '\'':
                buf.append(APO);
                break;
            case '\x09':
            case '\x0A':
            case '\x0D':
                buf.push_back(c);
                break;
            case '\x7F':
            case '\x98':
                buf.push_back('?');
                break;
            default:
                buf.push_back((unsigned)c < 0x20 ? '?' : c);
                break;
        }
        return buf;
    }

    TString&
    TUtils::Escape(const TString& src, TString& buf) {
        buf.clear();
        buf.reserve(src.size() + 32);
        for (const char c : src) {
            Escape(c, buf);
        }
        return buf;
    }

    TString
    TUtils::Escape(const TString& src) {
        TString buf;
        buf.reserve(src.size() + 32);
        for (char c : src) {
            Escape(c, buf);
        }
        return buf;
    }

    TString TUtils::EscapeUtf8(const TStringBuf src) {
        TString buf;
        buf.reserve(src.size() + 10);
        EscapeUtf8(src, buf);
        return buf;
    }

    TString& TUtils::EscapeUtf8(const TStringBuf src, TString& buf) {
        TStringBuf::const_iterator it = src.cbegin();

        while (it != src.cend()) {
            size_t l = NUtils::Utf8CharLength(it, src.cend());
            if (l > 1) {
                for (size_t i = 0; i < l; ++i) {
                    buf.push_back(*it++);
                }
            } else if (l == 1) {
                Escape(*it++, buf);
            } else {
                ++it;
            }
        }

        return buf;
    }

    TString&
    TUtils::StrictEscapeUtf8(const TString& src, TString& buf) {
        buf.clear();
        buf.reserve(src.size() * 2);
        TString::const_iterator it = src.cbegin();
        while (it != src.cend()) {
            size_t l = NUtils::Utf8CharLength(it, src.cend());
            if (l > 1) {
                for (size_t i = 0; i < l; ++i) {
                    buf.push_back(*it++);
                }
            } else if (l == 1) {
                Escape(*it++, buf);
            } else {
                throw yexception() << src << " is not a valid UTF-8 string";
            }
        }
        return buf;
    }
}
