#include "yp_cookie_parser.h"

#include <library/cpp/string_utils/quote/quote.h>

#include <util/generic/algorithm.h>
#include <util/generic/vector.h>
#include <util/string/join.h>


// https://wiki.yandex-team.ru/Cookies/Y/
// format: expire1.name1.value1#expire2.name2.value2#expire3.name3.value3

namespace NPassport::NYpCookie {
    TString TYpCookieElem::Serialize() const {
        return Join(".", Expire, Name, Value);
    }

    IOutputStream& operator <<(IOutputStream& out, const TYpCookieElem& ypCookieElem) {
        out << ypCookieElem.Expire << "." << ypCookieElem.Name << "." << CGIEscapeRet(ypCookieElem.Value);
        return out;
    }

    bool TryParseYpCookiePart(TStringBuf cookie, TStringBuf& expireField, TStringBuf& nameField, TStringBuf& valueField) {
        if (!cookie.NextTok('.', expireField) || expireField.empty()) {
            return false;
        }
        if (!cookie.NextTok('.', nameField) || nameField.empty()) {
            return false;
        }
        if (cookie.empty()) {
            return false;
        }

        valueField = cookie;
        return true;
    }

    bool TryGetYandexPermanentCookieKey(const TStringBuf& key, TStringBuf ypCookie, TString& cookiePartValue) {
        TStringBuf nextCookie;
        TStringBuf expireField;
        TStringBuf nameField;
        TStringBuf valueField;

        while (ypCookie.NextTok('#', nextCookie)) {
            if (!TryParseYpCookiePart(nextCookie, expireField, nameField, valueField)) {
                return false;
            }

            if (nameField == key) {
                cookiePartValue = valueField;
                CGIUnescape(cookiePartValue); // https://wiki.yandex-team.ru/Cookies/Y/#problemy
                return true;
            }
        }
        return false;
    }

    bool TryParseYpCookie(TStringBuf ypCookie, TVector<TYpCookieElem>& ypCookieParsed) {
        if (ypCookie.empty()) {
            return false;
        }

        ypCookieParsed.clear();
        size_t ypCookiePartsCount = CountIf(ypCookie.begin(), ypCookie.end(), [](char c){return c =='#';});
        ypCookieParsed.reserve(ypCookiePartsCount + 1);

        TStringBuf nextYpCookiePart;
        while (ypCookie.NextTok('#', nextYpCookiePart)) {
            TStringBuf expireField;
            TStringBuf nameField;
            TStringBuf valueField;

            if (!TryParseYpCookiePart(nextYpCookiePart, expireField, nameField, valueField)) {
                return false;
            }
            ypCookieParsed.emplace_back(TYpCookieElem{TString(expireField), TString(nameField), CGIUnescapeRet(TString(valueField))});
        }
        return true;
    }

    TString SerializeYpCookie(const TVector<TYpCookieElem>& ypCookieParsed) {
        if (ypCookieParsed.empty()) {
            return "";
        }
        TStringStream stream;
        stream << MakeRangeJoiner("#", ypCookieParsed);
        return stream.Str();
    }

    bool TryUpdateYpCookie(const TYpCookieElem& ypCookieElem, TStringBuf ypCookie, TString& ypCookieUpdated) {
        if (ypCookie.empty()) {
            ypCookieUpdated = ypCookieElem.Serialize();
            return true;
        }

        TVector<TYpCookieElem> ypCookieParsed;
        if (!TryParseYpCookie(ypCookie, ypCookieParsed)) {
            return false;
        }

        for (auto& cookie : ypCookieParsed) {
            if (cookie.Name == ypCookieElem.Name) {
                cookie.Expire = ypCookieElem.Expire;
                cookie.Value = ypCookieElem.Value;

                ypCookieUpdated = SerializeYpCookie(ypCookieParsed);
                return true;
            }
        }
        ypCookieParsed.insert(ypCookieParsed.end(), ypCookieElem);
        ypCookieUpdated = SerializeYpCookie(ypCookieParsed);
        return true;
    }
}
