#pragma once

#include "query.h"

#include <span>

namespace NPassport::NYt {
    struct TBatchSettings {
        size_t Limit = 0;
        TDuration Timeout = TDuration::Seconds(10);
    };

    class TBatch {
    public:
        struct TQueryResult {
            NThreading::TFuture<NYt::TYtClient::TWriteResult> Future;
            TString TableName;
            size_t Rows = 0;
            size_t Size = 0;
            TInstant StartTime = TInstant::Now();
        };

        TBatch(const TBatchSettings& settings,
               TYtClient& client)
            : Settings_(settings)
            , Client_(client)
        {
            Y_ENSURE(Settings_.Limit > 0, "limit == 0 is illegal size for batch");
        }

        template <typename Func> // void(TReqResult&)
        void Send(const TString& table,
                  const IQueryConverter& builder,
                  Func postSentFunc) {
            const size_t size = builder.size();
            SentReqs_.reserve(SentReqs_.size() + size / Settings_.Limit + 1);

            size_t offset = 0;
            while (offset < size) {
                const size_t rows = std::min(size - offset, Settings_.Limit);

                NYt::TWriteQuery req = builder.Convert(table, offset, rows);
                TQueryResult res{
                    .TableName = table,
                    .Rows = rows,
                    .Size = req.Size,
                };
                res.Future = Client_.Write(std::move(req));
                postSentFunc(res);

                SentReqs_.push_back(std::move(res));

                offset += Settings_.Limit;
            }
        }

        template <typename Func> // bool(TReqResult&)
        bool WaitResult(Func resultProccessFunc) {
            for (TQueryResult& req : SentReqs_) {
                Y_ENSURE(req.Future.Wait(Settings_.Timeout)); // TODO

                if (!resultProccessFunc(req)) {
                    return false;
                }
            }

            return true;
        }

    private:
        const TBatchSettings Settings_;
        TYtClient& Client_;
        std::vector<TQueryResult> SentReqs_;
    };
}
