#pragma once

#include "yt_client.h"

#include <util/stream/str.h>

#include <map>
#include <typeindex>
#include <vector>

#include <span>

namespace NPassport::NYt {
    class IQueryConverter;
    using TQueryConverterPtr = std::unique_ptr<IQueryConverter>;

    class IQueryConverter {
    public:
        virtual ~IQueryConverter() = default;

        virtual size_t size() const = 0;

        virtual NYt::TWriteQuery Convert(
            const TString& table,
            size_t offset,
            size_t count) const = 0;

        virtual TString DebugString() const = 0;

        virtual void Merge(TQueryConverterPtr) = 0;

        /**
         * This crutch allowes to ensure that merge() was called for same types.
         * dynamic_cast<>() could be used instead of std::type_index,
         *   but benchmark showes that std::type_index would be faster
         *
         *----------- dynamic_cast ---------------
         * samples:       90499
         * iterations:    5055704290
         * iterations hr:    5.06G
         * run time:      15.00928532
         * per iteration: 6.417057051 cycles
         *----------- typeid ---------------
         * samples:       136702
         * iterations:    11535665778
         * iterations hr:    11.5G
         * run time:      15.01390241
         * per iteration: 2.773405388 cycles
         */
        virtual std::type_index GetType() const = 0;
    };

    template <typename Value>
    class TBaseConverter: public IQueryConverter {
    public:
        void Reserve(size_t size) {
            Data_.reserve(size);
        }

        void Add(Value&& value) {
            Data_.push_back(std::move(value));
        }

        size_t size() const override {
            return Data_.size();
        }

        TString DebugString() const override {
            TStringStream res;

            for (const Value& v : Data_) {
                res << v << "#";
            }

            return res.Str();
        }

        std::type_index GetType() const override {
            return std::type_index(typeid(Value));
        }

        void Merge(TQueryConverterPtr o) override {
            Y_VERIFY(std::type_index(typeid(Value)) == o->GetType(),
                     "Merging for different types is not supported");
            TBaseConverter<Value>& other = *static_cast<TBaseConverter<Value>*>(o.get());

            Data_.reserve(Data_.size() + other.Data_.size());
            for (Value& q : other.Data_) {
                Data_.push_back(std::move(q));
            }
        }

    protected:
        std::span<const Value> GetSpan(size_t offset, size_t count) const {
            std::span s(Data_);
            return s.subspan(offset, count);
        }

    protected:
        std::vector<Value> Data_;
    };

    template <typename Table>
    struct TWriteQueries {
        void Merge(TWriteQueries&& o) {
            if (ByTable.empty()) {
                ByTable = std::move(o.ByTable);
                return;
            }

            for (auto& [table, conv] : o.ByTable) {
                if (!conv) {
                    continue;
                }

                auto it = ByTable.try_emplace(
                                     table,
                                     TQueryConverterPtr())
                              .first;
                if (!it->second) {
                    it->second = std::move(conv);
                    continue;
                }

                it->second->Merge(std::move(conv));
            }

            o.ByTable.clear();
        }

        std::map<Table, TQueryConverterPtr> ByTable;
    };
}
