#include "table_creator.h"

#include <passport/infra/libs/cpp/utils/log/global.h>

namespace NPassport::NYt {
    TTableCreator::TTableCreator(TYtClientFactoryPtr factory,
                                 const TTableCreatorSettings& settings)
        : Factory_(std::move(factory))
        , Settings_(settings)
        // create tables for 2 periods ago: to store all current data in tables
        , NextRequired_(TPeriodicTable(Settings_.PeriodType, TInstant::Now()).Prev().Prev())
    {
        Create();
        Worker_ = std::make_unique<NUtils::TRegularTask>(
            [this]() { Run(); },
            Settings_.CheckPeriod,
            "yt_tc");
    }

    TTableCreator::~TTableCreator() = default;

    void TTableCreator::Run() {
        if (NextRequired_.GetTimeToCreate() < TInstant::Now()) {
            Sleep(TDuration::MilliSeconds(rand() % Settings_.CheckPeriod.MilliSeconds()));
            Create();
        }
    }

    void TTableCreator::Create() {
        // Every new table would be created for next period.
        // Also we need to increase it's lifetime.
        // For example: daily tables with data ttl==7days.
        //   Table for 2020-01-10 should be created at 2020-01-09
        //   and should live until 2020-01-18.
        Settings_.CreateSettings.ExpirationTime =
            NextRequired_.GetStart() +
            NextRequired_.GetPeriod() + // to keep data available: for day/month part too
            Settings_.CreateSettings.TableTtl;

        TYtClient::TCreateResult result;
        try {
            result = Factory_->Create().CreateDynamicTable(
                GetNextTableName(),
                Settings_.CreateSettings);
        } catch (const std::exception& e) {
            TLog::Error() << "TableCreator: exception: " << e.what();
        }

        auto switchToNext = [this]() {
            NextRequired_ = NextRequired_.Next();
            TLog::Debug() << "TableCreator: next table: " << GetNextTableName();
        };

        switch (result.Status) {
            case TYtClient::TCreateResult::Ok:
                TLog::Info() << "TableCreator: " << result.Msg;
                switchToNext();
                break;
            case TYtClient::TCreateResult::AlreadyExists:
                TLog::Info() << "TableCreator: already exists: " << result.Msg;
                switchToNext();
                break;
            case TYtClient::TCreateResult::Error:
                TLog::Warning() << "TableCreator: failed to create table: " << result.Msg;
                break;
        }
    }

    TString TTableCreator::GetNextTableName() const {
        return Settings_.TablePrefix + NextRequired_.TableName();
    }
}
