#pragma once

#include <library/cpp/threading/future/future.h>

#include <util/datetime/base.h>
#include <util/generic/string.h>

#include <memory>
#include <optional>
#include <variant>
#include <vector>

namespace NPassport::NYt {
    class TYtClientFactoryImpl;
    class TYtClientImpl;

    struct TTableSchemaImpl;
    struct TWriteQueryImpl;

    struct TYtSettings {
        TString OauthToken;
        TString RpcCodec = "lz4";
        TString Cluster = "hahn";
        TString CypressDir;
        int LogLevel = 4;
        TString LogFile;
    };

    struct TTableSchema {
        TTableSchema();
        TTableSchema(TTableSchema&&) noexcept;
        TTableSchema(const TTableSchema&);
        ~TTableSchema();
        TTableSchema& operator=(TTableSchema&&) noexcept;
        TTableSchema& operator=(const TTableSchema&);

        std::shared_ptr<TTableSchemaImpl> Impl;
    };

    class TWriteQuery {
    public:
        TWriteQuery();
        TWriteQuery(TWriteQuery&&) noexcept;
        ~TWriteQuery();

        size_t Size = 0;
        std::unique_ptr<TWriteQueryImpl> Impl;
    };

    class TYtClient {
    protected:
        TYtClient();

    public:
        TYtClient(std::unique_ptr<TYtClientImpl> impl);
        TYtClient(TYtClient&&) noexcept;
        virtual ~TYtClient();

        struct TCreateSettings {
            // const attrs
            TDuration TableTtl = TDuration::Days(1);
            TString Codec = "lz4";
            TString ErasureCodec;
            size_t DesiredTabletCount = 0;
            bool OptimizeForScan = false;
            bool AtomicityFull = false;
            bool StoreChecksum = false;
            std::optional<ui32> MinDataVersions;
            std::optional<ui32> MaxDataVersions;
            std::optional<ui64> MinDataTtl; // MilliSeconds
            std::optional<ui64> MaxDataTtl; // MilliSeconds
            bool MergeRowsOnFlush = true;

            TTableSchema Schema;

            TDuration Timeout = TDuration::Seconds(7);

            // should be modified for every call
            TInstant ExpirationTime;
        };

        struct TCreateResult {
            enum EStatus {
                Ok,
                AlreadyExists,
                Error,
            } Status = Error;
            TString Msg;
        };
        virtual TCreateResult CreateDynamicTable(const TString& path,
                                                 const TCreateSettings& settings);

        struct TWriteResult {
            enum EStatus {
                Ok,
                Error,
            } Status = Error;
            TString Msg;
        };
        virtual NThreading::TFuture<TWriteResult> Write(TWriteQuery request);

        using TSelectResult = std::vector<std::vector<TString>>;
        virtual NThreading::TFuture<TSelectResult> Select(const TString& query); // for debug only

    private:
        std::unique_ptr<TYtClientImpl> Impl_;
    };

    class TYtClientFactory {
    public:
        TYtClientFactory(const TYtSettings& settings);
        ~TYtClientFactory(); // we need to reset all yt pointers and after that call Shutdown()

        TYtClient Create() const;
        static void RotateLogs();

    private:
        std::unique_ptr<TYtClientFactoryImpl> Impl_;
    };
    using TYtClientFactoryPtr = std::shared_ptr<TYtClientFactory>;
}
