#include "yt_impl.h"

#include <yt/yt/client/api/client.h>
#include <yt/yt/client/api/rowset.h>
#include <yt/yt/client/api/transaction.h>
#include <yt/yt/client/table_client/config.h>
#include <yt/yt/client/table_client/helpers.h>
#include <yt/yt/client/table_client/name_table.h>
#include <yt/yt/client/table_client/row_buffer.h>
#include <yt/yt/core/logging/config.h>
#include <yt/yt/core/logging/log_manager.h>
#include <yt/yt/core/ytree/fluent.h>

#include <util/string/builder.h>

namespace NPassport::NYt {
    template <class T>
    static TString GetErrorMsg(const T& fut) {
        TStringStream s;

        if (!fut.Get().IsOK()) {
            s << fut.Get().GetMessage();
            for (const auto& err : fut.Get().InnerErrors()) {
                s << ": " << ToString(err) << ")";
            }
            s << ". ";
        }

        return s.Str();
    }

    TYtClientImpl::TYtClientImpl(NYT::NApi::IClientPtr cl, const TYtSettings& settings)
        : Cl_(std::move(cl))
        , Settings_(settings)
    {
    }

    NThreading::TFuture<TYtClient::TSelectResult> TYtClientImpl::Select(const TString& query) {
        NYT::TFuture<NYT::NApi::TSelectRowsResult> result = Cl_->SelectRows(query);

        NThreading::TPromise promise = NThreading::NewPromise<TYtClient::TSelectResult>();
        result.Subscribe(BIND(
            [promise](const NYT::TErrorOr<NYT::NApi::TSelectRowsResult>& e) mutable {
                if (!e.IsOK()) {
                    promise.SetException(ToString(e));
                    return;
                }

                const auto& rows = e.Value().Rowset->GetRows();

                TYtClient::TSelectResult res;
                res.reserve(rows.size());

                for (const auto& row : rows) {
                    std::vector<TString> v;
                    v.reserve(row.GetCount());

                    for (const auto& col : row) {
                        v.push_back(ToString(col));
                    }

                    res.push_back(std::move(v));
                }

                promise.SetValue(std::move(res));
            }));

        return promise.GetFuture();
    }

    template <class T>
    static void SetBadResponse(const T& e, NThreading::TPromise<TYtClient::TWriteResult> promise) {
        TYtClient::TWriteResult res;
        res.Status = TYtClient::TWriteResult::Error;
        res.Msg = ToString(e);
        promise.SetValue(std::move(res));
    }

    static const TString OK_ = "OK";
    NThreading::TFuture<TYtClient::TWriteResult> TYtClientImpl::Write(TWriteQuery request) {
        NYT::NApi::TTransactionStartOptions opts;
        opts.Atomicity = NYT::NTransactionClient::EAtomicity::None;
        NYT::TFuture<NYT::NApi::ITransactionPtr> tr =
            Cl_->StartTransaction(NYT::NTransactionClient::ETransactionType::Tablet, opts);

        NThreading::TPromise<TYtClient::TWriteResult> promise = NThreading::NewPromise<TYtClient::TWriteResult>();
        tr.Subscribe(BIND(
            [this, promise, req = std::move(request)](const NYT::TErrorOr<NYT::NApi::ITransactionPtr>& e) mutable {
                if (!e.IsOK()) {
                    SetBadResponse(e, promise);
                    return;
                }

                for (TWriteSubQuery& pt : req.Impl->Subqueries) {
                    e.Value()->WriteRows(Settings_.CypressDir + pt.Path,
                                         pt.NameTable,
                                         std::move(pt.Range));
                }

                NYT::TFuture<NYT::NApi::TTransactionCommitResult> result = e.Value()->Commit();
                result.Subscribe(BIND(
                    [promise](const NYT::TErrorOr<NYT::NApi::TTransactionCommitResult>& e) mutable {
                        if (!e.IsOK()) {
                            SetBadResponse(e, promise);
                            return;
                        }

                        TYtClient::TWriteResult res;
                        res.Msg = OK_;
                        res.Status = TYtClient::TWriteResult::Ok;
                        promise.SetValue(std::move(res));
                    }));
            }));

        return promise.GetFuture();
    }

    TYtClient::TCreateResult TYtClientImpl::CreateTable(const TString& p,
                                                        const TYtClient::TCreateSettings& settings) {
        const TString path = Settings_.CypressDir + p;

        TYtClient::TCreateResult res;

        TString createErr = Create(path, settings);
        res.Msg = createErr;
        if (createErr.Contains("already exists")) { // TODO
        }

        TString alterErr = Alter(path, settings); // TODO
        res.Msg.append(alterErr);
        if (alterErr.Contains("Cannot change table schema since not all tablets are unmounted")) {
            res.Status = TYtClient::TCreateResult::AlreadyExists;
            return res;
        }
        if (!alterErr.empty()) {
            res.Status = TYtClient::TCreateResult::Error;
            return res;
        }

        TString mountErr = Mount(path, settings.Timeout);

        if (mountErr) {
            res.Msg.append(mountErr);
            res.Status = TYtClient::TCreateResult::Error;
        } else {
            res.Msg = TStringBuilder() << "success: " << path;
            res.Status = TYtClient::TCreateResult::Ok;
        }

        return res;
    }

    static NYT::NYTree::IAttributeDictionaryPtr TranslateToAttrs(const TYtClient::TCreateSettings& settings) {
        auto res = NYT::NYTree::CreateEphemeralAttributes();

        if (settings.ExpirationTime) {
            res->Set("expiration_time", settings.ExpirationTime.MilliSeconds());
        }
        if (settings.Codec) {
            res->Set("compression_codec", settings.Codec);
        }
        if (settings.ErasureCodec) {
            res->Set("erasure_codec", settings.ErasureCodec);
        }

        res->Set("optimize_for", settings.OptimizeForScan ? "scan" : "lookup");
        res->Set("atomicity", settings.AtomicityFull ? "full" : "none");

        if (settings.MinDataVersions) {
            res->Set("min_data_versions", *settings.MinDataVersions);
        }
        if (settings.MaxDataVersions) {
            res->Set("max_data_versions", *settings.MaxDataVersions);
        }
        if (settings.MinDataTtl) {
            res->Set("min_data_ttl", *settings.MinDataTtl);
        }
        if (settings.MaxDataTtl) {
            res->Set("max_data_ttl", *settings.MaxDataTtl);
        }

        res->Set("merge_rows_on_flush", settings.MergeRowsOnFlush);
        if (settings.DesiredTabletCount) {
            res->Set("desired_tablet_count", settings.DesiredTabletCount);
        }

        if (settings.StoreChecksum) {
            res->Set("chunk_writer", NYT::NYson::TYsonString(TStringBuf("{erasure_store_original_block_checksums=%true}")));
        }

        return res;
    }

    TString TYtClientImpl::Create(const TString& path,
                                  const TYtClient::TCreateSettings& settings) {
        NYT::NApi::TCreateNodeOptions opts;
        opts.Attributes = TranslateToAttrs(settings);

        NYT::TFuture<NYT::NCypressClient::TNodeId> result =
            Cl_->CreateNode(path, NYT::NObjectClient::EObjectType::Table, opts);
        Y_ENSURE(result.Wait(settings.Timeout));

        return GetErrorMsg(result);
    }

    TString TYtClientImpl::Alter(const TString& path,
                                 const TYtClient::TCreateSettings& settings) {
        NYT::NApi::TAlterTableOptions opts;
        opts.Schema = settings.Schema.Impl->Schema;
        opts.Dynamic = true;

        NYT::TFuture<void> result = Cl_->AlterTable(path, opts);
        Y_ENSURE(result.Wait(settings.Timeout));

        return GetErrorMsg(result);
    }

    TString TYtClientImpl::Mount(const TString& path,
                                 TDuration timeout) {
        NYT::TFuture<void> result = Cl_->MountTable(path);
        Y_ENSURE(result.Wait(timeout));

        return GetErrorMsg(result);
    }

    TYtClientFactoryImpl::TYtClientFactoryImpl(const TYtSettings& settings)
        : ClientSettings_(CreateOptions(settings.OauthToken))
        , Settings_(settings)
    {
        if (!Settings_.CypressDir.EndsWith("/")) {
            Settings_.CypressDir.push_back('/');
        }

        InitLogger();

        NYT::NApi::NRpcProxy::TConnectionConfigPtr cfg = NYT::New<NYT::NApi::NRpcProxy::TConnectionConfig>();
        cfg->ClusterUrl = Settings_.Cluster;
        if (settings.RpcCodec) {
            cfg->ResponseCodec = NYT::ParseEnum<NYT::NCompression::ECodec>(settings.RpcCodec);
            cfg->RequestCodec = cfg->ResponseCodec;
            cfg->EnableLegacyRpcCodecs = false;
        }
        Connection_ = NYT::NApi::NRpcProxy::CreateConnection(cfg);
    }

    TYtClientImpl TYtClientFactoryImpl::Create() const {
        return TYtClientImpl(Connection_->CreateClient(ClientSettings_), Settings_);
    }

    const TYtSettings& TYtClientFactoryImpl::GetSettings() const {
        return Settings_;
    }

    void TYtClientFactoryImpl::RotateLogs() {
        NYT::NLogging::TLogManager::Get()->Reopen();
    }

    void TYtClientFactoryImpl::InitLogger() {
        NYT::NLogging::TLogManagerConfigPtr log =
            Settings_.LogFile ? NYT::NLogging::TLogManagerConfig::CreateLogFile(Settings_.LogFile)
                              : NYT::New<NYT::NLogging::TLogManagerConfig>();

        if (log->Rules.empty()) {
            log->Rules.push_back(NYT::New<NYT::NLogging::TRuleConfig>());
        }
        NYT::NLogging::TRuleConfig& rule = *log->Rules[0];
        rule.MinLevel = (NYT::NLogging::ELogLevel)Settings_.LogLevel;
        rule.ExcludeCategories = {
            "Bus",
            "Concurrency",
            "RpcProxyClient",
        };

        NYT::NLogging::TLogManager::Get()->Configure(log);
    }

    NYT::NApi::TClientOptions TYtClientFactoryImpl::CreateOptions(const TString& oauthToken) {
        NYT::NApi::TClientOptions res;
        res.Token = oauthToken;
        return res;
    }
}
