package task

import (
	"context"
	"time"

	"github.com/cenkalti/backoff/v4"

	"a.yandex-team.ru/passport/shared/golibs/logger"
)

type Status int

const (
	StatusSuccess Status = iota
	StatusIdle
	StatusFailed
)

type RunFunc = func() (Status, error)

type RegularTaskSettings struct {
	Name string

	Period         time.Duration
	IdleBackoff    backoff.BackOff
	FailBackoff    backoff.BackOff
	BufferCapacity int

	Ctx context.Context
}

type RegularTask struct {
	runFunc  RunFunc
	settings RegularTaskSettings

	stopped  bool
	runChan  chan bool
	stopChan chan bool
}

func NewRegularTask(
	runFunc RunFunc,
	settings RegularTaskSettings,
) *RegularTask {
	if settings.IdleBackoff == nil {
		settings.IdleBackoff = &backoff.ConstantBackOff{
			Interval: settings.Period,
		}
	}

	if settings.FailBackoff == nil {
		settings.FailBackoff = settings.IdleBackoff
	}

	if settings.Ctx == nil {
		settings.Ctx = context.Background()
	}
	if settings.BufferCapacity == 0 {
		settings.BufferCapacity = 1
	}

	task := &RegularTask{
		runFunc:  runFunc,
		settings: settings,
		runChan:  make(chan bool, settings.BufferCapacity),
		stopChan: make(chan bool, 1),
	}

	go task.run()

	return task
}

func (task *RegularTask) RunNow() *RegularTask {
	if task.stopped {
		return task
	}

	select {
	case task.runChan <- true:
	default:
	}

	return task
}

func (task *RegularTask) Stop() {
	if task.stopped {
		return
	}

	close(task.runChan)
	task.stopped = true
}

func (task *RegularTask) StopHook() <-chan bool {
	return task.stopChan
}

func (task *RegularTask) run() {
	failed := false
	defer func() {
		task.stopChan <- failed
		close(task.stopChan)
	}()

	ticker := time.NewTicker(task.settings.Period)
	defer ticker.Stop()

	previousStatus := StatusSuccess
	for {
		select {
		case <-task.settings.Ctx.Done():
			return
		case _, ok := <-task.runChan:
			if !ok {
				return
			}
		case <-ticker.C:
		}

		status, err := task.runFunc()
		if err != nil {
			logger.Log().Errorf("RegularTask %s: failed with error: %s", task.settings.Name, err)
			failed = true
			return
		}

		var nextBackoff time.Duration
		switch status {
		case StatusSuccess:
			nextBackoff = task.settings.Period
		case StatusIdle:
			if previousStatus != StatusIdle {
				task.settings.IdleBackoff.Reset()
			}
			nextBackoff = task.settings.IdleBackoff.NextBackOff()
		case StatusFailed:
			if previousStatus != StatusFailed {
				task.settings.FailBackoff.Reset()
			}
			nextBackoff = task.settings.FailBackoff.NextBackOff()
		default:
			logger.Log().Errorf("Unknown task status '%v'", status)
			nextBackoff = task.settings.Period
		}

		ticker.Reset(nextBackoff)

		previousStatus = status
	}
}
