package tskv

import "strings"

type Line struct {
	builder *strings.Builder
}

func NewTskvLine(tskvFormat string) *Line {
	var builder strings.Builder

	builder.Grow(1024)

	builder.WriteString("tskv")

	line := &Line{
		builder: &builder,
	}
	line.AddValue("tskv_format", tskvFormat, false)

	return line
}

func (l *Line) AddValue(key, value string, escape bool) *Line {
	if value == "" {
		return l
	}

	l.builder.WriteByte('\t')
	l.builder.WriteString(key)
	l.builder.WriteByte('=')
	if escape {
		WriteEscaped(l.builder, value)
	} else {
		l.builder.WriteString(value)
	}

	return l
}

func (l *Line) AddValues(values map[string]string, escape bool) *Line {
	for key, value := range values {
		l.AddValue(key, value, escape)
	}

	return l
}

func (l *Line) String() string {
	return l.builder.String()
}

func Escape(str string) string {
	var builder strings.Builder
	builder.Grow(len(str) * 2)

	WriteEscaped(&builder, str)

	return builder.String()
}

func WriteEscaped(builder *strings.Builder, str string) {
	for _, c := range []byte(str) {
		if e := encodeMap[c]; e != dontEscapeChar {
			builder.WriteByte('\\')
			builder.WriteByte(e)
		} else {
			builder.WriteByte(c)
		}
	}
}

const (
	dontEscapeChar = byte(255)
)

var encodeMap [256]byte

func init() {
	for i := range encodeMap {
		encodeMap[i] = dontEscapeChar
	}
	for from, to := range map[byte]byte{
		'\b':   'b',
		'\f':   'f',
		'\n':   'n',
		'\r':   'r',
		'\t':   't',
		'\x00': '0',
		'"':    '"',
		'\'':   '\'',
		'\\':   '\\',
		'=':    '=',
	} {
		encodeMap[from] = to
	}
}
