from cython.operator cimport dereference

from libc.stdint cimport uint32_t
from libcpp.memory cimport shared_ptr as cshared_ptr
from libcpp.vector cimport vector

from util.generic.string cimport TString, TStringBuf


cdef extern from "passport/infra/libs/python/gammer/impl/exception.h":
    cdef void gammer_raise_py_error()

cdef extern from "passport/infra/libs/python/gammer/impl/logger.h" namespace "NPassport::NGammer" nogil:
    cdef cppclass TLogMessage:
        int Level
        TString Message

cdef extern from "passport/infra/libs/python/gammer/impl/keyring.h" namespace "NPassport::NGammer" nogil:
    cdef cppclass TAesMessage:
        TString Iv
        TString Text
        TString Tag
        uint32_t GammaId

    cdef cppclass IKeyRing:
        TAesMessage EncryptAES(TStringBuf input, TStringBuf random, TStringBuf aadata) except +gammer_raise_py_error
        TString DecryptAES(const TAesMessage& input, TStringBuf random, TStringBuf aadata) except +gammer_raise_py_error


cdef extern from "passport/infra/libs/python/gammer/impl/facade.h" namespace "NPassport::NGammer" nogil:
    cdef cppclass TFacade:
        void Init(const TString& configFilePath) except +gammer_raise_py_error
        vector[TLogMessage] ExtractLogMessagesOnStart() except +gammer_raise_py_error
        const IKeyRing& GetKeyring() except +gammer_raise_py_error


cdef class __Gammer:
    cdef TFacade facade

    def __init__(self, *args, **kwargs):
        pass

    def __cinit__(self):
        pass

    def initialize(self, config_file_path):
        self.facade.Init(<TString>config_file_path.encode('utf-8'))

    def get_logs(self):
        return [(p.Level, p.Message.decode("utf-8")) for p in self.facade.ExtractLogMessagesOnStart()]

    def encrypt_aes(self, input, random, aadata):
        cdef TAesMessage res = self.facade.GetKeyring().EncryptAES(input, random, aadata)
        return res.Iv, res.Text, res.Tag, res.GammaId

    def decrypt_aes(self, iv, text, tag, gamma_id, random, aadata):
        cdef TAesMessage message
        message.Iv = iv
        message.Text = text
        message.Tag = tag
        message.GammaId = gamma_id

        return self.facade.GetKeyring().DecryptAES(message, random, aadata)
