#include "facade.h"

#include "keyring.h"

#include <passport/infra/libs/cpp/gamma_fetcher/fetcher.h>
#include <passport/infra/libs/cpp/tvm/common/service_tickets.h>
#include <passport/infra/libs/cpp/tvm/logger/logger.h>
#include <passport/infra/libs/cpp/utils/file.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/xml/config.h>

#include <library/cpp/tvmauth/client/facade.h>

namespace NPassport::NGammer {
    TFacade::TFacade()
        : Logger_(std::make_shared<TLogger>())
    {
        TLog::Init(Logger_);
    }

    TFacade::~TFacade() {
        TLog::Reset();
    }

    static std::shared_ptr<NTvmAuth::TTvmClient> CreateTvmClient(const NXml::TConfig& config, const TString& xpath) {
        const TString& alias = NTvmCommon::TServiceTickets::ISSNKMS_;

        NTvmAuth::NTvmApi::TClientSettings settings{
            .DiskCacheDir = config.AsString(xpath + "/tvm/disk_cache_dir"),
            .SelfTvmId = config.AsNum<ui32>(xpath + "/tvm/self_tvm_id"),
            .Secret = (TStringBuf)NUtils::ReadFile(config.AsString(xpath + "/tvm/secret_path")),
            .FetchServiceTicketsForDstsWithAliases = {
                {
                    alias,
                    config.AsNum<ui32>(xpath + "/tvm/destinations/destination[@name='" + alias + "']"),
                },
            },
            .TvmHost = config.AsString(xpath + "/tvm/host", settings.TvmHost),
            .TvmPort = config.AsNum<ui16>(xpath + "/tvm/port", settings.TvmPort),
        };

        return std::make_shared<NTvmAuth::TTvmClient>(settings, NTvmLogger::TLogger::Create());
    }

    void TFacade::Init(const TString& configFilePath) {
        const TString rootXpath = "/doc";
        const NXml::TConfig config = NXml::TConfig::ReadFromFile(configFilePath);

        NGammaFetcher::TFetcher::TGammas gammas =
            NGammaFetcher::TFetcher::FetchGammas(
                NGammaFetcher::TIssPartBaseFetcher::Create(config, rootXpath, CreateTvmClient(config, rootXpath)),
                config,
                rootXpath);

        KeyRing_ = std::make_unique<TKeyRingImpl>(std::move(gammas));
    }

    std::vector<TLogMessage> TFacade::ExtractLogMessagesOnStart() {
        return Logger_->ExtractLogMessages();
    }

    const IKeyRing& TFacade::GetKeyring() const {
        Y_ENSURE(KeyRing_, "internal error: keyring is not inited");

        return *KeyRing_;
    }
}
