#include "keyring.h"

#include <passport/infra/libs/cpp/utils/crypto/hash.h>

#include <util/generic/yexception.h>

namespace NPassport::NGammer {
    TKeyRingImpl::TKeyRingImpl(NGammaFetcher::TFetcher::TGammas&& gammas)
        : Gammas_(std::move(gammas))
    {
        Y_ENSURE(Gammas_.Gammas.contains(Gammas_.SigningGamma),
                 "internal error: default gamma id=" << Gammas_.SigningGamma << ", but there is no such gamma");

        for (auto& [key, value] : Gammas_.Gammas) {
            value.Body = std::make_shared<NSecretString::TSecretString>(
                NUtils::TCrypto::Sha256(
                    NUtils::CreateStr("some very const string", (TStringBuf)*value.Body)));
        }
    }

    TAesMessage TKeyRingImpl::EncryptAES(TStringBuf input, TStringBuf random, TStringBuf aadata) const {
        Y_ENSURE(random, "random cannot be empty");

        NUtils::TCrypto::TCiphertext ctx;
        TString err;
        Y_ENSURE(NUtils::TCrypto::EncryptGcm(
                     BuildEffectiveKey(Gammas_.SigningGamma, random),
                     input,
                     ctx,
                     aadata,
                     &err),
                 "failed to encrypt: " << err);

        return TAesMessage{
            .Iv = ctx.Iv,
            .Text = ctx.Text,
            .Tag = ctx.Tag,
            .GammaId = Gammas_.SigningGamma,
        };
    }

    TString TKeyRingImpl::DecryptAES(const TAesMessage& input, TStringBuf random, TStringBuf aadata) const {
        Y_ENSURE(random, "random cannot be empty");

        NUtils::TCrypto::TCiphertext ctx{
            .Iv = input.Iv,
            .Text = input.Text,
            .Tag = input.Tag,
        };
        TString err;

        TString res;
        Y_ENSURE(NUtils::TCrypto::DecryptGcm(
                     BuildEffectiveKey(input.GammaId, random),
                     ctx,
                     res,
                     aadata,
                     &err),
                 "failed to decrypt: " << err);

        return res;
    }

    TString TKeyRingImpl::BuildEffectiveKey(ui32 gammaId, TStringBuf random) const {
        auto it = Gammas_.Gammas.find(gammaId);
        Y_ENSURE(it != Gammas_.Gammas.end(),
                 "missing gamma with id=" << gammaId);
        const TStringBuf gamma = *it->second.Body;

        const TString rndHash = NUtils::TCrypto::Sha256(random);
        Y_ENSURE(gamma.size() == rndHash.size(),
                 "internal error: gamma size = " << gamma.size() << ", but random size = " << rndHash.size());

        TString res(gamma.size(), 0);
        for (size_t idx = 0; idx < gamma.size(); ++idx) {
            res[idx] = gamma[idx] ^ rndHash[idx];
        }

        return res;
    }
}
