#include <passport/infra/libs/python/gammer/impl/keyring.h>

#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NPassport;
using namespace NPassport::NGammer;

Y_UNIT_TEST_SUITE(Keyring) {
    static NGammaFetcher::TFetcher::TGammas CreateGammas() {
        return {
            .Gammas = {
                {1, {.Body = std::make_shared<NSecretString::TSecretString>("some gamma 1")}},
                {3, {.Body = std::make_shared<NSecretString::TSecretString>("some gamma 3")}},
            },
            .SigningGamma = 3,
        };
    }

    Y_UNIT_TEST(ctor) {
        UNIT_ASSERT_EXCEPTION_CONTAINS(
            TKeyRingImpl({}),
            yexception,
            "internal error: default gamma id=0, but there is no such gamma");

        UNIT_ASSERT_NO_EXCEPTION(TKeyRingImpl(CreateGammas()));
    }

    Y_UNIT_TEST(common) {
        TKeyRingImpl k(CreateGammas());

        const TString sensetiveText = "my ugly secret";
        const TString random = "kek";
        const TString aadata = "lol";

        TAesMessage message = k.EncryptAES(sensetiveText, random, aadata);

        TString decrypted;
        UNIT_ASSERT_NO_EXCEPTION(decrypted = k.DecryptAES(message, random, aadata));
        UNIT_ASSERT_VALUES_EQUAL(decrypted, sensetiveText);
    }

    Y_UNIT_TEST(encryptFail) {
        TKeyRingImpl k(CreateGammas());

        UNIT_ASSERT_EXCEPTION_CONTAINS(
            k.EncryptAES("my ugly secret", "", "lol"),
            yexception,
            "random cannot be empty");
    }

    Y_UNIT_TEST(decryptSuccess) {
        TKeyRingImpl k(CreateGammas());

        const TString sensetiveText = "my ugly secret";
        const TString random = "kek";
        const TString aadata = "lol";

        const TAesMessage message{
            .Iv = NUtils::Base64ToBin("L9VYcWKkEI4q4NaI"),
            .Text = NUtils::Base64ToBin("EGgPxFiCe8lxycCvlCQ"),
            .Tag = NUtils::Base64ToBin("T13aAAFpbibWre0Jzcp0JA"),
            .GammaId = 3,
        };

        TString decrypted;
        UNIT_ASSERT_NO_EXCEPTION(decrypted = k.DecryptAES(message, random, aadata));
        UNIT_ASSERT_VALUES_EQUAL(decrypted, sensetiveText);
    }

    Y_UNIT_TEST(decryptFail) {
        TKeyRingImpl k(CreateGammas());

        const TAesMessage validMessage{
            .Iv = NUtils::Base64ToBin("L9VYcWKkEI4q4NaI"),
            .Text = NUtils::Base64ToBin("EGgPxFiCe8lxycCvlCQ"),
            .Tag = NUtils::Base64ToBin("T13aAAFpbibWre0Jzcp0JA"),
            .GammaId = 3,
        };
        const TString validRandom = "kek";
        const TString validAadata = "lol";

        struct TTestCase {
            TString Name;
            TAesMessage InMessage;
            TString InRandom;
            TString InAadata;
            TString Err;
        };
        const std::vector<TTestCase> cases = {
            {
                .Name = "empty random",
                .Err = "random cannot be empty",
            },
            {
                .Name = "no such gamma",
                .InRandom = "some random",
                .Err = "missing gamma with id=0",
            },
            {
                .Name = "bad iv",
                .InMessage = {
                    .GammaId = 1,
                },
                .InRandom = "some random",
                .Err = "failed to decrypt: too short IV given, at least 96 bit needed",
            },
            {
                .Name = "bad tag",
                .InMessage = {
                    .Iv = "qqqqqqqqqqqqqqqqqqqq",
                    .GammaId = 1,
                },
                .InRandom = "some random",
                .InAadata = "",
                .Err = "failed to decrypt: too short tag given, at least 128 bit needed",
            },
            {
                .Name = "invalid params #1",
                .InMessage = {
                    .Iv = "qqqqqqqqqqqqqqqqqqqq",
                    .Tag = "aaaaaaaaaaaaaaaaaaaaa",
                    .GammaId = 1,
                },
                .InRandom = "some random",
                .Err = "failed to decrypt: unable to perform",
            },
            {
                .Name = "invalid params #2",
                .InMessage = {
                    .Iv = "qqqqqqqqqqqqqqqqqqqq",
                    .Text = validMessage.Text,
                    .Tag = validMessage.Tag,
                    .GammaId = validMessage.GammaId,
                },
                .InRandom = validRandom,
                .InAadata = validAadata,
                .Err = "failed to decrypt:",
            },
            {
                .Name = "invalid params #3",
                .InMessage = {
                    .Iv = validMessage.Iv,
                    .Text = "qqqqqqqqqqqqqqqqqqqq",
                    .Tag = validMessage.Tag,
                    .GammaId = validMessage.GammaId,
                },
                .InRandom = validRandom,
                .InAadata = validAadata,
                .Err = "failed to decrypt:",
            },
            {
                .Name = "invalid params #4",
                .InMessage = {
                    .Iv = validMessage.Iv,
                    .Text = validMessage.Text,
                    .Tag = "qqqqqqqqqqqqqqqqqqqq",
                    .GammaId = validMessage.GammaId,
                },
                .InRandom = validRandom,
                .InAadata = validAadata,
                .Err = "failed to decrypt:",
            },
            {
                .Name = "invalid params #5",
                .InMessage = {
                    .Iv = validMessage.Iv,
                    .Text = validMessage.Text,
                    .Tag = validMessage.Tag,
                    .GammaId = 1,
                },
                .InRandom = validRandom,
                .InAadata = validAadata,
                .Err = "failed to decrypt:",
            },
            {
                .Name = "invalid params #6",
                .InMessage = {
                    .Iv = validMessage.Iv,
                    .Text = validMessage.Text,
                    .Tag = validMessage.Tag,
                    .GammaId = validMessage.GammaId,
                },
                .InRandom = "qqqqqqqqqqqqqqqqqqqq",
                .InAadata = validAadata,
                .Err = "failed to decrypt:",
            },
            {
                .Name = "invalid params #7",
                .InMessage = {
                    .Iv = validMessage.Iv,
                    .Text = validMessage.Text,
                    .Tag = validMessage.Tag,
                    .GammaId = validMessage.GammaId,
                },
                .InRandom = validRandom,
                .InAadata = "qqqqqqqqqqqqqqqqqqqq",
                .Err = "failed to decrypt:",
            },
        };

        for (const TTestCase& c : cases) {
            UNIT_ASSERT_EXCEPTION_CONTAINS_C(
                k.DecryptAES(c.InMessage, c.InRandom, c.InAadata),
                yexception,
                c.Err,
                c.Name);
        }
    }
}
