import os
import shutil

import yatest.common
from yatest.common import network
from library.python.testing.recipe import declare_recipe, set_env
from passport.infra.recipes.common import log, start_daemon, stop_daemon
from passport.infra.recipes.kolmogor.common import gen_default_config, make_space, write_config

MYSQL_PORTFILE_NAME = 'mysql.port'
PASSPORT_PORTFILE_NAME = 'passport.port'
TVM_PORTFILE_NAME = 'tvmapi.port'


def _get_port_from_file(file):
    try:
        with open(file) as f:
            return f.read()
    except IOError:
        log('Could not find port file: %s' % file)
        raise RuntimeError('%s port is unknown' % (file.split('.')[0]))


def _generate_blackbox_config(config_templates_path, configs_path, output_path, kolmogor_port):
    # prepare config templates
    shutil.copytree(yatest.common.source_path('passport/infra/daemons/blackbox/config/configs'), config_templates_path)
    shutil.copy(
        yatest.common.source_path('passport/infra/daemons/blackbox/config/usernets/usernets.json'),
        config_templates_path,
    )

    # autocheck runs tests with readonly tree - some part is copied to tmp dir without 'w' flag
    # we need to add new file to tmp dir, so we are adding missing flag
    os.system('cd %s; chmod +w -R .' % config_templates_path)

    with open(config_templates_path + '/hostname', 'w') as f:
        f.write('dev/sandbox-test\n')

    daemon_port = network.PortManager().get_tcp_port()
    monitoring_port = network.PortManager().get_tcp_port()
    sesskill_port = network.PortManager().get_tcp_port()

    mysql_port = _get_port_from_file(MYSQL_PORTFILE_NAME)
    passport_port = _get_port_from_file(PASSPORT_PORTFILE_NAME)
    tvm_port = _get_port_from_file(TVM_PORTFILE_NAME)

    os.environ['BLACKBOX_DAEMON_PORT'] = str(daemon_port)
    os.environ['BLACKBOX_MONITORING_PORT'] = str(monitoring_port)
    os.environ['BLACKBOX_PASSPORTDB_PORT'] = mysql_port
    os.environ['BLACKBOX_KOLMOGOR_PORT'] = str(kolmogor_port)
    os.environ['BLACKBOX_SESSKILL_PORT'] = str(sesskill_port)
    os.environ['BLACKBOX_PASSPORT_PORT'] = str(passport_port)
    os.environ['BLACKBOX_TVM_PORT'] = str(tvm_port)
    os.environ['BLACKBOX_ROOT_DIRECTORY'] = output_path
    os.environ['BLACKBOX_LOGS_DIRECTORY'] = output_path + '/logs'
    os.environ['BLACKBOX_CONFIG_DIRECTORY'] = configs_path
    os.environ['BLACKBOX_GEOBASE_DIRECTORY'] = configs_path

    path = os.getenv('PATH')
    newpath = (
        yatest.common.build_path('contrib/tools/xmllint') + ':' + yatest.common.build_path('contrib/tools/xsltproc')
    )
    newpath += ':' + (path if path else '/bin:/usr/bin')
    os.putenv('PATH', newpath)

    # generate config
    if os.system(config_templates_path + '/make_configs.sh'):
        raise RuntimeError('make_configs.sh failed!')

    # prepare additional files and directories
    shutil.copy(config_templates_path + '/usernets.json', configs_path)
    shutil.copy(config_templates_path + '/shard_ranges_dev.txt', configs_path)
    shutil.copy(config_templates_path + '/dev/sandbox-test/grants.xml', configs_path)
    shutil.copy(config_templates_path + '/dev/sandbox-test/grants.json', configs_path)
    shutil.copy(config_templates_path + '/dev/sandbox-test/oauth.scopes.json', configs_path)
    shutil.copytree(config_templates_path + '/dev/sandbox-test/test_keys', configs_path + '/blackbox_keys')
    shutil.copy('test-geodata.bin', configs_path + '/geodata6.bin')

    os.makedirs(output_path + '/var/run/passport')
    os.makedirs(output_path + '/var/cache/yandex/passport-blackbox/tvm')
    os.makedirs(output_path + '/var/cache/yandex/passport-blackbox/keyrings')
    os.makedirs(output_path + '/usr/local/www/passport')
    os.makedirs(output_path + '/etc/yandex/tvm')

    set_env('RECIPE_BLACKBOX_PORT', str(daemon_port))
    set_env('RECIPE_SESSKILL_PORT', str(sesskill_port))

    return daemon_port


def _generate_kolmogor_config(output_path):
    log("Making config for kolmogor")
    cfg, kolmo_url = gen_default_config()

    blackbox_tvm_id = 224
    cfg["component"]["spaces"].append(make_space(name="badauth", allowed_client_id=blackbox_tvm_id))
    cfg["component"]["spaces"].append(make_space(name="badauth_short", allowed_client_id=blackbox_tvm_id))
    cfg["component"]["spaces"].append(make_space(name="badauth_uniq", allowed_client_id=blackbox_tvm_id))
    write_config(output_path + '/kolmogor.conf', cfg)

    kolmogor_port = cfg["http_daemon"]["ports"][0]["port"]

    set_env('RECIPE_KOLMOGOR_PORT', str(kolmogor_port))

    return kolmogor_port


def start(argv):
    log('Starting BlackBox recipe')

    # first, init configuration parameters
    output_path = yatest.common.output_path()

    # to be able to examine config and log files after the test run
    # we put it all into output directory

    config_templates_path = output_path + '/conf_templates'
    configs_path = output_path + '/available'

    kolmogor_port = _generate_kolmogor_config(output_path)
    daemon_port = _generate_blackbox_config(
        config_templates_path,
        configs_path,
        output_path,
        kolmogor_port,
    )

    # launch blackbox
    environment = os.environ.copy()
    command = [
        yatest.common.build_path('passport/infra/daemons/blackbox/daemon') + '/blackbox',
        '-c',
        configs_path + '/blackbox.daemon',
    ]

    pid = start_daemon(command, environment, daemon_port, 600).pid
    set_env('RECIPE_BLACKBOX_PID', str(pid))


def stop(argv):
    log('Terminating BlackBox recipe')

    # kill blackbox executable
    if 'RECIPE_BLACKBOX_PID' not in os.environ:
        log('RECIPE_BLACKBOX_PID variable not found in the environment')
        return

    pid = int(os.environ['RECIPE_BLACKBOX_PID'])

    stop_daemon(pid)


if __name__ == "__main__":
    declare_recipe(start, stop)
