import datetime
import os.path
import psutil
import socket
import subprocess
import sys
import time
import tarfile
import yatest.common


def log(msg):
    sys.stderr.write('%s: %s\n' % (datetime.datetime.now(), msg))
    sys.stderr.flush()


def untar(file_name):
    log('Un-tar file {}'.format(file_name))
    tar = tarfile.open(file_name)
    tar.extractall()
    tar.close()


def start_daemon(command, env, port=None, timeout=60):
    """Starts the daemon, pings the port (if any) for timeout seconds
    and returns the daemon's Popen object"""
    daemon_name = os.path.basename(command[0])

    logdir = yatest.common.output_path()
    process = subprocess.Popen(
        command,
        env=env,
        stdout=open(logdir + '/{}.out.log'.format(daemon_name), 'w'),
        stderr=open(logdir + '/{}.err.log'.format(daemon_name), 'w'),
    )

    if port:
        addr = socket.getaddrinfo("localhost", str(port), socket.AF_INET, socket.SOCK_STREAM)

    for attempts in range(timeout):
        time.sleep(1)

        # check if process exited
        result = process.poll()
        if result is not None:
            raise RuntimeError('Could not launch {}'.format(daemon_name))

        # check socket
        if port:
            sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
            sock.settimeout(1)
            try:
                sock.connect(addr[0][4])
                log("Daemon for {} is up on port {} ({} attempts)".format(daemon_name, port, attempts))
                return process  # port acquired, it is a success
            except socket.error:
                pass

    raise RuntimeError('{}: Could not launch for {} seconds'.format(datetime.datetime.now(), timeout))


def pid_exists(pid):
    try:
        if psutil.Process(pid).status() == psutil.STATUS_ZOMBIE:
            return False
    except psutil.NoSuchProcess:
        return False
    return True


def stop_daemon(pid):
    if pid_exists(pid):
        os.kill(pid, 15)
        while pid_exists(pid):
            time.sleep(1)
    else:
        log('pid %d not found' % pid)
