import os
import os.path
import psutil
import subprocess
import sys
import time

from library.python.testing.recipe import declare_recipe, set_env
from passport.infra.recipes.common import log, untar, start_daemon
from yatest.common import network
import yatest.common
import platform

# We are using here a stripped down MySQL package by alexbool@

# How to update version
#
# - download fresh MySQL distribution (generic tar.gz version)
# - strip down to reduce size:
#
# # For linux:
# $ MYSQL_NAME=mysql-5.7.22-linux-glibc2.12-x86_64
# $ tar xvf $MYSQL_NAME.tar.gz
# $ cd $MYSQL_NAME
# $ rm -rf lib # we don't need static libs
# $ find bin -type f -not -name mysql -not -name mysqld -delete # we only need these two binaries
# $ cp /lib/x86_64-linux-gnu/libaio.so.1 bin/ # important: there is no libaio.so.1 on distbuild
# $ cd ..
# $ rm $MYSQL_NAME.tar.gz
# $ tar zcvf $MYSQL_NAME.tar.gz $MYSQL_NAME
#
# # For macOS: same as Linux, but skip adding libaio.so.1
#
# - ya upload distribution, change filename below and resource id in recipe.inc file

if platform.system() == "Linux":
    MYSQL_NAME = 'mysql-5.7.22-linux-glibc2.12-x86_64'
elif platform.system() == 'Darwin':
    MYSQL_NAME = 'mysql-5.7.22-macos10.13-x86_64'
else:
    raise RuntimeError('Unsupported platform {}'.format(platform.system()))

TARNAME = MYSQL_NAME + '.tar.gz'
PORTFILE_NAME = 'mysql.port'


def start(argv):
    log('MySQL: starting recipe with ' + str(argv))

    if 'RECIPE_MYSQL_PORT' in os.environ:
        set_env('SKIP_CLEANUP_MYSQL', '1')
        log('MySQL already running')
    else:
        # need to unpack and start MySQL daemon
        untar(TARNAME)

        port = network.PortManager().get_tcp_port()

        log('MySQL: starting daemon at port %d' % port)

        command = [
            # fmt: off
            MYSQL_NAME + '/bin/mysqld',
            '--no-defaults',
            '--skip-grant-tables',
            '--port', str(port),
            '--socket', './mysql-sock',
            '--datadir', './mysql-data',
            '--basedir', './' + MYSQL_NAME,
            '--sql_mode=NO_ENGINE_SUBSTITUTION,STRICT_ALL_TABLES',
            '--skip-performance-schema',
            # fmt: on
        ]

        environment = os.environ.copy()
        environment['TZ'] = 'Europe/Moscow'  # for time in tables
        if platform.system() == 'Linux':
            environment.update({'LD_LIBRARY_PATH': MYSQL_NAME + '/bin'})

        # first, init mysql tables
        _init_tables(command, environment)

        # launch daemon
        pid = start_daemon(command, environment, port).pid
        set_env('RECIPE_MYSQL_PID', str(pid))
        set_env('RECIPE_MYSQL_PORT', str(port))
        set_env('RECIPE_MYSQL_DIR', yatest.common.work_path(MYSQL_NAME) + '/bin/')

        with open(PORTFILE_NAME, 'w') as f:
            f.write(str(port))

    # daemon is running, fill it with data
    for path in argv:
        log('MySQL: processing path {}'.format(path))

        stdin = open(yatest.common.source_path(path))

        command = [
            # fmt: off
            MYSQL_NAME + '/bin/mysql',
            '--protocol', 'tcp',
            '--port', str(port),
            '--user', 'root',
            # fmt: on
        ]
        subprocess.check_call(command, stdin=stdin)

    log('MySQL: prepared and ready')


def _init_tables(command, environment, retries=3):
    def run():
        subprocess.check_call(
            command + ['--initialize-insecure'],
            stdout=sys.stdout,
            stderr=subprocess.STDOUT,
            env=environment,
        )

    for attempts in range(retries - 1):
        try:
            run()
            return
        except Exception as e:
            log('Failed to init tables: %s' % e)
            time.sleep(1)

    # last attempt should be out of try-except to show correct error on fail
    run()


def stop(argv):
    if 'SKIP_CLEANUP_MYSQL' in os.environ:
        return

    log('MySQL: terminating recipe')

    if 'RECIPE_MYSQL_PID' not in os.environ:
        log('MySQL: pid variable not found in the environment')
        return

    pid = int(os.environ['RECIPE_MYSQL_PID'])

    if psutil.pid_exists(pid):
        os.kill(pid, 9)
    else:
        log('MySQL: pid %d not found' % pid)


if __name__ == "__main__":
    declare_recipe(start, stop)
