#include <passport/infra/libs/cpp/utils/crypto/hash.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>

#include <util/datetime/base.h>

#include <atomic>
#include <functional>
#include <future>
#include <vector>

using namespace NPassport::NUtils;

const TString pwd = "qwerty";

struct TRes {
    ui64 Us = 0;
    ui64 Count = 0;

    constexpr TRes& operator+=(const TRes& o) noexcept {
        Us += o.Us;
        Count += o.Count;
        return *this;
    }
};

// NOLINTNEXTLINE(cppcoreguidelines-avoid-non-const-global-variables)
std::atomic_bool stop;

TRes RunMd5(const TString&, const TString& data) {
    TInstant start = TInstant::Now();
    ui64 count = 0;

    while (!stop.load(std::memory_order_acquire)) {
        TString hash = TCrypto::Md5(data);
        ++count;
    }

    return {(TInstant::Now() - start).MicroSeconds(), count};
}

TRes RunSha256(const TString&, const TString& data) {
    TInstant start = TInstant::Now();
    ui64 count = 0;

    while (!stop.load(std::memory_order_acquire)) {
        TString hash = TCrypto::Sha256(data);
        ++count;
    }

    return {(TInstant::Now() - start).MicroSeconds(), count};
}

TRes RunHmacSha256(const TString& key, const TString& data) {
    TInstant start = TInstant::Now();
    ui64 count = 0;

    while (!stop.load(std::memory_order_acquire)) {
        TString hash = TCrypto::HmacSha256(key, data);
        ++count;
    }

    return {(TInstant::Now() - start).MicroSeconds(), count};
}

TRes RunHmacSha1(const TString& key, const TString& data) {
    TInstant start = TInstant::Now();
    ui64 count = 0;

    while (!stop.load(std::memory_order_acquire)) {
        TString hash = TCrypto::HmacSha1(key, data);
        ++count;
    }

    return {(TInstant::Now() - start).MicroSeconds(), count};
}

TRes EncryptGcm(const TString& key, const TString& data) {
    ui64 count = 0;
    TCrypto::TCiphertext c;
    TString err;
    TInstant start = TInstant::Now();

    while (!stop.load(std::memory_order_acquire)) {
        if (!TCrypto::EncryptGcm(key, data, c, data, &err)) {
            Cerr << "encryptGCM failed with error: " << err << Endl;
        }
        ++count;
    }

    return {(TInstant::Now() - start).MicroSeconds(), count};
}

TRes DecryptGcm(const TString& key, const TString& data) {
    ui64 count = 0;
    TCrypto::TCiphertext c;
    TString out;
    TString err;

    if (!TCrypto::EncryptGcm(key, data, c, data, &err)) {
        Cerr << "encryptGCM failed with error: " << err << Endl;
    }

    TInstant start = TInstant::Now();

    while (!stop.load(std::memory_order_acquire)) {
        if (!TCrypto::DecryptGcm(key, c, out, data, &err)) {
            Cerr << "decryptGCM failed with error: " << err << Endl;
        }
        ++count;
    }

    return {(TInstant::Now() - start).MicroSeconds(), count};
}

void Test(ui32 threads, const TString& key, const TString& data, const std::function<decltype(EncryptGcm)> fun) {
    stop.store(false);

    std::vector<std::future<TRes>> vec;
    for (size_t i = 0; i < threads; ++i) {
        vec.push_back(std::async(std::launch::async, fun, key, data));
    }
    std::this_thread::sleep_for(std::chrono::seconds(10));
    stop.store(true, std::memory_order_relaxed);

    TRes total;
    for (auto& a : vec) {
        total += a.get();
    }

    Cout << "Threads=" << threads << "\t data.size=" << data.size()
         << "\t\tper_hash=" << total.Us / total.Count / 1000.0 << " ms" << Endl;
}

// NOLINTNEXTLINE(bugprone-exception-escape)
int main(int, char**) {
    TString key = Hex2bin("0123456789abcdef0123456789abcdef0123456789abcdef0123456789abcdef");
    ui32 thr[] = {1, 8, 16, 24, 32};
    ui32 lens[] = {32, 128, 1024, 2048, 1024 * 1024};

    for (ui32 threads : thr) {
        for (ui32 l : lens) {
            TString data(l, 0);
            Test(threads, key, data, &DecryptGcm);
        }
        Cout << Endl;
    }

    return 0;
}
