#include <passport/infra/libs/cpp/argon/argon.h>
#include <passport/infra/libs/cpp/argon/mem_pool.h>

#include <util/datetime/base.h>

#include <atomic>
#include <functional>
#include <future>
#include <vector>

using namespace NPassport::NArgon;

const TString pwd = "qwerty";

struct TRes {
    ui64 Us = 0;
    ui32 Count = 0;

    constexpr TRes& operator+=(const TRes& o) noexcept {
        Us += o.Us;
        Count += o.Count;
        return *this;
    }
};

// NOLINTNEXTLINE(cppcoreguidelines-avoid-non-const-global-variables)
std::atomic_bool stop;

TRes Gen(const TString& secret,
         ui32 hashlen,
         ui32 tcost,
         ui32 kb,
         const TString& salt) {
    TArgon2Factory f(hashlen);

    TInstant start = TInstant::Now();
    ui32 count = 0;
    while (!stop.load(std::memory_order_acquire)) {
        auto c = f.Get(secret, tcost, kb);
        c.MakeHash(pwd, salt);
        ++count;
    }

    return {(TInstant::Now() - start).MicroSeconds(), count};
}

TRes Check(const TString& secret,
           ui32 hashlen,
           ui32 tcost,
           ui32 kb,
           const TString& salt) {
    TArgon2Factory f(hashlen);

    TString hash;
    {
        auto c = f.Get(secret, tcost, kb);
        hash = c.MakeHash("qwerty", salt);
    }

    TInstant start = TInstant::Now();
    ui32 count = 0;
    while (!stop.load(std::memory_order_acquire)) {
        auto c = f.Get(secret, tcost, kb);
        c.VerifyHash(pwd, hash, salt);
        ++count;
    }

    return {(TInstant::Now() - start).MicroSeconds(), count};
}

void Test(ui32 threads,
          ui32 kb,
          const TString& secret,
          ui32 hashlen,
          ui32 tcost,
          const TString& salt,
          const std::function<decltype(Gen)> fun) {
    stop.store(false);

    std::vector<std::future<TRes>> vec;
    for (size_t i = 0; i < threads; ++i) {
        vec.push_back(std::async(std::launch::async,
                                 fun,
                                 secret,
                                 hashlen,
                                 tcost,
                                 kb,
                                 salt));
    }
    std::this_thread::sleep_for(std::chrono::seconds(10));
    stop.store(true, std::memory_order_relaxed);

    TRes total;
    for (auto& a : vec) {
        total += a.get();
    }

    Cout << " threads=" << threads
         << " kb=" << kb
         << " secret=" << secret.size()
         << " hashlen=" << hashlen
         << " tcost=" << tcost
         << " salt=" << salt.size()
         << " per_hash=" << total.Us / total.Count / 1000.0 << " ms"
         << Endl;
}

// NOLINTNEXTLINE(bugprone-exception-escape)
int main(int, char**) {
    std::vector<TString> secrets = {"0123456789abcdef0123456789abcdef0123456789abcdef0123456789abcdef"};
    std::vector<TString> salts = {"0123456789abcdef0123456789abcdef"};
    ui32 thr[] = {1, 16, 32};
    ui32 kbs[] = {32, 64, 128, 256, 512, 1024, 4096};

    int idx = 0;
    for (ui32 threads : thr) {
        for (ui32 kb : kbs) {
            for (const TString& secret : secrets) {
                for (ui32 hashlen = 128; hashlen <= 128; hashlen *= 2) {
                    for (ui32 tcost = 1; tcost <= 8; tcost *= 2) {
                        for (const TString& salt : salts) {
                            Cout << "Idx=" << ++idx;
                            Test(threads, kb, secret, hashlen, tcost, salt, &Gen);
                            //                            test(threads, kb, secret, hashlen, tcost, salt, &check);
                            //                            ++idx;
                        }
                    }
                }
            }
        }
    }

    Cerr << idx << Endl;

    return 0;
}
