#include <library/cpp/geobase/lookup.hpp>
#include <library/cpp/getopt/small/last_getopt.h>

void TestAsFromIp(const NGeobase::TLookup& geo, const TString& ip, const std::string& as) {
    std::vector<std::string> vec = geo.GetAsTraitsByIp(ip).AsList;
    auto it = std::find(vec.begin(), vec.end(), as);
    if (it == vec.end()) {
        TStringStream msg;
        msg << "Failed to get expected as for ip=" << ip << ". Found " << vec.size() << " as:" << Endl;
        for (const TStringBuf s : vec) {
            msg << "  " << s << Endl;
        }
        throw yexception() << msg.Str();
    }
}

void TestAsFromIp(const NGeobase::TLookup& geo) {
    TestAsFromIp(geo, "95.108.173.106", "AS13238");
    TestAsFromIp(geo, "8.8.8.8", "AS15169");
}

void TestBadIp(const NGeobase::TLookup& geo) {
    try {
        geo.GetAsTraitsByIp("8.8.8.800");
        Cerr << "Exception expected for ip '8.8.8.800'" << Endl;
        exit(1);
    } catch (const std::exception&) {
    }
}

void TestRegionByIp(const NGeobase::TLookup& geo, const TString& ip, const TString& region) {
    const TString res(geo.GetRegionByIp(ip).GetEnName());
    if (res != region) {
        throw yexception()
            << "Failed to get region from ip=" << ip << ". Found '" << res << "'. Expected: '" << region << "'" << Endl;
    }
}

void TestRegionByIp(const NGeobase::TLookup& geo) {
    TestRegionByIp(geo, "95.108.173.106", "Moscow");
}

void TestRegionById(const NGeobase::TLookup& geo, int id) {
    const int res = geo.GetRegionById(id).GetId();
    if (res != id) {
        throw yexception()
            << "Failed to get region by id. Found '" << res << "'. Expected: '" << id << "'" << Endl;
    }
}

void Test(const NGeobase::TLookup& geo) {
    TestAsFromIp(geo);
    TestBadIp(geo);
    TestRegionByIp(geo);
    TestRegionById(geo, 213);
}

int main(int argc, char** argv) {
    try {
        NLastGetopt::TOpts opts;
        opts.AddHelpOption();
        opts.AddCharOption('d', "data file from geobase").Required();
        NLastGetopt::TOptsParseResult res(&opts, argc, argv);

        Cout << "Starting to read data file..." << Endl;
        NGeobase::TLookup geo(res.Get('d'));
        Cout << "Starting to read data file...OK" << Endl;

        Cout << "Starting tests..." << Endl;
        Test(geo);
        Cout << "Starting tests...OK" << Endl;
    } catch (const std::exception& e) {
        Cerr << "Exception: " << e.what() << Endl;
        return 1;
    } catch (...) {
        Cerr << "Unknown exception" << Endl;
        return 1;
    }

    return 0;
}
