#include "encryptor.h"

#include <passport/infra/libs/cpp/utils/file.h>
#include <passport/infra/libs/cpp/utils/crypto/hash.h>
#include <passport/infra/libs/cpp/utils/log/global.h>
#include <passport/infra/libs/cpp/utils/string/coder.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

namespace NPassport::NMauditpipe {
    TEncryptor::TEncryptor(const TString& dir,
                           TDuration rotationPeriod,
                           TInstant now)
        : Dir_(dir)
        , RotationPeriod_(rotationPeriod)
    {
        // Minor check on start
        GetKey(GetCurrentKeyId(now));
    }

    TString TEncryptor::Encrypt(TStringBuf str, TInstant now) {
        const ui64 keyId = GetCurrentKeyId(now);
        const TString& key = GetKey(keyId);

        TString aaData = NUtils::CreateStr("key_number=", keyId);

        TString err;
        NUtils::TCrypto::TCiphertext ctx;
        Y_ENSURE(NUtils::TCrypto::EncryptGcm(key, str, ctx, aaData, &err),
                 "Encryptor: failed to encrypt: " << err);

        const char delim = ':';

        return NUtils::CreateStr(
            "1", delim, //  format version
            NUtils::BinToBase64(ctx.Iv), delim,
            NUtils::BinToBase64(ctx.Text), delim,
            aaData, delim,
            NUtils::BinToBase64(ctx.Tag));
    }

    static const TDuration PRINT_PERIOD = TDuration::Minutes(10);

    const TString& TEncryptor::GetKey(ui64 id) {
        if (id == CurrentKeyId_) {
            const TInstant now = TInstant ::Now();
            if (NextPrint_ < now) {
                NextPrint_ = now + PRINT_PERIOD;
                TLog::Info() << "Encryptor: current key_id=" << CurrentKeyId_;
            }

            return CurrentKey_;
        }

        CurrentKey_ = LoadKey(Dir_, id);
        CurrentKeyId_ = id;

        return CurrentKey_;
    }

    ui64 TEncryptor::GetCurrentKeyId(TInstant now) const {
        return now.Seconds() / RotationPeriod_.Seconds();
    }

    TString TEncryptor::LoadKey(const TString& dir, ui64 id) {
        const TString path = NUtils::CreateStr(dir, "/", id, ".key");
        TLog::Info() << "Encryptor: trying to load new key from disk: "
                     << path;

        TString res = NUtils::ReadFile(path);
        TLog::Info() << "Encryptor: succeed to load new key from disk: "
                     << path;
        return res;
    }
}
