#include "facade.h"

#include <passport/infra/libs/cpp/utils/log/global.h>

namespace NPassport::NMauditpipe {
    // NOLINTNEXTLINE(cppcoreguidelines-avoid-non-const-global-variables)
    static TPipe* PIPE = nullptr;

    static void SignalsHandlers(int signo) {
        if (SIGUSR1 == signo) {
            TLog::LogGotSignal(TLog::ESignal::Sigusr1);
            if (PIPE) {
                PIPE->Stop();
                PIPE = nullptr;
            }
        }
    }

    TFacade::TFacade(const TString& configPath) {
        const NJson::TConfig cfg = NJson::TConfig::ReadFromFile(configPath);
        const TString path = "";

        InitLogs(cfg, path + "/logger");

        Encryptor_ = InitEncryptor(cfg, path + "/encryptor");
        TPipeSettings settings = InitPipeSettings(cfg, path + "/parser");
        Pipe_ = std::make_unique<TPipe>(settings, *Encryptor_);

        PIPE = Pipe_.get();
        Y_ENSURE(signal(SIGUSR1, SignalsHandlers) != SIG_ERR, "Cannot set up SIGUSR1 handler");
    }

    void TFacade::Run() {
        TLog::Info() << "Facade: Started";
        Pipe_->Run(Cin, Cout);
        TLog::Info() << "Facade: Stopping";
    }

    void TFacade::InitLogs(const NJson::TConfig& cfg, const TString& path) {
        cfg.InitCommonLog(path);
        TLog::Info() << "Facade: Starting";
    }

    std::unique_ptr<TEncryptor> TFacade::InitEncryptor(const NJson::TConfig& cfg,
                                                       const TString& path) {
        const TString dir = cfg.As<TString>(path + "/keys_dir");
        TDuration rotationPeriod = TDuration::Seconds(
            cfg.As<ui64>(path + "/key_rotation_period__sec"));

        return std::make_unique<TEncryptor>(dir, rotationPeriod);
    }

    TPipeSettings TFacade::InitPipeSettings(const NJson::TConfig& cfg, const TString& path) {
        TPipeSettings res;

        res.InType = TypeFromString(cfg.As<TString>(path + "/input_log_format"));
        res.MatchSubstrings = cfg.As<std::vector<TString>>(path + "/match_substrings_in_query");
        for (const TString& p : res.MatchSubstrings) {
            TLog::Debug() << "Facade: match query with substring '" << p << "'";
        }

        return res;
    }
}
