#include "parser.h"

#include <passport/infra/libs/cpp/json/writer.h>
#include <passport/infra/libs/cpp/utils/string/string_utils.h>

#include <unordered_map>

namespace NPassport::NMauditpipe {
    static const std::unordered_map<TString, EInLogType> TYPES = {
        {"percona", EInLogType::Percona},
        {"mcafee", EInLogType::Mcafee},
    };

    EInLogType TypeFromString(const TString& s) {
        auto it = TYPES.find(s);
        if (it != TYPES.end()) {
            return it->second;
        }

        TString all;
        for (const auto& [key, _] : TYPES) {
            NUtils::AppendSeparated(all, ',', key);
        }

        ythrow yexception() << "unknown input log type: " << s
                            << ". Known are: " << all;
    }

    TParserPtr CreateParser(EInLogType type) {
        switch (type) {
            case EInLogType::Percona:
                return std::make_unique<TPerconaParser>();
            case EInLogType::Mcafee:
                return std::make_unique<TMcafeeParser>();
        }
    }

    static TString JsonToString(const rapidjson::Value& value) {
        rapidjson::StringBuffer sb;

        NJson::TRapidWriter wr(sb);
        value.Accept(wr);

        return TString(sb.GetString(), sb.GetSize());
    }

    void TBaseParser::BuildBase(const rapidjson::Value& data, TStringBuf queryKey, ISerializer& out) {
        for (auto it = data.MemberBegin(); it != data.MemberEnd(); ++it) {
            TStringBuf key(it->name.GetString(), it->name.GetStringLength());
            TStringBuf value;

            TString tmp;
            if (it->value.IsString()) {
                value = TStringBuf(it->value.GetString(), it->value.GetStringLength());
            } else if (it->value.IsNumber()) {
                tmp = ToString(it->value.GetDouble());
            } else if (it->value.IsBool()) {
                tmp = it->value.GetBool() ? "true" : "false";
            } else {
                tmp = JsonToString(it->value);
            }
            if (value.empty()) {
                value = tmp;
            }

            if (key == queryKey) {
                out.AddQuery(key, value);
            } else {
                out.Add(key, value);
            }
        }
    }

    void TPerconaParser::Parse(TStringBuf line) {
        Y_ENSURE(NJson::TReader::DocumentAsObject(line, Doc_),
                 "Malformed json: '" << line << "'");

        Y_ENSURE(NJson::TReader::MemberAsObject(Doc_, "audit_record", Data_),
                 "unexpected json format: missing 'audit_record'");
    }

    TStringBuf TPerconaParser::GetQuery() {
        TStringBuf res;
        NJson::TReader::MemberAsString(*Data_, QUERYKEY.c_str(), res);
        return res;
    }

    void TPerconaParser::Build(ISerializer& out) {
        BuildBase(*Data_, QUERYKEY, out);
    }

    void TMcafeeParser::Parse(TStringBuf line) {
        Y_ENSURE(NJson::TReader::DocumentAsObject(line, Doc_),
                 "Malformed json: '" << line << "'");
    }

    TStringBuf TMcafeeParser::GetQuery() {
        TStringBuf res;
        NJson::TReader::MemberAsString(Doc_, QUERYKEY.c_str(), res);
        return res;
    }

    void TMcafeeParser::Build(ISerializer& out) {
        BuildBase(Doc_, QUERYKEY, out);
    }
}
