#include "pipe.h"

#include <passport/infra/libs/cpp/utils/log/global.h>

#include <library/cpp/string_utils/tskv_format/builder.h>

#include <util/string/cast.h>
#include <util/string/subst.h>

namespace NPassport::NMauditpipe {
    TPipe::TPipe(const TPipeSettings& settings, IEncryptor& enc)
        : InType_(settings.InType)
        , SkipChecker_(std::make_unique<TSkipChecker>(settings.MatchSubstrings))
        , Encryptor_(enc)
    {
    }

    void TPipe::Stop() {
        Exit_ = true;
    }

    static const TDuration PRINT_PERIOD = TDuration::Minutes(10);

    void TPipe::Run(IInputStream& in, IOutputStream& out) {
        TInstant nextPrint;
        size_t inLines = 0;
        size_t outLines = 0;
        TString line;

        while (in.ReadLine(line) > 0 && !Exit_.load(std::memory_order_relaxed)) {
            ++inLines;

            try {
                const TInstant now = TInstant::Now();
                if (nextPrint < now) {
                    nextPrint = now + PRINT_PERIOD;
                    TLog::Debug() << "Pipe: heartbeat: in_lines: " << inLines
                                  << ", out_lines: " << outLines;
                }

                const char delim = ';';

                TStringBuf buf = line;
                TStringBuf magic1 = buf.NextTok(delim);
                TStringBuf magic2 = buf.NextTok(delim);
                TStringBuf magic3 = buf.NextTok(delim);

                std::optional<TString> res = ProcessLine(buf, InType_, *SkipChecker_, Encryptor_);
                if (!res) {
                    continue;
                }

                ++outLines;
                out
                    << magic1 << delim
                    << magic2 << delim
                    << magic3 << delim
                    << *res;
            } catch (const std::exception& e) {
                TLog::Error() << "Pipe: exception while processing line: " << e.what()
                              << ". '" << line << "'";
                out << line << Endl;
            }
        }

        TLog::Info() << "Pipe: exiting, in_lines: " << inLines
                     << ", out_lines: " << outLines;
    }

    std::optional<TString> TPipe::ProcessLine(TStringBuf line,
                                              EInLogType inType,
                                              const TSkipChecker& skipChecker,
                                              IEncryptor& encryptor) {
        TParserPtr parser = CreateParser(inType);

        parser->Parse(line);
        if (skipChecker.ShouldSkip(parser->GetQuery())) {
            return {};
        }

        TSerializer ser(encryptor, line.size());
        parser->Build(ser);

        return ser.Finish();
    }
}
