#include <passport/infra/tools/mauditpipe/src/pipe.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/stream/file.h>
#include <util/system/fs.h>

using namespace NPassport;
using namespace NPassport::NMauditpipe;

Y_UNIT_TEST_SUITE(Pipe) {
    class TTestEncryptor: public IEncryptor {
    public:
        TString Encrypt(TStringBuf, TInstant) override {
            return "<ENCRYPTED_STRING>";
        }
    };

    Y_UNIT_TEST(processLine) {
        TTestEncryptor e;

        struct TCase {
            TString In;
            std::optional<TString> Out;
            EInLogType Type = EInLogType::Percona;
        };

        std::vector<TCase> cases = {
            {
                .In = R"({"audit_record":{"name":"Connect","record":"6_2021-02-16T13:42:12","timestamp":"2021-02-16T13:42:14 UTC","connection_id":"10","status":0,"user":"monitor","priv_user":"monitor","os_login":"","proxy_user":"","host":"localhost","ip":"","db":""}})",
                .Out = "tskv	tskv_format=mauditpipe	name=Connect	record=6_2021-02-16T13:42:12	timestamp=2021-02-16T13:42:14 UTC	connection_id=10	status=0	user=monitor	priv_user=monitor	os_login=	proxy_user=	host=localhost	ip=	db=\n",
                .Type = EInLogType::Percona,
            },
            {
                .In = R"({"audit_record":{"name":"Query","record":"19_2021-02-16T13:42:12","timestamp":"2021-02-16T13:42:17 UTC","command_class":"select","connection_id":"15","status":0,"sqltext":"select @@version_comment limit 1","user":"monitor[monitor] @ localhost []","host":"localhost","os_user":"","ip":"","db":""}})",
                .Out = {},
                .Type = EInLogType::Percona,
            },
            {
                .In = R"({"audit_record":{"name":"Query","record":"99998_2021-02-16T13:42:12","timestamp":"2021-02-16T13:52:16 UTC","command_class":"select","connection_id":"649","status":0,"sqltext":"INSERT INTO randoms (keybody, start, valid) VALUES ('aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa', 'bbbbbbbbbbbbbbbbbbbbbbbbbb', '2021-05-10 21:10:02') ","user":"blackbox_ro[blackbox_ro] @  [2a02:6b8:c02:756:8000:636:0:3]","host":"","os_user":"","ip":"2a02:6b8:c02:756:8000:636:0:3","db":"passportdbcentral"}})",
                .Out = "tskv	tskv_format=mauditpipe	name=Query	record=99998_2021-02-16T13:42:12	timestamp=2021-02-16T13:52:16 UTC	command_class=select	connection_id=649	status=0	sqltext=<ENCRYPTED_STRING>	user=blackbox_ro[blackbox_ro] @  [2a02:6b8:c02:756:8000:636:0:3]	host=	os_user=	ip=2a02:6b8:c02:756:8000:636:0:3	db=passportdbcentral\n",
                .Type = EInLogType::Percona,
            },
            {
                .In = R"({"msg-type":"activity","date":"1612026493316","thread-id":"2","query-id":"0","user":"monitor","priv_user":"monitor","ip":"","host":"localhost","pid":"478099","os_user":"monitor","appname":"/usr/local/yasmagent/python/bin/python","cmd":"Connect","query":"Connect"})",
                .Out = "tskv\ttskv_format=mauditpipe\tmsg-type=activity\tdate=1612026493316\tthread-id=2\tquery-id=0\tuser=monitor\tpriv_user=monitor\tip=\thost=localhost\tpid=478099\tos_user=monitor\tappname=/usr/local/yasmagent/python/bin/python\tcmd=Connect\tquery=<ENCRYPTED_STRING>\n",
                .Type = EInLogType::Mcafee,
            },
            {
                .In = R"({"msg-type":"activity","date":"1612544199555","thread-id":"7656","query-id":"8522451","user":"blackbox_ro","priv_user":"blackbox_ro","ip":"2a02:6b8:c02:f58:8000:636:0:f","host":"2a02:6b8:c02:f58:8000:636:0:f","connect_attrs":{"_pid":"987949","_os":"Linux","_platform":"x86_64","_client_version":"8.0.17","_client_name":"libmysql"},"rows":"1","status":"0","cmd":"select","objects":[{"db":"passportdbcentral","name":"randoms_signsmth_long","obj_type":"TABLE"}],"query":"select min(id), max(id), count(id) from randoms_signsmth_long"})",
                .Out = "tskv\ttskv_format=mauditpipe\tmsg-type=activity\tdate=1612544199555\tthread-id=7656\tquery-id=8522451\tuser=blackbox_ro\tpriv_user=blackbox_ro\tip=2a02:6b8:c02:f58:8000:636:0:f\thost=2a02:6b8:c02:f58:8000:636:0:f\tconnect_attrs={\\\"_pid\\\":\\\"987949\\\",\\\"_os\\\":\\\"Linux\\\",\\\"_platform\\\":\\\"x86_64\\\",\\\"_client_version\\\":\\\"8.0.17\\\",\\\"_client_name\\\":\\\"libmysql\\\"}\trows=1\tstatus=0\tcmd=select\tobjects=[{\\\"db\\\":\\\"passportdbcentral\\\",\\\"name\\\":\\\"randoms_signsmth_long\\\",\\\"obj_type\\\":\\\"TABLE\\\"}]\tquery=<ENCRYPTED_STRING>\n",
                .Type = EInLogType::Mcafee,
            },
            {
                .In = R"({"msg-type":"activity","date":"1612030202110","thread-id":"17223","query-id":"2195461","user":"api_rw","priv_user":"api_rw","ip":"2a02:6b8:c04:156:8000:635:0:5","host":"2a02:6b8:c04:156:8000:635:0:5","connect_attrs":{"_pid":"28298","_os":"Linux","_platform":"x86_64","_client_version":"8.0.17","_client_name":"libmysql"},"rows":"1","status":"0","cmd":"insert","objects":[{"db":"passportdbcentral","name":"randoms","obj_type":"TABLE"}],"query":"INSERT INTO randoms (keybody, start, valid) VALUES ('aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa', 'bbbbbbbbbbbbbbbbbbbbbbbbbb', '2021-05-10 21:10:02') "})",
                .Out = "tskv\ttskv_format=mauditpipe\tmsg-type=activity\tdate=1612030202110\tthread-id=17223\tquery-id=2195461\tuser=api_rw\tpriv_user=api_rw\tip=2a02:6b8:c04:156:8000:635:0:5\thost=2a02:6b8:c04:156:8000:635:0:5\tconnect_attrs={\\\"_pid\\\":\\\"28298\\\",\\\"_os\\\":\\\"Linux\\\",\\\"_platform\\\":\\\"x86_64\\\",\\\"_client_version\\\":\\\"8.0.17\\\",\\\"_client_name\\\":\\\"libmysql\\\"}\trows=1\tstatus=0\tcmd=insert\tobjects=[{\\\"db\\\":\\\"passportdbcentral\\\",\\\"name\\\":\\\"randoms\\\",\\\"obj_type\\\":\\\"TABLE\\\"}]\tquery=<ENCRYPTED_STRING>\n",
                .Type = EInLogType::Mcafee,
            },
        };

        TPipeSettings settings{
            .MatchSubstrings = {
                "randoms",
            },
        };

        for (const TCase& c : cases) {
            UNIT_ASSERT_VALUES_EQUAL_C(
                c.Out,
                TPipe::ProcessLine(c.In, c.Type, TSkipChecker(settings.MatchSubstrings), e),
                c.In);
        }
    }
}

template <>
void Out<std::optional<TString>>(IOutputStream& o, const std::optional<TString>& value) {
    if (value) {
        o << *value;
    } else {
        o << "<NULL>";
    }
}
