import argparse
import datetime
import os
from os.path import (
    isdir,
    isfile,
)
import shutil
import socket
import subprocess
import sys


class Base:
    def __init__(self, args):
        parser = argparse.ArgumentParser()
        self.add_arguments(parser)

        parser.add_argument('--name', help="project name", required=True)
        parser.add_argument('--common_port', type=int, required=True)
        parser.add_argument('--unistat_port', type=int, required=True)
        parser.add_argument('--dst_dir', default=os.getcwd())
        parser.add_argument('-v', '--verbose', action='store_true')
        parser.add_argument('--owner', default='g:passport_infra')
        parser.add_argument('--package_type', default='debian')
        parser.add_argument('--author', default='passport-dev@yandex-team.ru')
        parser.add_argument('--add_xunistater_with_port', type=int, default=None)
        parser.add_argument('--add_nginx_with_hosts', help="formated: production[,testing[,development]]", default=None)
        parser.add_argument('--purge', action='store_true', default=False)
        parser.add_argument('--add_only_new', action='store_true', default=False)

        self.options = parser.parse_args(args)
        if not isdir(self.options.dst_dir):
            raise Exception('%s does not exist' % self.options.dst_dir)

        self.dir = os.path.join(self.options.dst_dir, self.options.name)
        if self.options.purge and isdir(self.dir):
            shutil.rmtree(self.dir)
        self.arcadia_related_path = self._get_arcadia_related_path(self.dir)

        self.dirs = set()
        self.project_dir = self.options.name.replace('_', '-')

        hosts = self.options.add_nginx_with_hosts.split(',') if self.options.add_nginx_with_hosts is not None else []
        self.nginx = {}
        if len(hosts) >= 1:
            self.nginx['production'] = hosts[0]
        if len(hosts) >= 2:
            self.nginx['testing'] = hosts[1]
            self.nginx['development'] = hosts[1]
        if len(hosts) >= 3:
            self.nginx['development'] = hosts[2]

    def __call__(self):
        self._create_dir('')

        self.create_config()
        self.create_bin()
        self.create_src()

        if self.options.package_type == 'debian':
            self.create_debian()

        self.create_yamake()

        self.style()

    def create_config(self):
        self._create_dir('config')

        self._write_file(
            'config/prepare.py',
            """import json
import sys


SETTINGS = {
    "development": {
        "SELF_TVM_ID": 00000,
    },
    "testing": {
        "SELF_TVM_ID": 00000,
    },
    "production": {
        "SELF_TVM_ID": 00000,
    },
}


def do(template, env):
    if env not in SETTINGS:
        raise Exception("env '%s' is unknown" % env)

    setts = SETTINGS[env]
    for key, value in setts.items():
        template = template.replace('<<%s>>' % key, str(value))

    if '<<' in template or '>>' in template:
        raise Exception("some options were not specified: %s\\n" % template)

    print(template)


if __name__ == '__main__':
    if len(sys.argv) != 3:
        print("Usage: %s <template_file> <env>" % (sys.argv[0]))
        sys.exit(1)

    with open(sys.argv[1]) as f:
        template = f.read()
        assert len(template) > 0

    do(template, sys.argv[2])""",
        )

        self._write_file(
            'config/template.json',
            self._format_json(
                """{
    |{http_config}|,
    "http_unistat": {
        "listen_address": "localhost",
        "port": |{unistat_port}|
    },
    "logger": {
        "file": "/var/log/yandex/passport-|{project_dir}|/common.log"
    },
    "service": {
        "tvm": {
            "self_tvm_id": <<SELF_TVM_ID>>,
            "secret_filepath": "/etc/yandex/passport-|{project_dir}|/tvm.secret",
            "disk_cache_dir": "/etc/yandex/passport-|{project_dir}|/tvm_cache"
        }
    }
}""",
                http_config=self.get_http_daemon_config(),
                unistat_port=self.options.unistat_port,
                project_dir=self.project_dir,
            ),
        )

        xuni_head = ''
        xuni = ''
        if self.options.add_xunistater_with_port is not None:
            xuni_head = '{name}responses = unistat'
            xuni = """
[options_{name}responses]
url = /{name}
port = {xunistat_port}
""".format(
                name=self.options.name,
                xunistat_port=self.options.add_xunistater_with_port,
            )

        self._write_file(
            'config/yasm.conf',
            (
                """[sources]
{name} = unistat
"""
                + xuni_head
                + """

[options_{name}]
url = /unistat
port = {unistat_port}
"""
                + xuni
            ).format(
                name=self.options.name,
                unistat_port=self.options.unistat_port,
            ),
        )

        self._create_dir('config/logstore')

        self._write_file(
            'config/logstore/{name}.json'.format(name=self.options.name),
            (
                """{
    "file": "/var/log/yandex/passport-|{project_dir}|/*.log",
    "host": "logstore-s1.passport.yandex.net"
}
"""
            ).format(
                project_dir=self.project_dir,
            ),
        )

        self._create_dir('config/yav_deploy')

        nginx = ''
        if self.options.add_nginx_with_hosts is not None:
            nginx = """
[nginx]
owner = www-data:www-data
prefix = /etc/nginx/certs
"""
        self._write_file(
            'config/yav_deploy/defaults.conf',
            (
                """[DEFAULT]
mode  = 0400
owner = root:root

[{name}]
owner = www-data:www-data
prefix = /etc/yandex/passport-{project_dir}
"""
                + nginx
            ).format(
                name=self.options.name,
                project_dir=self.project_dir,
            ),
        )

        def build_yav_env(env):
            nginx = ''
            if env in self.nginx:
                nginx = """
[nginx]
/{host}.crt = sec-00000000000000000000000000[{host}.crt]
/{host}.key = sec-00000000000000000000000000[{host}.key]
""".format(
                    host=self.nginx[env],
                )
            self._write_file(
                'config/yav_deploy/{env}.conf'.format(env=env),
                (
                    """[{name}]
/tvm.secret = sec-00000000000000000000000000[client_secret]
"""
                    + nginx
                ).format(
                    name=self.options.name,
                ),
            )

        build_yav_env('development')
        build_yav_env('testing')
        build_yav_env('production')

        if self.options.add_xunistater_with_port is not None:
            self._write_file(
                'config/xunistater.xml',
                """<?xml version="1.0"?>
<config>
    <http_daemon>
        <listen_address>localhost</listen_address>
        <port>{xunistat_port}</port>
        <threads>4</threads>
        <max_connections>4</max_connections>
        <max_queue_size>4</max_queue_size>
        <max_delay>1000</max_delay>
        <keep_alive>yes</keep_alive>
    </http_daemon>

    <component>
        <logger>
            <file>/var/log/yandex/passport-{project_dir}/xunistater.log</file>
            <level>DEBUG</level>
            <print-level>yes</print-level>
            <time-format>_DEFAULT_</time-format>
        </logger>

        <parser file_name="/var/log/nginx/{project_dir}-api.access.log" missingok="1">
            <http_path>/{name}</http_path>

            <signals>
                <signal type="rps">
                    <name>rps</name>
                </signal>
                <signal type="hgram">
                    <column>6</column>
                    <name>upstream_answer_time</name>
                    <allow_many_values>yes</allow_many_values>
                    <bounds>0,50,100,150,200,250,300,350,400,450,500,550,600,650,700,750,800,850,900,950,1000,1100,1200,1300,1400,1500,1600,1700,1800,1900,2000,2100,2200,2300,2400,2500,2600,2700,2800,2900,3000,3250,3500,3750,4000,4250,4500,4750,5000,10000</bounds>
                </signal>
                <signal type="hgram">
                    <column>7</column>
                    <name>full_answer_time</name>
                    <bounds>0,50,100,150,200,250,300,350,400,450,500,550,600,650,700,750,800,850,900,950,1000,1100,1200,1300,1400,1500,1600,1700,1800,1900,2000,2100,2200,2300,2400,2500,2600,2700,2800,2900,3000,3250,3500,3750,4000,4250,4500,4750,5000,10000</bounds>
                </signal>
                <signal type="string_set">
                    <column>9</column>
                    <prefix>code.</prefix>
                    <unquote>yes</unquote>
                    <persistent_name>200</persistent_name>
                    <persistent_name>400</persistent_name>
                    <persistent_name>403</persistent_name>
                    <persistent_name>499</persistent_name>
                    <persistent_name>500</persistent_name>
                    <persistent_name>502</persistent_name>
                    <persistent_name>503</persistent_name>
                    <persistent_name>504</persistent_name>
                </signal>
            </signals>
        </parser>
    </component>

    <daemon>
        <pidfile>/var/run/yandex/passport-xunistater/{name}.pid</pidfile>
    </daemon>
</config>
""".format(
                    name=self.options.name,
                    project_dir=self.project_dir,
                    xunistat_port=self.options.add_xunistater_with_port,
                ),
            )

        if self.options.add_nginx_with_hosts is not None:

            def gen_setts(env):
                if env not in self.nginx:
                    return ''
                ip = '::' if env == 'development' else socket.getaddrinfo(self.nginx[env], 443)[0][4][0]
                return """
        "HOST": '{host}',\n
        "IP": '{ip}',\n""".format(
                    host=self.nginx[env],
                    ip=ip,
                )

            # dev = self.nginx['development'] if 'development' in self.nginx else ''
            # test = self.nginx['testing'] if 'testing' in self.nginx else ''
            # prod = self.nginx['production'] if 'production' in self.nginx else ''
            self._write_file(
                'config/prepare_nginx.py',
                """import json
import sys


SETTINGS = {
    "development": {
"""
                + gen_setts('development')
                + """},
    "testing": {
"""
                + gen_setts('testing')
                + """},
    "production": {
"""
                + gen_setts('production')
                + """},
}


def do(template, env):
    if env not in SETTINGS:
        raise Exception("env '%s' is unknown" % env)

    setts = SETTINGS[env]
    for key, value in setts.items():
        template = template.replace('<<%s>>' % key, str(value))

    if '<<' in template or '>>' in template:
        raise Exception("some options were not specified: %s\\n" % template)

    print(template)


if __name__ == '__main__':
    if len(sys.argv) != 3:
        print("Usage: %s <template_file> <env>" % (sys.argv[0]))
        sys.exit(1)

    with open(sys.argv[1]) as f:
        template = f.read()
        assert len(template) > 0

    do(template, sys.argv[2])""",
            )

            self._write_file(
                'config/template_nginx.txt',
                """
upstream {name} {{
        keepalive 4;
        server 127.0.0.1:{common_port};
}}

log_format {name} '$remote_addr - $remote_user [$time_local] $upstream_response_time $request_time $request_length "$status" $body_bytes_sent "$scheme" "$host" $request_id';

server {{
        listen [<<IP>>]:443 default_server ipv6only=off ssl http2;

        ssl_certificate             /etc/nginx/certs/<<HOST>>.crt;
        ssl_certificate_key         /etc/nginx/certs/<<HOST>>.key;
        ssl_session_cache           shared:SSL:32m;
        ssl_session_timeout         24h;
        ssl_protocols               TLSv1.2 TLSv1.3;
        ssl_ciphers                 EECDH+AESGCM;
        ssl_ecdh_curve              X25519:prime256v1;
        ssl_prefer_server_ciphers   on;

        return 404;
}}

server {{
        listen [<<IP>>]:443;

        server_name <<HOST>>;

        ssl_certificate             /etc/nginx/certs/<<HOST>>.crt;
        ssl_certificate_key         /etc/nginx/certs/<<HOST>>.key;
        ssl_session_cache           shared:SSL:32m;
        ssl_session_timeout         24h;
        ssl_protocols               TLSv1.2 TLSv1.3;
        ssl_ciphers                 EECDH+AESGCM;
        ssl_ecdh_curve              X25519:prime256v1;
        ssl_prefer_server_ciphers   on;

        access_log /var/log/nginx/{project_dir}-api.access.log {name};
        error_log  /var/log/nginx/{project_dir}-api.error.log;

        location / {{
                proxy_pass                       http://{name};
                proxy_http_version               1.1;
                proxy_set_header Connection      "";
                proxy_set_header Host            $host;
                proxy_set_header X-Real-IP       $remote_addr;
                proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
                proxy_set_header X-Request-Id    $request_id;
        }}
}}
""".format(
                    common_port=self.options.common_port,
                    name=self.options.name,
                    project_dir=self.project_dir,
                ),
            )

            self._write_file(
                'config/nginx.conf',
                """
user                    www-data www-data;

pid                     /var/run/nginx.pid;
lock_file               /var/lock/nginx.lock;

worker_processes        24;
worker_rlimit_nofile    65536;
working_directory       /var/tmp;

events {
    use                 epoll;
    worker_connections  16384;
    multi_accept        off;
    accept_mutex        off;
}

http {
    root       /var/www;

    include         mime.types;
    default_type    text/plain;

    log_format      main    '$remote_addr - $host $remote_user [$time_local] $server_name:$server_port '
                            '$upstream_response_time $request_time $request_length "$request" '
                            '"$status" $body_bytes_sent "$http_referer" '
                            '"$http_user_agent" "$http_x_forwarded_for" "$http_cookie" "$gzip_ratio"';

    proxy_store                 off;
    proxy_buffering             on;
    proxy_buffers               128 16k;
    proxy_buffer_size           32k;

    server_name_in_redirect     off;
    server_tokens               off;

    keepalive_timeout           0;

    gzip                        off;

    client_body_temp_path       /dev/shm/nginx_client_body_temp;
    proxy_temp_path             /dev/shm/nginx_proxy_temp;

    client_header_buffer_size   16k;
    large_client_header_buffers 64 16k;
    client_body_buffer_size     1m;
    client_max_body_size        32m;

    client_header_timeout       60s;
    client_body_timeout         60s;
    send_timeout                60s;
    reset_timedout_connection   on;

    variables_hash_max_size     2048;

    include                     /etc/nginx/sites-enabled/*;
}
""",
            )

        self.dirs.remove('config')

    def create_debian(self):
        self._create_dir('debianization')

        xuni = ''
        deps = ''
        if self.options.add_xunistater_with_port is not None:
            xuni = """        {
            "source" : {
                "type" : "ARCADIA",
                "path" : "|{arcadia_related_path}|/config/xunistater.xml"
            },
            "destination" : {
                "path" : "/etc/yandex/passport-xunistater/|{name}|.conf"
            }
        },
"""
            deps += """\n            "yandex-passport-xunistater","""

        nginx = ''
        if self.options.add_nginx_with_hosts is not None:
            deps += """\n            "nginx","""
            nginx = """        {
            "source" : {
                "type" : "ARCADIA",
                "path" : "|{arcadia_related_path}|/config/nginx.conf"
            },
            "destination" : {
                "path" : "/etc/nginx/nginx.conf"
            }
        },
"""

        self._write_file(
            'debianization/debian.json',
            self._format_json(
                """{
    "meta" : {
        "name" : "yandex-passport-|{project_dir}|",
        "maintainer" : "|{author}|",
        "description" : "|{name}|",
        "version" : "1.0.0",
        "depends": [|{xunidep}|
            "yandex-passport-logstoreagent",
            "yandex-passport-vault-client",
            "passport-admin-config-golovan"
        ]
    },
    "build" : {
        "targets" :  [
            "|{arcadia_related_path}|/|{bin_dir_name}|"
        ]
    },
    "data" : [
        {
            "source" : {
                "type" : "BUILD_OUTPUT",
                "path" : "|{arcadia_related_path}|/|{bin_dir_name}|/|{name}|"
            },
            "destination" : {
                "path" : "/usr/sbin/"
            }
        },
        {
            "source" : {
                "type" : "ARCADIA",
                "path" : "|{arcadia_related_path}|/config",
                "files" : [
                    "*.py",
                    "template*"
                ]
            },
            "destination" : {
                "path" : "/etc/yandex/passport-|{project_dir}|/"
            }
        },
        {
            "source" : {
                "type" : "ARCADIA",
                "path" : "|{arcadia_related_path}|/config/logstore",
                "files": ["*"]
            },
            "destination" : {
                "path" : "/etc/yandex/passport-logstoreagent/instances/"
            }
        },
        {
            "source" : {
                "type" : "ARCADIA",
                "path" : "|{arcadia_related_path}|/config/yav_deploy",
                "files": ["*"]
            },
            "destination" : {
                "path" : "/etc/yandex/yav-deploy/pkg/yandex-passport-|{project_dir}|/"
            }
        },
        {
            "source" : {
                "type" : "ARCADIA",
                "path" : "|{arcadia_related_path}|/config/yasm.conf"
            },
            "destination" : {
                "path" : "/usr/local/yasmagent/CONF/agent.passport|{name}|.conf"
            }
        },
"""
                + xuni
                + nginx
                + """        {
            "source" : {
                "type" : "ARCADIA",
                "path" : "|{arcadia_related_path}|/debianization/yandex-passport-|{project_dir}|.service"
            },
            "destination" : {
                "path" : "/lib/systemd/system/"
            }
        }
    ]
}
""",
                name=self.options.name,
                author=self.options.author,
                arcadia_related_path=self.arcadia_related_path,
                bin_dir_name=self.bin_dir_name(),
                project_dir=self.project_dir,
                xunidep=deps,
            ),
        )
        self._write_file(
            'debianization/yandex-passport-%s.service' % self.project_dir,
            """[Unit]
Description={name}
AssertPathExists=/etc/yandex/passport-{project_dir}/tvm.secret

[Service]
User=www-data
Group=www-data
ExecStart=/usr/sbin/{name} -c /etc/yandex/passport-{project_dir}/{name}.conf
Restart=on-failure
RestartSec=1
LimitCORE=infinity
LimitNOFILE=262144
TasksMax=4096

[Install]
WantedBy=multi-user.target
""".format(
                name=self.options.name,
                project_dir=self.project_dir,
            ),
        )

        self._create_dir('debianization/debian')

        self._write_file(
            'debianization/debian/logrotate',
            """/var/log/yandex/passport-{project_dir}/*.log
{{
    daily
    rotate 7
    compress
    delaycompress
    missingok
    sharedscripts
    postrotate
        pkill -f -HUP {name}
    endscript
}}
""".format(
                name=self.options.name,
                project_dir=self.project_dir,
            ),
        )

        self._write_file(
            'debianization/debian/changelog',
            """yandex-passport-{project_dir} (1.0.0) unstable; urgency=low

  * Initial release

 -- {author}  {date}
""".format(
                project_dir=self.project_dir,
                author=self.options.author,
                date=datetime.datetime.now().strftime('%a, %-d %b %Y %H:%M:%S +0300'),
            ),
        )

        self._write_file(
            'debianization/debian/dirs',
            """/etc/yandex/passport-{project_dir}/tvm_cache
/var/log/yandex/passport-{project_dir}
""".format(
                project_dir=self.project_dir,
            ),
        )

        xuni = ''
        if self.options.add_xunistater_with_port is not None:
            xuni = """
    ## xunistater
    echo "Restarting xunistater."
    if ! systemctl is-enabled yandex-passport-xunistater@{name} > /dev/null; then
        systemctl enable yandex-passport-xunistater@{name} || true
    fi
    deb-systemd-invoke restart yandex-passport-xunistater@{name} || true
"""
        nginx = ''
        if self.options.add_nginx_with_hosts is not None:
            nginx = """
    ## nginx
    echo "Generating nginx config."
    python3 /etc/yandex/passport-{project_dir}/prepare_nginx.py \\
        "/etc/yandex/passport-{project_dir}/template_nginx.txt" \\
        "$CURRENT_ENV" > /etc/nginx/sites-enabled/{project_dir}.conf

    echo "Reloading nginx."
    systemctl reload nginx
"""

        self._write_file(
            'debianization/debian/postinst',
            (
                """#!/bin/sh
# postinst script for test-0.1
#
# see: dh_installdeb(1)

set -e

# summary of how this script can be called:
#        * <postinst> `configure' <most-recently-configured-version>
#        * <old-postinst> `abort-upgrade' <new version>
#        * <conflictor's-postinst> `abort-remove' `in-favour' <package>
#          <new-version>
#        * <postinst> `abort-remove'
#        * <deconfigured's-postinst> `abort-deconfigure' `in-favour'
#          <failed-install-package> <version> `removing'
#          <conflicting-package> <version>
# for details, see http://www.debian.org/doc/debian-policy/ or
# the debian-policy package

case "$1" in
    configure)

    if [ -z "${{2}}" ]; then
        yav-deploy --force --skip-post-update \\
            -c /etc/yandex/yav-deploy/pkg/yandex-passport-{project_dir}/ \\
            || true
    else
        yav-deploy --force \\
            -c /etc/yandex/yav-deploy/pkg/yandex-passport-{project_dir}/ \\
            || true
    fi

    CURRENT_ENV=`cat "/etc/yandex/environment.type"`

    ## common
    chown -R www-data: /etc/yandex/passport-{project_dir}
    chown -R www-data: /var/log/yandex/passport-{project_dir}
"""
                + xuni
                + nginx
                + """
    ## cfg
    echo "Generating config."
    python3 /etc/yandex/passport-{project_dir}/prepare.py \\
        "/etc/yandex/passport-{project_dir}/template.json" \\
        "$CURRENT_ENV" > /etc/yandex/passport-{project_dir}/{name}.conf

    ## yasm
    echo "Restarting yasmagent."
    YENV_PROJECT="passport.{name}" /usr/share/passport-admin-config-golovan/passport_golovan_config.py \\
        --itype="passport{name}" > \\
        /etc/yandex/yasmagent/instances.d/passport.{name}.instance
    invoke-rc.d yasmagent restart || /bin/true

    ## {name}
    echo "Updating sysmtemctl."
    if ! systemctl is-enabled yandex-passport-{project_dir} >/dev/null; then
        systemctl enable yandex-passport-{project_dir} || true
    fi
    systemctl --system daemon-reload || true

    echo "Trying to check logstoreagent configs..."
    sudo -u www-data /usr/sbin/logstoreagent_config_checker
    echo "Trying to check logstoreagent configs...OK"
    service yandex-passport-logstoreagent restart

    ;;

    abort-upgrade|abort-remove|abort-deconfigure)
    ;;

    *)
        echo "postinst called with unknown argument \\`$1'" >&2
        exit 1
    ;;
esac

# dh_installdeb will replace this with shell code automatically
# generated by other debhelper scripts.

#DEBHELPER#

exit 0
"""
            ).format(
                project_dir=self.project_dir,
                name=self.options.name,
            ),
        )

        self.dirs.remove('debianization')

    def create_yamake(self):
        files = ''
        for f in self.dirs:
            files += '    %s\n' % f

        self._write_file(
            'ya.make',
            """OWNER(%s)

RECURSE(
%s)"""
            % (self.options.owner, files),
        )

    def style(self):
        subprocess.check_call('ya fix-includes -r .'.split(' '), cwd=self.dir)
        subprocess.check_call('ya style .'.split(' '), cwd=self.dir)

    def get_ci_extra_directories(self):
        raise NotImplementedError

    def get_http_daemon_config(self):
        raise NotImplementedError

    def _create_dir(self, dir):
        if dir != '':
            self.dirs.add(dir.split('/', 1)[0])

        dir = '%s/%s' % (self.dir, dir)
        if self.options.verbose:
            print('Creating dir: %s ...' % dir)
        if self.options.add_only_new and isdir(dir):
            if self.options.verbose:
                print('dir already exists: %s . Do nothing' % dir)
            return

        os.mkdir(dir)
        if self.options.verbose:
            print('Created dir: %s !' % dir)

    def _write_file(self, filename, filebody):
        filename = '%s/%s' % (self.dir, filename)
        if not filebody.endswith('\n'):
            filebody = filebody + '\n'

        if self.options.verbose:
            print('Writing file: %s ...' % filename)
        if self.options.add_only_new and isfile(filename):
            if self.options.verbose:
                print('file already exists: %s . Do nothing' % filename)
            return

        with open(filename, 'w') as f:
            f.write(filebody)
        if self.options.verbose:
            print('Writed file: %s !' % filename)

    def _get_arcadia_related_path(self, path):
        original_path = path
        root = path
        while not isfile(root + '/.arcadia.root'):
            if root == '/':
                raise Exception('%s is not in Arcadia!' % original_path)
            root = '/'.join(root.split('/')[:-1])
        return os.path.relpath(original_path, root)

    def _format_json(self, format, **kwargs):
        format = format.replace('{', '{{')
        format = format.replace('}', '}}')
        format = format.replace('|{{', '{')
        format = format.replace('}}|', '}')

        return format.format(**kwargs)


class Cpp(Base):
    def __init__(self, args):
        super(Cpp, self).__init__(args)

        self.package_name = self.options.name.replace('_', '')

        tokens = [t[:1].upper() + t[1:] for t in self.options.name.split('_')]
        self.struct_name = ''.join(tokens)
        self.namespace_name = 'N{}'.format(self.struct_name)
        self.class_name = 'T{}'.format(self.struct_name)

    def add_arguments(self, parser):
        pass

    def style(self):
        super(Cpp, self).style()

    def bin_dir_name(self):
        return 'daemon'

    def create_bin(self):
        self._create_dir('daemon')
        self._write_file(
            'daemon/main.cpp',
            """#include <{dst_dir}/src/{package_name}.h>

#include <passport/infra/libs/cpp/http_daemon/main.h>

int main(int argc, char** argv) {{
    using namespace NPassport;
    return mainImpl<{namespace_name}::{class_name}, NJson::TConfig>("{package_name}", argc, argv);
}}
            """.format(
                dst_dir=self.arcadia_related_path,
                package_name=self.package_name,
                namespace_name=self.namespace_name,
                class_name=self.class_name,
            ),
        )
        self._write_file(
            'daemon/ya.make',
            """PROGRAM({package_name})

OWNER(g:passport_infra)

PEERDIR(
    {dst_dir}/src
    passport/infra/libs/cpp/http_daemon
)

SRCS(
    main.cpp
)

END()
""".format(
                dst_dir=self.arcadia_related_path, package_name=self.package_name
            ),
        )

    def create_src(self):
        self._create_dir('src')
        self._write_file(
            'src/{package_name}.cpp'.format(
                package_name=self.package_name,
            ),
            """#include "{package_name}.h"

#include <passport/infra/libs/cpp/utils/log/global.h>

namespace NPassport::{namespace_name} {{
    {class_name}::{class_name}()
    {{
    }}

    void {class_name}::init(const NJson::TConfig& config) {{
        Y_UNUSED(config);
    }}

    void {class_name}::handleRequest(NCommon::TRequest& req) {{
        Y_UNUSED(req);
    }}

    void {class_name}::addUnistat(NUnistat::TBuilder& builder) {{
        Y_UNUSED(builder);
    }}
}}
""".format(
                package_name=self.package_name,
                namespace_name=self.namespace_name,
                class_name=self.class_name,
            ),
        )

        self._write_file(
            'src/{package_name}.h'.format(
                package_name=self.package_name,
            ),
            """#pragma once
#include <passport/infra/libs/cpp/request/request.h>

namespace NPassport::NJson {{
    class TConfig;
}}

namespace NPassport::NUnistat {{
    class TBuilder;
}}

namespace NPassport::{namespace_name} {{
    class {class_name} {{
    public:
        {class_name}();
        ~{class_name}() = default;

        void init(const NJson::TConfig& config);

        void handleRequest(NCommon::TRequest& req);

        void addUnistat(NUnistat::TBuilder& builder);
    }};
}}
""".format(
                class_name=self.class_name, namespace_name=self.namespace_name
            ),
        )

        self._write_file(
            'src/ya.make',
            """LIBRARY()

OWNER(g:passport_infra)

PEERDIR(
    passport/infra/libs/cpp/utils
    passport/infra/libs/cpp/unistat
)

SRCS(
    {package_name}.cpp
)

END()
""".format(
                package_name=self.package_name
            ),
        )

    def get_ci_extra_directories(self):
        return """- passport/infra/libs/cpp/http_daemon
    - passport/infra/libs/cpp/unistat
    - passport/infra/libs/cpp/utils"""

    def get_http_daemon_config(self):
        return """"http_daemon": {{
    "listen_address": "localhost",
        "ports": [
            {{
                "port": {common_port}
            }}
        ],
        "threads": 1,
        "max_connections": 4,
        "max_queue_size": 4
    }}""".format(
            common_port=self.options.common_port,
        )


class Go(Base):
    def __init__(self, args):
        super(Go, self).__init__(args)

        self.package_name = self.options.name.replace('_', '')

        tokens = [t[:1].upper() + t[1:] for t in self.options.name.split('_')]
        self.struct_name = ''.join(tokens)

    def add_arguments(self, parser):
        pass

    def style(self):
        super(Go, self).style()
        subprocess.check_call('ya tool yo fix .'.split(' '), cwd=self.dir)

    def bin_dir_name(self):
        return 'cmd'

    def create_bin(self):
        self._create_dir('cmd')
        self._write_file(
            'cmd/main.go',
            """package main

import (
\t"a.yandex-team.ru/{dst_dir}/internal/{package_name}"
\t"a.yandex-team.ru/passport/shared/golibs/httpdaemon"
)

func main() {{
\thttpdaemon.SimpleRun(&{package_name}.Factory{{}})
}}
""".format(
                dst_dir=self.arcadia_related_path,
                package_name=self.package_name,
            ),
        )
        self._write_file(
            'cmd/ya.make',
            """GO_PROGRAM(%s)

SRCS(main.go)

END()
"""
            % self.options.name,
        )

    def create_src(self):
        self._create_dir('internal')
        self._create_dir('internal/' + self.package_name)

        self._write_file(
            'internal/{package_name}/{package_name}.go'.format(
                package_name=self.package_name,
            ),
            """package {package_name}

import (
\t"net/http"
\t"github.com/labstack/echo/v4"
\t"a.yandex-team.ru/library/go/yandex/tvm"
\t"a.yandex-team.ru/passport/shared/golibs/httpdaemon"
\t"a.yandex-team.ru/passport/shared/golibs/logger"
\t"a.yandex-team.ru/passport/shared/golibs/unistat"
)

type {struct_name} struct {{
\ttvm       tvm.Client
\tunistat   stats
}}

type Config struct {{
\tTvm    httpdtvm.TvmConfig `json:"tvm"`
}}

type stats struct {{
}}

type Factory struct{{}}

func (f *Factory) NewService(config httpdaemon.ServiceConfig) (httpdaemon.Service, error) {{
\tvar cfg Config
\tif err := httpdaemon.ParseServiceConfig(config, &cfg); err != nil {{
\t\treturn nil, err
\t}}

\ttvmClient, err := httpdtvm.InitTvm(cfg.Tvm)
\tif err != nil {{
\t\treturn nil, err
\t}}

\tres := &{struct_name}{{
\ttvm:       tvmClient,
\t\tunistat: stats{{
\t\t}},
\t}}
\treturn res, nil
}}

func (t *{struct_name}) AddHandlers(e *echo.Echo) {{
\te.GET(
\t\t"/ping",
\t\tt.HandlePing(),
\t)
}}

func (t *{struct_name}) HandlePing() echo.HandlerFunc {{
\treturn func(c echo.Context) error {{
\t\tlogger.Log().Debugf("Ping: service is up")
\t\treturn c.String(http.StatusOK, "")
\t}}
}}
""".format(
                package_name=self.package_name,
                struct_name=self.struct_name,
            ),
        )

    def get_http_daemon_config(self):
        return """"http_common": {{
        "listen_address": "localhost",
        "port": {common_port}
    }}""".format(
            common_port=self.options.common_port
        )

    def get_ci_extra_directories(self):
        return """- passport/shared/golibs/httpdaemon
    - passport/shared/golibs/logger
    - passport/shared/golibs/unistat"""


commands = {
    'cpp': Cpp,
    'go': Go,
}


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('command', choices=commands)
    args = sys.argv[1:2]
    command_args = sys.argv[2:]
    options = parser.parse_args(args)
    command = commands[options.command](command_args)
    command()


if __name__ == '__main__':
    main()
