import datetime
import json
import requests
import socket
import sys
import xml.etree.ElementTree as xml_


TOKEN_HEADER = None
ABC_INFO = {
    2444: {'abc_admins': [], 'abc_managers': ['oxcd8o']},
    908: {'abc_admins': [], 'abc_managers': ['medanya']},
}
CONDUCTOR_INFO = {
    '%is.splunk-master': {
        'c_admins': [
            'a-abakumov',
            'buglloc',
            'gots',
            'anton-k',
            'ezaitov',
            'spellanser',
            'strawberry',
            'procenkoeg',
            'limetime',
            'shashial',
            'energen',
            'horus',
            'melkikh',
            'shikari',
        ]
    },
}
FW_INFO = {
    '_YMBACKNETS_': {'fw_owners': ['ymvorobevda']},
    '_SPNETS_': {'fw_owners': ['batuto']},
    '_YM_SCRAT_BACK_': {'fw_owners': ['pkzimin', 'smirnovmr', 'lokshin', 'eemolchanov']},
    '_QUERYHISTORY_': {
        'fw_owners': [
            'maxk',
            'stunder',
            'squirrel',
            'sivanichkin',
            'mixas',
            'jack239',
            'efmv',
            'le087',
            'mbabich',
            'alovtsyus',
            'dldmitry',
            'monkey',
            'alximik',
            'rikevoltz',
            'flagist',
            'affe',
            'xifos',
            'rocco66',
            'knuzhdin',
            'alexsmirnov',
            'artgromov',
            'nulltime',
            'zuci42',
            'qwizzy',
            'warwish',
            'ndolganov',
            'n-malakhov',
            'basic',
            'k-malov',
            'noiseless',
            'reddog',
            'wint',
        ]
    },
    '_BLOGSNETS_': {
        'fw_owners': [
            'efmv',
            'rikevoltz',
            'le087',
            'mbabich',
            'basic',
            'alovtsyus',
            'stunder',
            'sivanichkin',
            'reddog',
            'dldmitry',
            'jack239',
            'rocco66',
            'n-malakhov',
            'flagist',
            'knuzhdin',
            'maxk',
            'wint',
            'artgromov',
            'nulltime',
            'warwish',
            'mixas',
            'monkey',
            'xifos',
            'affe',
            'alexsmirnov',
            'qwizzy',
            'noiseless',
            'ndolganov',
            'k-malov',
            'zuci42',
            'alximik',
            'squirrel',
        ]
    },
    '_WORKSPACENETS_': {'fw_owners': ['khunafin', 'chapson', 'itjune', 'art']},
    '_SEARCHMETANETS_': {'fw_owners': ['buglloc', 'a-abakumov']},
    '_SEARCHSAND_': {'fw_owners': ['buglloc', 'a-abakumov']},
    '_SEARCHROBOT_': {'fw_owners': ['buglloc', 'a-abakumov']},
}
HOST_INFO = {
    'saw-stv02h.maps.yandex-team.ru': {'host_owners': ['ykirpichev']},
}
STAFF_INFO = {}

IS_NETWORK_INFO_BY_SUB_INITED = None
NETWORK_INFO_BY_SUB = {
    '185.71.76.0/27': {'owners': ['gromovad', 'carev', 'pavlyachek', 'budylnikov']},
}
USER_IS_BAD = {}

SUMMARY = "Актуализация грантов в passport-api для потребителя "

SKIP_IP_CONSUMERS = [
    'morda',  # https://st.yandex-team.ru/PASSPORTGRANTS-2078
    'browser-api',  # https://st.yandex-team.ru/PASSPORTGRANTS-2079
    'taxi-partner-contracts',  # https://st.yandex-team.ru/PASSPORTGRANTS-2080
    'oauth_admin',  # https://st.yandex-team.ru/PASSPORTGRANTS-2081
    'xmlsearch',  # https://st.yandex-team.ru/PASSPORTGRANTS-2082
    'kinopoisk',  # https://st.yandex-team.ru/PASSPORTGRANTS-2083
    'wwwdgt',  # https://st.yandex-team.ru/PASSPORTGRANTS-2084
    'taxi_study',  # https://st.yandex-team.ru/PASSPORTGRANTS-2085
    'merchant',  # https://st.yandex-team.ru/PASSPORTGRANTS-2086
    'syncserver',  # https://st.yandex-team.ru/PASSPORTGRANTS-2087
    'daria2',  # https://st.yandex-team.ru/PASSPORTGRANTS-2089
    'yasms',  # https://st.yandex-team.ru/PASSPORTGRANTS-2090
    'gazeta',  # https://st.yandex-team.ru/PASSPORTGRANTS-2091
    'public',  # https://st.yandex-team.ru/PASSPORTGRANTS-2092
    'radiojam',  # https://st.yandex-team.ru/PASSPORTGRANTS-2093
    'betatest',  # https://st.yandex-team.ru/PASSPORTGRANTS-2094
    'mailadmins',  # https://st.yandex-team.ru/PASSPORTGRANTS-2095
    'webreport',  # https://st.yandex-team.ru/PASSPORTGRANTS-2096
    'trust',  # https://st.yandex-team.ru/PASSPORTGRANTS-2097
    'avia',  # https://st.yandex-team.ru/PASSPORTGRANTS-2098
    'publicdemo',  # https://st.yandex-team.ru/PASSPORTGRANTS-2099
    'friends',  # https://st.yandex-team.ru/PASSPORTGRANTS-2100
    'kuda',  # https://st.yandex-team.ru/PASSPORTGRANTS-2101
    'bugbounty',  # https://st.yandex-team.ru/PASSPORTGRANTS-2102
    'webmaster',  # https://st.yandex-team.ru/PASSPORTGRANTS-2103
    'avtofotograf',  # https://st.yandex-team.ru/PASSPORTGRANTS-2104
    'octopus',  # https://st.yandex-team.ru/PASSPORTGRANTS-2105
    'cloudmigrant',  # https://st.yandex-team.ru/PASSPORTGRANTS-2106
    'education',  # https://st.yandex-team.ru/PASSPORTGRANTS-2107
    'psa_accounts',  # https://st.yandex-team.ru/PASSPORTGRANTS-2108
    'calendar',  # https://st.yandex-team.ru/PASSPORTGRANTS-2109
    'botplatform',  # https://st.yandex-team.ru/PASSPORTGRANTS-2110
    'create_test_yandex_login_yndx',  # https://st.yandex-team.ru/PASSPORTGRANTS-2111
    'islandsapi',  # https://st.yandex-team.ru/PASSPORTGRANTS-2112
    'sitesearch',  # https://st.yandex-team.ru/PASSPORTGRANTS-2113
    'disk',  # https://st.yandex-team.ru/PASSPORTGRANTS-2114
    'slovari',  # https://st.yandex-team.ru/PASSPORTGRANTS-2115
    'poputka',  # https://st.yandex-team.ru/PASSPORTGRANTS-2116
    'cloudsoft',  # https://st.yandex-team.ru/PASSPORTGRANTS-2117
    'podpiska',  # https://st.yandex-team.ru/PASSPORTGRANTS-2118
    'xeno',  # https://st.yandex-team.ru/PASSPORTGRANTS-2119
    'kassa',  # https://st.yandex-team.ru/PASSPORTGRANTS-2120
    'so',  # https://st.yandex-team.ru/PASSPORTGRANTS-2121
    'mobilecloud',  # https://st.yandex-team.ru/PASSPORTGRANTS-2122
    'direct_ui',  # https://st.yandex-team.ru/PASSPORTGRANTS-2123
    'market-api',  # https://st.yandex-team.ru/PASSPORTGRANTS-2124
    'create_test_yandex_login_yandex-team',  # https://st.yandex-team.ru/PASSPORTGRANTS-2125
    'cloudpaid',  # https://st.yandex-team.ru/PASSPORTGRANTS-2126
    'historydb',  # https://st.yandex-team.ru/PASSPORTGRANTS-2127
    'mynews',  # https://st.yandex-team.ru/PASSPORTGRANTS-2128
    'gorynych',  # https://st.yandex-team.ru/PASSPORTGRANTS-2129
    'cloud',  # https://st.yandex-team.ru/PASSPORTGRANTS-2130
    'fotki',  # https://st.yandex-team.ru/PASSPORTGRANTS-2131
    'taxi',  # https://st.yandex-team.ru/PASSPORTGRANTS-2132
    'livemail',  # https://st.yandex-team.ru/PASSPORTGRANTS-2133
    'zen',  # https://st.yandex-team.ru/PASSPORTGRANTS-2134
    'toloka',  # https://st.yandex-team.ru/PASSPORTGRANTS-2135
    'strongpwd',  # https://st.yandex-team.ru/PASSPORTGRANTS-2136
    'blackbox',  # https://st.yandex-team.ru/PASSPORTGRANTS-2137
    'market',  # https://st.yandex-team.ru/PASSPORTGRANTS-2138
    'yandex-bus',  # https://st.yandex-team.ru/PASSPORTGRANTS-2139
    'oauth',  # https://st.yandex-team.ru/PASSPORTGRANTS-2140
    'slova',  # https://st.yandex-team.ru/PASSPORTGRANTS-2141
    'kopalka',  # https://st.yandex-team.ru/PASSPORTGRANTS-2142
    'parking-gate',  # https://st.yandex-team.ru/PASSPORTGRANTS-2143
    'newspartner',  # https://st.yandex-team.ru/PASSPORTGRANTS-2144
    'yapic',  # https://st.yandex-team.ru/PASSPORTGRANTS-2145
    'musicpremium',  # https://st.yandex-team.ru/PASSPORTGRANTS-2146
    'mail-qa',  # https://st.yandex-team.ru/PASSPORTGRANTS-2147
    'upravlyator',  # https://st.yandex-team.ru/PASSPORTGRANTS-2148
    'carsharing',  # https://st.yandex-team.ru/PASSPORTGRANTS-2149
    'adfox',  # https://st.yandex-team.ru/PASSPORTGRANTS-2150
    'medicine',  # https://st.yandex-team.ru/PASSPORTGRANTS-2151
    'musicpartner',  # https://st.yandex-team.ru/PASSPORTGRANTS-2152
    'socialism',  # https://st.yandex-team.ru/PASSPORTGRANTS-2153
    'mailadm',  # https://st.yandex-team.ru/PASSPORTGRANTS-2154
    'tune',  # https://st.yandex-team.ru/PASSPORTGRANTS-2155
    'pddback',  # https://st.yandex-team.ru/PASSPORTGRANTS-2156
    'pddeula',  # https://st.yandex-team.ru/PASSPORTGRANTS-2157
    'telephony',  # https://st.yandex-team.ru/PASSPORTGRANTS-2158
    'music',  # https://st.yandex-team.ru/PASSPORTGRANTS-2159
    'mobileproxy',  # https://st.yandex-team.ru/PASSPORTGRANTS-2160
    'realty',  # https://st.yandex-team.ru/PASSPORTGRANTS-2161
    'dev',  # https://st.yandex-team.ru/PASSPORTGRANTS-2162
    'pi',  # https://st.yandex-team.ru/PASSPORTGRANTS-2163
    'meltingpot',  # https://st.yandex-team.ru/PASSPORTGRANTS-2164
    'mobilemusic',  # https://st.yandex-team.ru/PASSPORTGRANTS-2165
    'sauth',  # https://st.yandex-team.ru/PASSPORTGRANTS-2166
    'passport',  # https://st.yandex-team.ru/PASSPORTGRANTS-2167
    'yastaff',  # https://st.yandex-team.ru/PASSPORTGRANTS-2168
    'taximeter',  # https://st.yandex-team.ru/PASSPORTGRANTS-2169
    'mail',  # https://st.yandex-team.ru/PASSPORTGRANTS-2170
    'distribution',  # https://st.yandex-team.ru/PASSPORTGRANTS-2171
    'realty-moderator',  # https://st.yandex-team.ru/PASSPORTGRANTS-2172
    'directory',  # https://st.yandex-team.ru/PASSPORTGRANTS-2173
    'wdgt',  # https://st.yandex-team.ru/PASSPORTGRANTS-2174
    'adminka',  # https://st.yandex-team.ru/PASSPORTGRANTS-2175
    'personal_cards',  # https://st.yandex-team.ru/PASSPORTGRANTS-2176
    'money',  # https://st.yandex-team.ru/PASSPORTGRANTS-2177
    'sologger.mail.yandex.net',  # https://st.yandex-team.ru/PASSPORTGRANTS-2178
    'spdaemon',  # https://st.yandex-team.ru/PASSPORTGRANTS-2179
    'disk-dataapi',  # https://st.yandex-team.ru/PASSPORTGRANTS-2180
    'metrika',  # https://st.yandex-team.ru/PASSPORTGRANTS-2181
    'directory-portal',  # https://st.yandex-team.ru/PASSPORTGRANTS-2182
]

SKIP_TVM_CONSUMERS = [
    'taxi-admin',  # https://st.yandex-team.ru/PASSPORTGRANTS-2183
    'taxi-integration-api',  # https://st.yandex-team.ru/PASSPORTGRANTS-2184
    'taxi-driver_protocol',  # https://st.yandex-team.ru/PASSPORTGRANTS-2185
    'taxi-exam-backend',  # https://st.yandex-team.ru/PASSPORTGRANTS-2186
    'taxi-companion',  # https://st.yandex-team.ru/PASSPORTGRANTS-2187
    'aura',  # https://st.yandex-team.ru/PASSPORTGRANTS-2188
    'cloud_plus',  # https://st.yandex-team.ru/PASSPORTGRANTS-2189
    'talents',  # https://st.yandex-team.ru/PASSPORTGRANTS-2190
    'taxi-corp',  # https://st.yandex-team.ru/PASSPORTGRANTS-2191
    'taxi-selfemployed',  # https://st.yandex-team.ru/PASSPORTGRANTS-2192
    'yamb',  # https://st.yandex-team.ru/PASSPORTGRANTS-2193
    'marketplace-api',  # https://st.yandex-team.ru/PASSPORTGRANTS-2194
    'takeout',  # https://st.yandex-team.ru/PASSPORTGRANTS-2195
    'kopusha',  # https://st.yandex-team.ru/PASSPORTGRANTS-2196
    'chats-with-businesses',  # https://st.yandex-team.ru/PASSPORTGRANTS-2197
    'mssngr',  # https://st.yandex-team.ru/PASSPORTGRANTS-2198
    'pdd-adapter',  # https://st.yandex-team.ru/PASSPORTGRANTS-2199
]


def text_header(consumer_name, is_used):
    if is_used:
        sub = "**есть** запросы"
    else:
        sub = "**нет** запросов"

    return (
        "=== Описание события\n"
        "\n"
        "Привет!\n"
        "В рамках подготовки сервиса Яндекс.Паспорт к аудиту мы запустили процесс пересмотра выданных прав потребителям внешнего (не корпоративного) Паспорта.\n"
        "\n"
        "Для потребителя **%s** был выдан доступ на использование методов и данных Паспорта. По результатам исследования (с 2019.04.17 до 2019.05.16) в логах Паспорта %s от этого потребителя.\n"
        % (consumer_name, sub)
    )


def text_footer():
    return (
        "=== Что нужно сделать\n"
        "\n"
        "Внимательно ознакомься со списком выданных прав. Расшифровку прав (грантов) можно найти ((https://wiki.yandex-team.ru/users/lspliner/soc3passportgrants/ здесь)).\n"
        "Если они действительно все еще актуальны, то **напиши** об этом в комментарии и **закрой** тикет.\n"
        "Если какие-либо из выданных прав больше не требуются, сообщи об этом в тикете и призови lspliner@.\n"
        "В случае если ты больше не занимаешься этим вопросом, мы будем признательны, если ты призовешь актуального ответственного.\n"
        "**Дедлайн: 14 июня 2019**\n"
        "\n"
        "=== Обратная связь\n"
        "\n"
        "Если есть вопросы, просто призови в тикет irmin@.\n"
    )


def ip_to_hostname(ip):
    try:
        return socket.gethostbyaddr(ip)[0]
    except:
        pass


def get_autorization():
    global TOKEN_HEADER
    if TOKEN_HEADER is not None:
        return TOKEN_HEADER
    with open('./token') as f:
        TOKEN_HEADER = {'Authorization': 'OAuth %s' % f.read()}
    return TOKEN_HEADER


def try_do(func, always_raise=False):
    tries = 5
    while tries > 0:
        tries = tries - 1
        try:
            return func()
        except Exception as e:
            print('exception: %s' % (e), file=sys.stderr)
            if always_raise:
                raise


def get_abc_info(abc_id):
    if abc_id in ABC_INFO:
        return ABC_INFO[abc_id]

    def foo():
        next = (
            'https://abc-back.yandex-team.ru/api/v3/services/members/?role__scope=administration&role__scope=services_management&service=%s'
            % abc_id
        )
        res = {
            'abc_admins': [],
            'abc_managers': [],
        }

        while next is not None:
            r = requests.get(next, headers=get_autorization())
            if r.status_code != 200 and r.status_code < 500:
                print('abc service %s was not found (%s)' % (abc_id, r.status_code), file=sys.stderr)
                return
            assert r.status_code == 200, '(%s) %s' % (r.text, r.status_code)

            j = json.loads(r.text)
            next = j['next']

            res['abc_admins'] = res['abc_admins'] + [
                a['person']['login'] for a in j['results'] if a['role']['scope']['slug'] == 'administration'
            ]
            res['abc_managers'] = res['abc_managers'] + [
                a['person']['login'] for a in j['results'] if a['role']['scope']['slug'] == 'services_management'
            ]

        return res

    res = try_do(foo)
    ABC_INFO[abc_id] = res
    return res


def filter_out_bad_users(logins):
    to_check = []
    result = []
    for log in logins:
        if log in USER_IS_BAD:
            if USER_IS_BAD[log] == False:
                result.append(log)
        else:
            to_check.append(log)

    if len(to_check) == 0:
        return result

    def foo():
        next = (
            'https://staff-api.yandex-team.ru/v3/persons?_fields=official.is_dismissed,official.is_robot,login&_limit=100&login=%s'
            % (','.join(to_check))
        )

        while next is not None:
            r = requests.get(next, headers=get_autorization())
            if r.status_code != 200 and r.status_code < 500:
                print('logins in staff-api %s was not found (%s)' % (abc_id, r.status_code), file=sys.stderr)
                return
            assert r.status_code == 200, '(%s) %s' % (r.text, r.status_code)

            j = json.loads(r.text)
            next = None

            res = {}
            for r in j['result']:
                log = r['login']
                is_dismissed = r['official']['is_dismissed']
                is_robot = r['official']['is_robot']
                print('%s is dismissed: %s. is robot: %s' % (log, is_dismissed, is_robot), file=sys.stderr)

                res[log] = is_dismissed or is_robot

            for log in to_check:
                if log in res:
                    USER_IS_BAD[log] = res[log]
                else:
                    USER_IS_BAD[log] = True
                    print('%s not found in response' % (log), file=sys.stderr)

    try_do(foo)
    return [a for a in logins if a in USER_IS_BAD and USER_IS_BAD[a] == False]


class BaseStats:
    def __init__(self):
        self._consumers = {}
        self._used_consumers = {}

    def found(self, consumer):
        if consumer in self._consumers:
            self._used_consumers[consumer] = self._consumers[consumer]
            del self._consumers[consumer]

    def as_string_used(self):
        return self.__as_string_impl(self._used_consumers, True)

    def as_string_unused(self):
        return self.__as_string_impl(self._consumers, False)

    def __as_string_impl(self, cont, is_used):
        res = {}

        for k, v in cont.items():
            r = self._as_string(k, v)
            t = self.build_text(r, is_used, v)
            out = {'text': t, 'summary': SUMMARY + r['name'], 'deadline': '2019-06-14T18:00:00.00+0000'}

            def add_users(type):
                if type in r and len(r[type]) > 0:
                    out[type] = [x for x in sorted(filter_out_bad_users(list(set(r[type]))))]

            add_users('c_admins')
            add_users('abc_admins')
            add_users('abc_managers')
            add_users('abc_owners')
            add_users('fw_owners')
            add_users('host_owners')
            add_users('staff_owners')
            add_users('net_owners')

            def add_field(name):
                if name in r:
                    out[name] = r[name]

            add_field('tvm_id')
            add_field('warning')
            add_field('error')

            if (
                'c_admins' not in out
                and 'abc_admins' not in out
                and 'abc_owners' not in out
                and 'fw_owners' not in out
                and 'host_owners' not in out
                and 'staff_owners' not in out
                and 'abc_managers' not in out
                and 'net_owners' not in out
            ):
                print('nobody for %s' % r['name'])

            res[r['name']] = out

        return json.dumps(res, indent=4, sort_keys=True)


class IpStats(BaseStats):
    def build_text(self, r, is_used, meta):
        macros, grants = meta

        macros = [a[0] for a in macros]

        source = (
            "<{Доступ был выдан без TVM-аутентификации для:\n%%(json)"
            + json.dumps(macros, indent=4, sort_keys=True)
            + "%%}>\n"
        )
        gr_text = (
            "<{Доступ был выдан с правами (грантами) на:\n%%(json)"
            + json.dumps(grants, indent=4, sort_keys=True)
            + "%%}>\n"
        )

        return text_header(r['name'], is_used) + source + gr_text + text_footer()

    def add(self, consumer, grants):
        if consumer not in self._consumers:
            self._consumers[consumer] = ([], grants)

    def tsv_to_macro(self, f):
        for line in open(f):
            values = line[:-1].split('|')
            if len(values) != 5:
                continue
            m = values[1].strip()
            type = values[2].strip()
            name = values[3].strip()

            if name not in self._consumers:
                continue

            macros, grants = self._consumers[name]
            macros.append((m, type))
            self._consumers[name] = (macros, grants)

    def get_conductor_info(self, macro):
        if macro in CONDUCTOR_INFO:
            return CONDUCTOR_INFO[macro]

        def foo():
            r = requests.get(
                'https://c.yandex-team.ru/api/v2/groups/%s/project' % macro.strip('%'), headers=get_autorization()
            )
            if r.status_code != 200 and r.status_code < 500:
                print('conductor macro %s was not found (%s)' % (macro, r.status_code), file=sys.stderr)
                return
            assert r.status_code == 200, '(%s) %s' % (r.text, r.status_code)

            j = json.loads(r.text)
            abc = j['data']['attributes']['abc_service_id']
            if abc is None:
                print('conductor macro %s with none abc' % (macro), file=sys.stderr)
            admins_link = j['data']['relationships']['admins']['links']['related']

            return abc, admins_link

        res = try_do(foo)
        if res is None:
            return

        abc, admins_link = res
        if abc is not None:
            CONDUCTOR_INFO[macro] = {'abc': abc}
            return {'abc': abc}

        def foo():
            r = requests.get(admins_link, headers=get_autorization())
            if r.status_code != 200 and r.status_code < 500:
                print('conductor link was bad: %s (%s)' % (admins_link, r.status_code), file=sys.stderr)
                return
            assert r.status_code == 200, '(%s) %s' % (r.text, r.status_code)

            j = json.loads(r.text)
            admins = []
            for d in j['data']:
                admins.append(d['attributes']['login'])

            return admins

        admins = try_do(foo)
        if admins is None or len(admins) == 0:
            print('conductor macro %s has no admins' % macro, file=sys.stderr)
            return None

        print('conductor macro %s has admins: %s' % (macro, admins), file=sys.stderr)
        CONDUCTOR_INFO[macro] = {'c_admins': admins}
        return {'c_admins': admins}

    def get_firewall_info(self, macro):
        if macro in FW_INFO:
            return FW_INFO[macro]

        def foo():
            r = requests.get(
                'https://racktables.yandex.net/export/project-id-networks.php?op=show_macro&name=%s' % macro,
                headers=get_autorization(),
            )
            if r.status_code != 200 and r.status_code < 500:
                print('admins for macro faield to be got: %s (%s)' % (macro, r.status_code), file=sys.stderr)
                return
            assert r.status_code == 200, '(%s) %s' % (r.text, r.status_code)

            j = json.loads(r.text)
            return j['owners']

        res = try_do(foo)
        if res is None:
            FW_INFO[macro] = res
            return

        print('macro -> owners: %s -> %s' % (macro, res), file=sys.stderr)
        res = {'fw_owners': res}
        FW_INFO[macro] = res
        return res

    def get_firewall_hosts(self, macro):
        def foo():
            r = requests.get('http://hbf.yandex.net/macros/%s' % macro)
            if r.status_code != 200 and r.status_code < 500:
                print('hosts for macro faield to be got: %s (%s)' % (macro, r.status_code), file=sys.stderr)
                return
            assert r.status_code == 200, '(%s) %s' % (r.text, r.status_code)

            j = json.loads(r.text)
            return j

        res = try_do(foo)
        if res is None:
            return

        print('macro -> hosts: %s -> %s' % (macro, res), file=sys.stderr)
        return res

    def get_host_info(self, host):
        if host in HOST_INFO:
            return HOST_INFO[host]

        def foo():
            r = requests.get('http://ro.admin.yandex-team.ru/api/get_object_resps.sbml?object=%s' % host)
            if r.status_code != 200 and r.status_code < 500:
                print('admins for host faield to be got: %s (%s)' % (host, r.status_code), file=sys.stderr)
                return
            assert r.status_code == 200, '(%s) %s' % (r.text, r.status_code)

            j = json.loads(r.text)
            return [a['name'] for a in j['resps'] + j['resps_expand'] if a['type'] == 'user']

        res = try_do(foo)
        if res is None:
            return

        print('host -> owners: %s -> %s' % (host, res), file=sys.stderr)

        res = {'host_owners': res}
        HOST_INFO[host] = res
        return res

    def get_staff_group_members(self, group):
        if group in STAFF_INFO:
            return STAFF_INFO[group]

        def foo():
            r = requests.get(
                'https://staff-api.yandex-team.ru/v3/groupmembership?_fields=person.login&person.official.is_dismissed=false&group.url=%s'
                % group,
                headers=get_autorization(),
            )
            if r.status_code != 200 and r.status_code < 500:
                print('users for group failed to be got: %s (%s)' % (group, r.status_code), file=sys.stderr)
                return
            assert r.status_code == 200, '(%s) %s' % (r.text, r.status_code)

            j = json.loads(r.text)
            return [a['person']['login'] for a in j['result']]

        res = try_do(foo)
        if res is None:
            return

        print('group -> users: %s -> %s' % (group, res), file=sys.stderr)

        res = {'owners': res}
        STAFF_INFO[group] = res
        return res

    def get_network_info(self, net):
        global NETWORK_INFO_BY_SUB
        global IS_NETWORK_INFO_BY_SUB_INITED

        def foo():
            r = requests.get('https://racktables.yandex.net/export/nets-by-project.php')
            assert r.status_code == 200, '(%s) %s' % (r.text, r.status_code)

            res = {}
            values = r.text.split('\n')
            for line in values:
                v = line.split('\t')
                if len(v) == 1:
                    continue
                sub, macro, vlan, desc, users, dc = v
                res[sub] = {'owners': [u.strip('%') for u in users.split(',')]}
            return res

        if not IS_NETWORK_INFO_BY_SUB_INITED:
            NETWORK_INFO_BY_SUB.update(try_do(foo))
            IS_NETWORK_INFO_BY_SUB_INITED = True

        if net in NETWORK_INFO_BY_SUB:
            res = NETWORK_INFO_BY_SUB[net]
            print('network -> users: %s -> %s' % (net, res), file=sys.stderr)
            return res

    def _as_string(self, k, meta):
        macros, grants = meta
        print('===getting "%s" with meta %s' % (k, macros), file=sys.stderr)

        res = {
            'name': k,
            'c_admins': [],
            'abc_admins': [],
            'abc_managers': [],
            'abc_owners': [],
            'fw_owners': [],
            'staff_owners': [],
            'host_owners': [],
            'net_owners': [],
        }

        for macro, type in macros:
            if 'C' == type:
                info = self.get_conductor_info(macro)
                if info is None:
                    print('no info for macro %s (%s)' % (macro, k), file=sys.stderr)
                    continue

                if 'abc' in info:
                    r = get_abc_info(info['abc'])
                    if r is None:
                        res['error'] = 'macro has bad abc: %s' % info
                        print('macro has bad abc: %s (%s): %s' % (macro, k, info), file=sys.stderr)
                        continue
                    res['abc_admins'] = res['abc_admins'] + r['abc_admins']
                    res['abc_managers'] = res['abc_managers'] + r['abc_managers']
                else:
                    res['c_admins'] = res['c_admins'] + info['c_admins']
            elif 'F' == type:
                info = self.get_firewall_info(macro)
                if info is not None:
                    res['fw_owners'] = res['fw_owners'] + [
                        a for a in info['fw_owners'] if not a.startswith('yandex_') and not a.startswith('svc_')
                    ]
                    for gr in [a for a in info['fw_owners'] if a.startswith('yandex_')]:
                        r = self.get_staff_group_members(gr)
                        if r is None:
                            print('group is empty: %s: %s' % (macro, gr), file=sys.stderr)
                        else:
                            res['staff_owners'] = res['staff_owners'] + r['owners']

                    for gr in [a for a in info['fw_owners'] if a.startswith('svc_')]:
                        r = self.get_staff_group_members(gr)
                        if r is None:
                            print('group is empty: %s: %s' % (macro, gr), file=sys.stderr)
                        else:
                            res['abc_owners'] = res['abc_owners'] + r['owners']
                    continue
                print('no info for macro %s (%s)' % (macro, k), file=sys.stderr)

                hosts = self.get_firewall_hosts(macro)
                if hosts is None:
                    print('no hosts in macro %s (%s)' % (macro, k), file=sys.stderr)
                    continue

                for host in hosts:
                    adm = self.get_host_info(host)
                    if adm is not None:
                        res['host_owners'] = res['host_owners'] + adm['host_owners']
                        continue
                    else:
                        print('no adm for host %s (%s)' % (host, k), file=sys.stderr)

                    info = self.get_network_info(macro)
                    if info is not None:
                        res['net_owners'] = res['net_owners'] + info['owners']
                        continue
                    else:
                        print('no info for macro %s (%s) - Network' % (macro, k), file=sys.stderr)
            elif 'I' == type:
                if macro == '::1' or macro == '127.0.0.1':
                    res['host_owners'] = res['host_owners'] + ['dyor', 'abbat', 'avmm']
                    continue

                h = ip_to_hostname(macro)
                if h is None:
                    print('bad ip %s (%s)' % (macro, k), file=sys.stderr)
                    continue

                adm = self.get_host_info(h)
                if adm is None:
                    print('no info for macro %s (%s)' % (macro, k), file=sys.stderr)
                    continue

                res['host_owners'] = res['host_owners'] + adm['host_owners']
            elif 'H' == type:
                adm = self.get_host_info(macro)
                if adm is None:
                    print('no info for macro %s (%s)' % (macro, k), file=sys.stderr)
                    continue

                res['host_owners'] = res['host_owners'] + adm['host_owners']
            elif 'N' == type:
                info = self.get_network_info(macro)
                if info is None:
                    print('no info for macro %s (%s) - Network' % (macro, k), file=sys.stderr)
                    continue

                res['net_owners'] = res['net_owners'] + info['owners']
            else:
                raise Exception('unknown type %s for %s' % (type, k))

        return res


class TvmStats(BaseStats):
    def build_text(self, r, is_used, meta):
        tvm_id, grants = meta

        source = "Доступ был выдан для TVM-приложения %s (##%s##)\n" % (tvm_id, r['tvm_name'])
        gr_text = (
            "<{Доступ был выдан с правами (грантами) на:\n%%(json)"
            + json.dumps(grants, indent=4, sort_keys=True)
            + "%%}>\n"
        )

        return text_header(r['name'], is_used) + source + gr_text + text_footer()

    def add(self, consumer, grants, tvm_id):
        if consumer not in self._consumers:
            self._consumers[consumer] = (tvm_id, grants)

    def get_client_info(self, tvm_id):
        def foo():
            r = requests.get('https://tvm.yandex-team.ru/client/%d/info?with_deleted=true' % tvm_id)
            if r.status_code != 200 and r.status_code < 500:
                print('tvm_id %d was not found' % tvm_id, file=sys.stderr)
                return
            assert r.status_code == 200, '(%s) %s' % (r.text, r.status_code)

            j = json.loads(r.text)

            deleted = None
            if 'deletion_time' in j:
                deleted = j['deletion_time']

            return (j['name'], j['abc_service_id'], j['creator_uid'], deleted)

        return try_do(foo)

    def _as_string(self, k, meta):
        tvm_id, grants = meta
        print('===getting "%s" with tvm_id %s' % (k, tvm_id), file=sys.stderr)

        res = {'name': k, 'tvm_id': tvm_id}
        info = self.get_client_info(tvm_id)
        assert info is not None, 'tvm_id=%s was not found' % tvm_id

        name, abc, author, deleted = info
        res['tvm_name'] = name
        assert abc is not None, 'tvm_id has not abc_id: %s' % tvm_id

        if deleted is not None:
            res['warning'] = 'was deleted at %s' % datetime.datetime.utcfromtimestamp(deleted).strftime(
                '%Y-%m-%dT%H:%M:%S'
            )

        r = get_abc_info(abc)
        res['abc_admins'] = r['abc_admins']
        res['abc_managers'] = r['abc_managers']

        print('abc amdins: %s -> %s' % (abc, r['abc_admins']), file=sys.stderr)
        print('abc managers: %s -> %s' % (abc, r['abc_managers']), file=sys.stderr)

        return res


ip_grants = IpStats()
tvm_grants = TvmStats()


def from_grants(f):
    with open(f, 'r') as f_:
        root = json.load(f_)

    for k, v in root.items():
        grants = v["grants"]
        if 'admreg' in grants:
            del grants['admreg']

        if len(grants) == 0:
            continue

        is_tvm = "client_id" in v["client"]

        if is_tvm:
            if k in SKIP_TVM_CONSUMERS:
                continue
            tvm_grants.add(k, grants, v["client"]["client_id"])
        else:
            if k in SKIP_IP_CONSUMERS:
                continue
            ip_grants.add(k, grants)


def parse_file(f):
    for line in open(f, 'r'):
        try:
            values = line[:-1].split(' ')
            is_tvm = values[1] != "-"
            c = values[2]
            if is_tvm:
                tvm_grants.found(c)
            else:
                ip_grants.found(c)
        except Exception as e:
            print("=failed to parse line: %s. %s" % (e, line))
            raise


if __name__ == '__main__':
    from_grants('./tmp/consumer_grants.production.json')

    # tmp
    ip_grants.tsv_to_macro('./PASSPADMIN-1217.txt')

    outdir = sys.argv[1]
    for f in ['./passport.txt']:
        parse_file(f)

    print(ip_grants.as_string_used(), file=open(outdir + '/out.ip.used.json', 'w'))
    print(ip_grants.as_string_unused(), file=open(outdir + '/out.ip.unused.json', 'w'))
    print(tvm_grants.as_string_used(), file=open(outdir + '/out.tvm.used.json', 'w'))
    print(tvm_grants.as_string_unused(), file=open(outdir + '/out.tvm.unused.json', 'w'))
