#!/usr/bin/env python3
import json
import sys

SEZAM_ENV = {
    'dev': ('dev', 'sas'),
    'testing': ('testing', 'sas, man'),
    'rc': ('rc', 'sas, man, myt'),
    'mimino': ('production.other', 'sas, man, myt'),
    'prestable': ('production', 'sas'),
    'stable': ('production', 'sas, man, vla, myt, iva'),
    'yateam_testing': ('testing', 'sas, man'),
    'yateam_prestable': ('production', 'sas'),
    'yateam_stable': ('production', 'sas, man, vla, myt, iva'),
}

NONE_SUBSET = ('none', '', '')

BLACKBOX_SUBSET = {
    'dev': NONE_SUBSET,
    'testing': NONE_SUBSET,
    'rc': NONE_SUBSET,
    'mimino': ('mimino', '', ''),
    'all': ('common, mail', '(common + mail)', ''),
    'common': ('common', '(common)', '_common'),
    'mail': ('mail', '(mail)', '_mail'),
    'yateam': NONE_SUBSET,
}

DB_SET = {
    'yateam': [
        'passportdbcentral',
        'passportdbcentral_domains',
        'passportdbshard1',
        'oauthdbcentral',
        'oauthdbshard1',
        'badauth',
        'perimeter',
    ]
}

DB_SET_DEFAULT = [
    'passportdbcentral',
    'passportdbcentral_slow',
    'passportdbshard1',
    'passportdbshard2',
    'passportdbshard3',
    'passportdbshard4',
    'oauthdbcentral',
    'oauthdbshard1',
    'oauthdbshard2',
    'badauth',
    'sess_kill',
    'passport_internal',
]


def add_default_panels(env, subset, project, subset_params):
    ctype, geo = SEZAM_ENV[env]
    tier, description, path_suffix = subset_params

    if subset == 'yateam':
        ctype += '.intranet'

    return [
        {
            'name': 'default' + path_suffix,
            'title': 'default %s' % description,
            'type': 'template_panel',
            'item_id': 'passport_common_daemon_default',
            'substitutions': {
                '__itype': 'passport' + project,
                '__prefix': project,
                '__geo': geo,
                '__ctype': ctype,
                '__tier': tier,
                '__prj': 'passport.' + project,
            },
        },
        {
            'name': 'health' + path_suffix,
            'title': 'health %s' % description,
            'type': 'template_panel',
            'item_id': 'passport_common_daemon_health',
            'substitutions': {
                '__itype': 'passport' + project,
                '__prefix': project,
                '__geo': geo,
                '__ctype': ctype,
                '__tier': tier,
                '__prj': 'passport.' + project,
            },
        },
        {
            'name': 'dbpools' + path_suffix,
            'title': 'dbpools %s' % description,
            'type': 'template_panel',
            'item_id': 'passport_common_daemon_dbpool',
            'substitutions': {
                '__itype': 'passport' + project,
                '__prefix': project,
                '__geo': geo,
                '__ctype': ctype,
                '__tier': tier,
                '__prj': 'passport.' + project,
            },
        },
        {
            'name': 'workpool' + path_suffix,
            'title': 'workpool %s' % description,
            'type': 'template_panel',
            'item_id': 'passport_common_daemon_workpool',
            'substitutions': {
                '__itype': 'passport' + project,
                '__prefix': project,
                '__geo': geo,
                '__ctype': ctype,
                '__tier': tier,
                '__prj': 'passport.' + project,
            },
        },
    ]


def add_dbpool_ext(env, subset, project, subset_params, db, geo, daemon=''):
    ctype, _geo = SEZAM_ENV[env]
    tier, description, path_suffix = subset_params
    if len(daemon) == 0:
        daemon = project

    if subset == 'yateam':
        ctype += '.intranet'

    return {
        'name': db + path_suffix,
        'title': db,
        'type': 'template_panel',
        'item_id': 'passport_common_daemon_dbpool_ext',
        'substitutions': {
            '__itype': 'passport' + project,
            '__prefix': project,
            '__geo': geo,
            '__ctype': ctype,
            '__tier': tier,
            '__prj': 'passport.' + project,
            '__db': db,
        },
    }


def add_blackbox_panels(env, subset):
    ctype, geo = SEZAM_ENV[env]
    tier, description, path_suffix = BLACKBOX_SUBSET[subset]

    db_set = DB_SET[subset] if subset in DB_SET else DB_SET_DEFAULT

    if subset == 'yateam':
        ctype += '.intranet'

    res = (
        add_default_panels(env, subset, 'blackbox', BLACKBOX_SUBSET[subset])
        + [
            {
                'name': 'specific' + path_suffix,
                'title': 'specific %s' % description,
                'type': 'template_panel',
                'item_id': 'passport_sezam_blackbox_specific',
                'substitutions': {
                    '__geo': geo,
                    '__ctype': ctype,
                    '__tier': tier,
                },
            },
        ]
        + [
            {
                'name': 'ops' + path_suffix,
                'title': 'ops %s' % description,
                'type': 'template_panel',
                'item_id': 'passport_sezam_blackbox_ops',
                'substitutions': {
                    '__geo': geo,
                    '__ctype': ctype,
                    '__tier': tier,
                },
            },
        ]
        + [
            {
                'name': 'consumers' + path_suffix,
                'title': 'consumers %s' % description,
                'type': 'template_panel',
                'item_id': 'passport_sezam_blackbox_consumers',
                'substitutions': {
                    '__geo': geo,
                    '__ctype': ctype,
                    '__tier': tier,
                },
            },
        ]
        + [
            {
                'name': 'times' + path_suffix,
                'title': 'times %s' % description,
                'type': 'template_panel',
                'item_id': 'passport_sezam_blackbox_times',
                'substitutions': {
                    '__geo': geo,
                    '__ctype': ctype,
                    '__tier': tier,
                },
            },
        ]
    )

    if subset in ['all', 'yateam', 'testing']:
        res += [
            {
                'name': 'dbpool_ext_' + g,
                'title': 'dbpool_ext (%s)' % g,
                'type': 'dummy',
                'children': [add_dbpool_ext(env, subset, 'blackbox', BLACKBOX_SUBSET[subset], db, g) for db in db_set],
            }
            for g in geo.split(', ')
        ]

    return res


def add_host_panels(env, subset):
    ctype, geo = SEZAM_ENV[env]
    tier, description, path_suffix = BLACKBOX_SUBSET[subset]

    if subset == 'yateam':
        ctype += '.intranet'

    return [
        {
            'name': 'default' + path_suffix,
            'title': 'metrics %s' % description,
            'type': 'template_panel',
            'item_id': 'passport_host_metrics',
            'substitutions': {
                '__itype': 'common',
                '__geo': geo,
                '__ctype': ctype,
                '__tier': tier,
                '__prj': 'passport.blackbox',
            },
        },
    ]


def build_host_metrics(env):
    result = {
        'name': 'host_metrics',
        'title': 'host_metrics',
        'type': 'dummy',
        'children': [],
    }

    if env == 'dev':
        result['children'] += add_host_panels(env, 'dev')
    elif env == 'testing':
        result['children'] += add_host_panels(env, 'testing')
    elif env == 'rc':
        result['children'] += add_host_panels(env, 'rc')
    elif env == 'mimino':
        result['children'] += add_host_panels(env, 'mimino')
    elif env == 'yateam_testing':
        result['children'] += add_host_panels(env, 'yateam')
    elif env == 'yateam_prestable':
        result['children'] += add_host_panels(env, 'yateam')
    elif env == 'yateam_stable':
        result['children'] += add_host_panels(env, 'yateam')
    else:
        result['children'] += add_host_panels(env, 'all')
        result['children'] += add_host_panels(env, 'mail')
        result['children'] += add_host_panels(env, 'common')

    return result


def build_env(env):
    result = {
        'name': env,
        'title': env if env != 'stable' else 'stable+prestable',
        'type': 'dummy',
        'children': [],
    }

    if env == 'dev':
        result['children'] += add_blackbox_panels(env, 'dev')
    elif env == 'testing':
        result['children'] += add_blackbox_panels(env, 'testing')
    elif env == 'rc':
        result['children'] += add_blackbox_panels(env, 'rc')
    elif env == 'mimino':
        result['children'] += add_blackbox_panels(env, 'mimino')
    elif env == 'yateam_testing':
        result['children'] += add_blackbox_panels(env, 'yateam')
    elif env == 'yateam_prestable':
        result['children'] += add_blackbox_panels(env, 'yateam')
    elif env == 'yateam_stable':
        result['children'] += add_blackbox_panels(env, 'yateam')
    else:
        result['children'] += add_blackbox_panels(env, 'all')
        result['children'] += add_blackbox_panels(env, 'mail')
        result['children'] += add_blackbox_panels(env, 'common')

    return result


def build_simple_alerts():
    return {
        'name': 'simple_alerts',
        'title': 'simple_alerts',
        'type': 'template_panel',
        'item_id': 'passport_simple_alerts',
    }


def main():
    result = {}
    result['path'] = '/passport/blackbox'
    result['tree'] = {
        'owners': ['cerevra'],
        'name': 'blackbox',
        'title': 'blackbox',
        'type': 'dummy',
        'children': [],
    }
    result['tree']['children'].append(build_env('stable'))
    result['tree']['children'].append(build_env('prestable'))
    result['tree']['children'].append(build_env('testing'))
    result['tree']['children'].append(build_env('dev'))
    result['tree']['children'].append(build_env('rc'))
    result['tree']['children'].append(build_env('mimino'))
    result['tree']['children'].append(build_env('yateam_testing'))
    result['tree']['children'].append(build_env('yateam_prestable'))
    result['tree']['children'].append(build_env('yateam_stable'))
    result['tree']['children'].append(build_simple_alerts())

    with open('passport_blackbox.json', 'wt') as f:
        f.write(json.dumps(result, indent=4))
        f.write('\n')


if __name__ == '__main__':
    main()
